/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.blocks.security;

import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.security.model.BasicUser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;

/**
 * The Class SecurityContextUtil.
 */
public class SecurityContextUtil {
	public static final Logger LOG = LoggerFactory.getLogger(SecurityContextUtil.class);

	/**
	 * Gets the username.
	 *
	 * @return the username
	 */
	public static String getUsername() {
		final Authentication auth = SecurityContextHolder.getContext().getAuthentication();

		if (auth.getPrincipal() instanceof UserDetails) {
			return ((UserDetails) auth.getPrincipal()).getUsername();
		} else {
			return null;
		}
	}

	/**
	 * Gets the me.
	 *
	 * @param <T> the generic type
	 * @return current user
	 */
	@SuppressWarnings("unchecked")
	public static <T extends BasicUser<?>> T getMe() {
		final Authentication auth = SecurityContextHolder.getContext().getAuthentication();

		if (auth != null) {
			Object principal = auth.getPrincipal();
			if (principal instanceof BasicUser) {
				return (T) principal;
			} else {
				LOG.warn("Principal {} is not BasicUser, but type {}", principal, principal.getClass());
			}
		} 

		return null;
	}

	/**
	 * Gets the current principal (User or OAuthClient)
	 *
	 * @param <T> the generic type
	 * @return the current security principal
	 */
	@SuppressWarnings("unchecked")
	public static <T extends AclSid> T getCurrentUser() {
		final Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

		if (authentication != null) {
			Object principal = authentication.getPrincipal();
			if (principal instanceof AclSid) {
				return (T) principal;
			} else {
				LOG.warn("Principal {} is not AclSid, but type {}", principal, principal.getClass());
			}
		}

		return null;
	}
}
