/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.persistence;

import java.util.List;

import org.genesys.blocks.oauth.model.AccessToken;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.querydsl.QueryDslPredicateExecutor;
import org.springframework.stereotype.Repository;

/**
 * The Interface AccessTokenRepository.
 */
@Repository
public interface AccessTokenRepository extends JpaRepository<AccessToken, String>, QueryDslPredicateExecutor<AccessToken> {

	/**
	 * Find by authentication id.
	 *
	 * @param key the key
	 * @return the access token
	 */
	AccessToken findByAuthenticationId(String key);

	/**
	 * Delete by refresh token.
	 *
	 * @param refreshToken the refresh token
	 */
	@Modifying
	@Query("delete from AccessToken at where at.refreshToken = ?1")
	void deleteByRefreshToken(String refreshToken);

	/**
	 * Find by client id.
	 *
	 * @param clientId the client id
	 * @return the list
	 */
	List<AccessToken> findByClientId(String clientId);

	/**
	 * Find by client id and username.
	 *
	 * @param clientId the client id
	 * @param username the username
	 * @return the list
	 */
	List<AccessToken> findByClientIdAndUsername(String clientId, String username);

	/**
	 * Delete by authentication id.
	 *
	 * @param key the key
	 */
	@Modifying
	@Query("delete from AccessToken at where at.authenticationId = ?1")
	void deleteByAuthenticationId(String key);

	/**
	 * Find by username.
	 *
	 * @param uuid the uuid
	 * @return the list
	 */
	List<AccessToken> findByUsername(String uuid);

}
