package org.genesys.blocks.security.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.genesys.blocks.security.SecurityContextUtil;
import org.genesys.blocks.security.model.AclAwareModel;
import org.genesys.blocks.security.model.AclClass;
import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclObjectIdentity;
import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.security.persistence.AclClassPersistence;
import org.genesys.blocks.security.persistence.AclEntryPersistence;
import org.genesys.blocks.security.persistence.AclObjectIdentityPersistence;
import org.genesys.blocks.security.persistence.AclSidPersistence;
import org.genesys.blocks.security.service.CustomAclService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.acls.domain.BasePermission;
import org.springframework.security.acls.model.Permission;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

@Service
@Transactional
public class CustomAclServiceImpl implements CustomAclService {

	private static Permission[] basePermissions;

	@Autowired
	private AclObjectIdentityPersistence aclObjectIdentityPersistence;

	@Autowired
	private AclSidPersistence aclSidPersistence;

	@Autowired
	private AclClassPersistence aclClassPersistence;

	@Autowired
	private AclEntryPersistence aclEntryPersistence;

	@Autowired
	private CacheManager cacheManager;

	static {
		basePermissions = new Permission[] { BasePermission.CREATE, BasePermission.READ, BasePermission.WRITE, BasePermission.DELETE, BasePermission.ADMINISTRATION };
	}

	private static final Logger LOG = LoggerFactory.getLogger(CustomAclServiceImpl.class);

	@Override
	@Transactional(propagation = Propagation.REQUIRED)
	public void addCreatorPermissions(final AclAwareModel target) {
		if (target == null || target.getId() <= 0l) {
			LOG.warn("No target specified for ACL permissions, bailing out!");
			return;
		}

		final String uuid = SecurityContextUtil.getUsername();
		if (uuid == null) {
			LOG.warn("No user in security context, not doing ACL");
			return;
		}
		
		LOG.debug("Inserting owner ACL entries for owner={} class={} id={}", uuid, target.getClass().getName(), target.getId());

		// it can be pre-authorized Admin
		final AclSid aclSid = ensureSid(uuid, true);

		final AclClass aclClass = ensureAclClass(target.getClass().getName());

		// create object identity
		final AclObjectIdentity objectIdentity = new AclObjectIdentity();
		objectIdentity.setObjectIdIdentity((Long) target.getId());
		objectIdentity.setAclClass(aclClass);
		objectIdentity.setOwnerSid(aclSid);
		objectIdentity.setParentObject(null);
		objectIdentity.setEntriesInheriting(false);

		// save object identity
		AclObjectIdentity savedAclObjectIdentity = aclObjectIdentityPersistence.findByObjectIdIdentityAndClassName(objectIdentity.getObjectIdIdentity(), objectIdentity.getAclClass().getAclClass());
		if (savedAclObjectIdentity == null) {
			savedAclObjectIdentity = aclObjectIdentityPersistence.save(objectIdentity);
			final Map<Integer, Boolean> permissionsMap = new HashMap<>();
			for (final Permission permission : basePermissions) {
				permissionsMap.put(permission.getMask(), true);
			}

			addPermissions(aclSid, savedAclObjectIdentity, permissionsMap);
		}
	}

	@Override
	@Transactional(propagation = Propagation.REQUIRED)
	public void removePermissions(final AclAwareModel target) {
		final AclObjectIdentity savedAclObjectIdentity = aclObjectIdentityPersistence.findByObjectIdIdentityAndClassName((Long) target.getId(), target.getClass().getName());
		if (savedAclObjectIdentity != null) {
			final List<AclEntry> aclEntries = aclEntryPersistence.findByObjectIdentity(savedAclObjectIdentity);
			if (aclEntries != null) {
				aclEntryPersistence.delete(aclEntries);
				aclObjectIdentityPersistence.delete(savedAclObjectIdentity.getId());
			}
		}
	}

	private void addPermissions(final AclSid ownerSid, final AclObjectIdentity objectIdentity, final Map<Integer, Boolean> permissions) {
		// create Acl Entry
		for (final Permission permission : basePermissions) {
			final int mask = permission.getMask();
			final AclEntry aclEntry = new AclEntry();
			aclEntry.setAclObjectIdentity(objectIdentity);
			aclEntry.setAclSid(ownerSid);
			aclEntry.setAceOrder(getAceOrder(objectIdentity.getId()));
			aclEntry.setGranting(permissions.get(mask));
			aclEntry.setAuditSuccess(true);
			aclEntry.setAuditFailure(true);
			// set full access for own organization
			aclEntry.setMask(mask);

			// save ACL
			aclEntryPersistence.save(aclEntry);
		}
		
		Cache aclCache = cacheManager.getCache("aclCache");
		if (aclCache != null)
			aclCache.clear();
	}

	/**
	 * Generates next ace_order value (to avoid DuplicateIndex exception : acl_object_identity + ace_order is unique
	 * index)
	 *
	 * @param aclObjectEntityId - id of acl_object_identity table
	 * @return - ace_order value
	 */
	private Long getAceOrder(final long aclObjectEntityId) {
		final Long maxAceOrder = aclEntryPersistence.getMaxAceOrderForObjectEntity(aclObjectEntityId);
		return maxAceOrder != null ? maxAceOrder + 1 : 1;
	}

	private AclClass ensureAclClass(final String className) {
		AclClass aclClass = aclClassPersistence.findByAclClass(className);

		if (aclClass == null) {
			LOG.warn("Missing AclClass '{}'", className);
			aclClass = new AclClass();
			aclClass.setAclClass(className);
			return aclClassPersistence.save(aclClass);
		}

		return aclClass;
	}

	private AclSid ensureSid(final String uuid, final boolean principal) {
		AclSid aclSid = aclSidPersistence.findBySidAndPrincipal(uuid, principal);

		if (aclSid == null) {
			// create Acl Sid
			aclSid = new AclSid();
			aclSid.setPrincipal(principal);
			aclSid.setSid(uuid);

			// save it into db
			LOG.warn("New SID sid={} principal={}", aclSid.getSid(), aclSid.isPrincipal());
			return aclSidPersistence.save(aclSid);
		}

		return aclSid;
	}

	@Override
	@Transactional(readOnly = true)
	public AclObjectIdentity getObjectIdentity(final String className, final long id) {
		return aclObjectIdentityPersistence.findByObjectIdIdentityAndClassName(id, className);
	}

	@Override
	@Transactional(readOnly = true)
	public AclObjectIdentity getObjectIdentity(final long id) {
		return aclObjectIdentityPersistence.findOne(id);
	}

	@Override
	@Transactional(readOnly = true)
	public AclObjectIdentity getObjectIdentity(final AclAwareModel entity) {
		if (entity == null) {
			LOG.error("getObjectIdentity: Entity is null");
		}
		AclObjectIdentity oid = aclObjectIdentityPersistence.findByObjectIdIdentityAndClassName((long) entity.getId(), entity.getClass().getName());
		if (oid == null) {
			LOG.warn("ACL object identity not found for class={} id={}", entity.getClass().getName(), entity.getId());
		}
		return oid;
	}

	@Override
	@Transactional(readOnly = true)
	public Permission[] getAvailablePermissions(final String className) {
		// Do not remove parameter. We may change available permissions based on
		// parameter type!
		return basePermissions;
	}

	@Transactional(readOnly = true)
	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#id, #className, 'ADMINISTRATION')")
	public Map<String, Map<Integer, Boolean>> getPermissions(final long id, final String className) {
		final Map<String, Map<Integer, Boolean>> perm = new HashMap<>();

		final List<AclEntry> aclEntries = getAclEntries(getObjectIdentity(className, id));
		for (final AclEntry aclEntry : aclEntries) {
			Map<Integer, Boolean> granted = perm.get(aclEntry.getAclSid().getSid());
			if (granted == null) {
				perm.put(aclEntry.getAclSid().getSid(), granted = new HashMap<>());
			}
			granted.put((int) aclEntry.getMask(), aclEntry.isGranting());
		}

		return perm;
	}

	@Transactional(readOnly = true)
	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity, 'ADMINISTRATION')")
	public Map<String, Map<Integer, Boolean>> getPermissions(final AclAwareModel entity) {
		return getPermissions((long) entity.getId(), entity.getClass().getName());
	}

	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity.aclClass.id, #entity.aclClass.aclClass, 'ADMINISTRATION')")
	public void updatePermission(final AclObjectIdentity entity, final String sid, final Map<Integer, Boolean> permissionMap) {
		boolean oneGranting = false;
		final List<AclEntry> aclEntries = aclEntryPersistence.findBySidAndAclClass(sid, entity.getAclClass().getAclClass());
		for (final AclEntry aclEntry : aclEntries) {
			aclEntry.setGranting(permissionMap.get((int) aclEntry.getMask()));
			oneGranting |= aclEntry.isGranting();
		}
		if (oneGranting) {
			LOG.info("Saving " + aclEntries);
			aclEntryPersistence.save(aclEntries);
		} else {
			LOG.info("Deleting " + aclEntries);
			aclEntryPersistence.delete(aclEntries);
		}
		cacheManager.getCache("aclCache").clear();
	}

	@Override
	@Transactional(readOnly = true)
	public List<AclEntry> getAclEntries(final AclObjectIdentity objectIdentity) {
		return aclEntryPersistence.findByObjectIdentity(objectIdentity);
	}

	@Override
	@Transactional(readOnly = true)
	public List<AclEntry> getAclEntries(final AclAwareModel entity) {
		return aclEntryPersistence.findByObjectIdentity(getObjectIdentity(entity));
	}

	@Override
	@Transactional(readOnly = true)
	public List<AclSid> getSids(final long id, final String className) {
		return aclEntryPersistence.getSids(id, className);
	}

	@Override
	@Transactional(readOnly = true)
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity, 'ADMINISTRATION')")
	public List<AclSid> getSids(final AclAwareModel entity) {
		return aclEntryPersistence.getSids((long) entity.getId(), entity.getClass().getName());
	}

	@Override
	@Transactional(readOnly = true)
	public List<AclSid> getAllSids() {
		return aclSidPersistence.findAll();
	}

	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#objectIdIdentity, #className, 'ADMINISTRATION')")
	public boolean addPermissions(final long objectIdIdentity, final String className, final String uuid, final boolean principal, final Map<Integer, Boolean> permissions) {
		final AclSid sid = ensureSid(uuid, principal);
		final AclObjectIdentity oid = ensureObjectIdentity(className, objectIdIdentity);

		addPermissions(sid, oid, permissions);
		return true;
	}

	@Override
	@Transactional
	public AclObjectIdentity ensureObjectIdentity(final String className, final long objectIdIdentity) {
		AclObjectIdentity aoi = aclObjectIdentityPersistence.findByObjectIdIdentityAndClassName(objectIdIdentity, className);
		if (aoi == null) {
			aoi = new AclObjectIdentity();
			aoi.setObjectIdIdentity(objectIdIdentity);
			aoi.setAclClass(ensureAclClass(className));
			// System user UUID
			final String uuid = SecurityContextUtil.getMe().getUuid();
			final AclSid ownerSid = ensureSid(uuid, true);
			aoi.setOwnerSid(ownerSid);
			aoi = aclObjectIdentityPersistence.save(aoi);
		}
		return aoi;
	}

	@Override
	public List<Integer> permissionsBySid(final String className, final Long id, final String sid) {
		Map<String, Map<Integer, Boolean>> map = getPermissions(id, className);
		Map<Integer, Boolean> permissionMap = map.get(sid);

		List<Integer> list = new ArrayList<>();

		for (Map.Entry<Integer, Boolean> e : permissionMap.entrySet()) {
			Integer key = e.getKey();
			Boolean value = e.getValue();
			if (value) {
				list.add(key);
			}
		}
		return list;
	}
	
	

	@Override
	@Transactional(readOnly = true)
	public List<Long> listIdentitiesForSid(Class<? extends AclAwareModel> clazz, UserDetails authUser, Permission permission) {
		return aclEntryPersistence.findObjectIdentitiesBySidAndAclClassAndMask(authUser.getUsername(), clazz.getName(), permission.getMask());
	}

}
