/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service.impl;

import java.util.Date;
import java.util.Set;

import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.model.BasicUser.AccountType;
import org.genesys.blocks.security.service.BasicUserService;
import org.genesys.blocks.security.service.PasswordPolicy;
import org.genesys.blocks.security.service.PasswordPolicy.PasswordPolicyException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.NoOpPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.transaction.annotation.Transactional;

@Transactional(readOnly = true)
public abstract class BasicUserServiceImpl<R extends GrantedAuthority, T extends BasicUser<R>> implements BasicUserService<R, T> {

	public static final Logger LOG = LoggerFactory.getLogger(BasicUserServiceImpl.class);

	/// A non-password used for system and Google accounts
	private static final String THIS_IS_NOT_A_PASSWORD = "THIS-IS-NOT-A-PASSWORD";

	private long accountLockoutTime = 5 * 60 * 1000;

	@Autowired
	private JpaRepository<T, Long> userRepository;

	@Autowired(required = false)
	protected final PasswordEncoder passwordEncoder = NoOpPasswordEncoder.getInstance();

	@Autowired(required = false)
	private PasswordPolicy passwordPolicy;

	public void setAccountLockoutTime(long accountLockoutTime) {
		this.accountLockoutTime = accountLockoutTime;
	}

	@Override
	public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
		return getUserByEmail(username);
	}

	@Override
	public T getUser(long id) throws NoUserFoundException {
		T user = userRepository.findOne(id);
		if (user == null) {
			throw new NoUserFoundException(id);
		}
		return user;
	}
	
	@Override
	@Transactional
	public T updateUser(T user, String email, String fullName) {
		user.setEmail(email);
		user.setFullName(fullName);
		return userRepository.save(user);
	}

	@Override
	@Transactional
	public void deleteUser(T user) {
		userRepository.delete(user);
	}

	@Override
	@Transactional
	// FIXME Needs permission check
	public T setRoles(T user, Set<R> newRoles) {
		user = userRepository.findOne(user.getId());

		// If roles match, do nothing
		if (newRoles.containsAll(user.getRoles()) && user.getRoles().containsAll(newRoles)) {
			LOG.debug("Roles {} match {}. No change.", newRoles, user.getRoles());
			return user;
		}

		user.getRoles().clear();
		user.getRoles().addAll(newRoles);
		LOG.info("Setting roles for user {} to {}", user.getEmail(), user.getRoles());
		return userRepository.save(user);
	}

	@Override
	@Transactional
	public T changePassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			setPassword(user, password);
			return userRepository.save(user);
		} else {
			throw new PasswordPolicyException("Password can be set only for LOCAL account types");
		}
	}

	protected void setPassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			assureGoodPassword(password);
			user.setPassword(password == null ? null : passwordEncoder.encode(password));
		} else {
			user.setPassword(THIS_IS_NOT_A_PASSWORD);
		}
	}

	/**
	 * Test if password passes the password policy (if set)
	 * 
	 * @param password candidate password
	 * @throws PasswordPolicyException if password does not match policy
	 */
	public void assureGoodPassword(final String password) throws PasswordPolicyException {
		if (passwordPolicy != null) {
			passwordPolicy.assureGoodPassword(password);
		}
	}

	/**
	 * For internal use only.
	 * 
	 * @throws NoUserFoundException
	 */
	@Override
	@Transactional
	public void setAccountLockLocal(long userId, boolean locked) throws NoUserFoundException {
		final T user = getUser(userId);
		if (locked) {
			// Lock for account until some time
			user.setLockedUntil(new Date(System.currentTimeMillis() + accountLockoutTime));
			LOG.warn("Locking user account for user=" + user.getEmail() + "  until=" + user.getLockedUntil());
		} else {
			LOG.warn("Unlocking user account for user=" + user.getEmail());
			user.setLockedUntil(null);
		}
		userRepository.save(user);
	}

	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public void setAccountLock(long userId, boolean locked) throws NoUserFoundException {
		setAccountLockLocal(userId, locked);
	}

}
