/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service;

import java.util.Set;

import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.NotUniqueUserException;
import org.genesys.blocks.security.UserException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.service.PasswordPolicy.PasswordPolicyException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

public interface BasicUserService<R extends GrantedAuthority, T extends BasicUser<R>> extends UserDetailsService {

    /**
     * Get User by id
     *
     * @param id
     * @return the user or <code>null</code>
     */
    T getUser(long id) throws NoUserFoundException;

    /**
     * Get User by email
     *
     * @param email
     * @return the user
     */
    T getUserByEmail(String email) throws UsernameNotFoundException;

    /**
	 * Create a new user account
	 *
	 * @param email unique email address
	 * @param fullName Full name
	 * @param password initial account password
	 * @param accountType TODO
	 * @throws PasswordPolicyException password violates current password policy
	 * @throws NotUniqueUserException user is not unique in the system
	 * @throws UserException any other exception
	 * @return the new user
	 */
    T createUser(String email, String fullName, String password, BasicUser.AccountType accountType) throws NotUniqueUserException, PasswordPolicyException, UserException;

    /**
     * Grant specified roles to user. The {@link UserRole#USER} will be added if missing.
     *
     * @param user
     * @param roles
     * @return the updated user
     */
    T setRoles(T user, Set<R> roles);

    /**
     * Update user information
     *
     * @param user the user
     * @param email new email address
     * @param fullName new fullName
     * @return
     */
    T updateUser(T user, String email, String fullName) throws NotUniqueUserException;

    /**
     * Change password
     *
     * @param user the user
     * @param password new password
     * @throws PasswordPolicyException 
     * @return
     */
    T changePassword(T user, String password) throws PasswordPolicyException;

    /**
     * Try to delete user.
     *
     * @param user user to delete
     */
    void deleteUser(T user);

    /**
     * Lock user account
     * 
     * @param userID User ID
     * @param locked Is account locked
     * @throws NoUserFoundException 
     */
	void setAccountLockLocal(long userId, boolean locked) throws NoUserFoundException;

	void setAccountLock(long userId, boolean locked) throws NoUserFoundException;
}

