/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.model;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.MappedSuperclass;
import javax.persistence.PrePersist;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Transient;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonView;

import org.genesys.blocks.model.AuditedVersionedModel;
import org.genesys.blocks.model.JsonViews;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

@MappedSuperclass
public abstract class BasicUser<R extends GrantedAuthority> extends AuditedVersionedModel implements UserDetails {

	private static final long serialVersionUID = -5318892732608111516L;

	public static enum AccountType {
		LOCAL, LDAP, GOOGLE, SYSTEM
	}

	@Column(length = 36, unique = true)
	private String uuid;

	@JsonView(JsonViews.Public.class)
	@Column(nullable = false, unique = true, length = 60)
	private String email;

	@JsonView(JsonViews.Public.class)
	@Column(unique = true, length = 20)
	private String shortName;

	@JsonView(JsonViews.Public.class)
	@Column(length = 60)
	private String fullName;

	@JsonIgnore
	@Column(length = 60)
	private String password;

	/** Account control */
	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date accountExpires;

	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date lockedUntil;

	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date passwordExpires;

	@JsonView(JsonViews.Protected.class)
	@ElementCollection(fetch = FetchType.EAGER)
	@Enumerated(EnumType.STRING)
	@CollectionTable(name = "userrole", joinColumns = @JoinColumn(name = "userId"))
	@Column(name = "role")
	private Set<R> roles = new HashSet<>();

	@JsonView(JsonViews.Protected.class)
	@Enumerated(EnumType.STRING)
	@Column(length = 20)
	private AccountType accountType = AccountType.LOCAL;

	@PrePersist
	void ensureUUID() {
		if (this.uuid == null) {
			this.uuid = UUID.randomUUID().toString();
		}
	}

	public String getEmail() {
		return email;
	}

	public void setEmail(final String email) {
		this.email = email;
	}

	public String getShortName() {
		return shortName;
	}

	public void setShortName(final String shortName) {
		this.shortName = shortName;
	}

	public String getFullName() {
		return fullName;
	}

	public void setFullName(final String fullName) {
		this.fullName = fullName;
	}

	public Date getAccountExpires() {
		return accountExpires;
	}

	public void setAccountExpires(final Date accountExpires) {
		this.accountExpires = accountExpires;
	}

	public Date getLockedUntil() {
		return lockedUntil;
	}

	public void setLockedUntil(final Date lockedUntil) {
		this.lockedUntil = lockedUntil;
	}

	public Date getPasswordExpires() {
		return passwordExpires;
	}

	public void setPasswordExpires(final Date passwordExpires) {
		this.passwordExpires = passwordExpires;
	}

	public void setPassword(final String password) {
		this.password = password;
	}

	public Set<R> getRoles() {
		return roles;
	}

	public void setRoles(final Set<R> roles) {
		this.roles = roles;
	}

	public String getUuid() {
		return uuid;
	}

	public void setUuid(final String uuid) {
		this.uuid = uuid;
	}

	@Override
	public String toString() {
		return MessageFormat.format("id={0} email={1} fullName={2}", getId(), email, fullName);
	}

	@Transient
	@Override
	public Collection<? extends GrantedAuthority> getAuthorities() {
		return getRoles().stream().map(role -> new SimpleGrantedAuthority(role.getAuthority())).collect(Collectors.toSet());
	}

	@Override
	public String getPassword() {
		return password;
	}

	@Override
	public String getUsername() {
		return uuid;
	}

	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isAccountNonExpired() {
		return accountExpires == null || !accountExpires.before(new Date());
	}

	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isAccountNonLocked() {
		return lockedUntil == null || !lockedUntil.after(new Date());
	}

	@JsonView(JsonViews.Protected.class)
	public boolean isAccountLocked() {
		return !isAccountNonLocked();
	}

	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isCredentialsNonExpired() {
		return passwordExpires == null || !passwordExpires.before(new Date());
	}

	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isEnabled() {
		return super.isActive();
	}

	public AccountType getAccountType() {
		return accountType;
	}

	public void setAccountType(final AccountType accountType) {
		this.accountType = accountType;
	}

}
