/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.model;

import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;

import com.fasterxml.jackson.annotation.JsonIgnore;

import org.genesys.blocks.model.BasicModel;

/**
 * ACL Object Identity represents a specific ACL-aware entity (combination of {@link AclClass} and {#link objectIdIdentity}).
 * 
 * It records the owner of the entity and entity's parent object if any.
 */
@Entity
@Table(name = "acl_object_identity", uniqueConstraints = @UniqueConstraint(columnNames = { "object_id_class", "object_id_identity" }))
public class AclObjectIdentity extends BasicModel {

	private static final long serialVersionUID = -8987281958564243010L;

	@ManyToOne(fetch = FetchType.EAGER, cascade = CascadeType.PERSIST)
	@JoinColumn(name = "object_id_class", nullable = false)
	private AclClass aclClass;

	@ManyToOne(fetch = FetchType.EAGER, cascade = CascadeType.PERSIST)
	@JoinColumn(name = "parent_object", nullable = true)
	private AclObjectIdentity parentObject;

	@ManyToOne(fetch = FetchType.EAGER, cascade = CascadeType.PERSIST)
	@JoinColumn(name = "owner_sid", nullable = true)
	private AclSid ownerSid;

	@Column(name = "object_id_identity", nullable = false)
	private long objectIdIdentity;

	@Column(name = "entries_inheriting", nullable = false, length = 1)
	private boolean entriesInheriting;

	@JsonIgnore
	@OneToMany(mappedBy = "aclObjectIdentity", fetch = FetchType.LAZY, cascade = CascadeType.REMOVE, orphanRemoval = true)
	private List<AclEntry> aclEntries;

	public AclClass getAclClass() {
		return aclClass;
	}

	public void setAclClass(final AclClass aclClass) {
		this.aclClass = aclClass;
	}

	public AclObjectIdentity getParentObject() {
		return parentObject;
	}

	public void setParentObject(final AclObjectIdentity parentObject) {
		this.parentObject = parentObject;
	}

	public AclSid getOwnerSid() {
		return ownerSid;
	}

	public void setOwnerSid(final AclSid ownerSid) {
		this.ownerSid = ownerSid;
	}

	public long getObjectIdIdentity() {
		return objectIdIdentity;
	}

	public void setObjectIdIdentity(final long objectIdIdentity) {
		this.objectIdIdentity = objectIdIdentity;
	}

	public boolean isEntriesInheriting() {
		return entriesInheriting;
	}

	public void setEntriesInheriting(final boolean entriesInheriting) {
		this.entriesInheriting = entriesInheriting;
	}

	public List<AclEntry> getAclEntries() {
		return aclEntries;
	}

	public void setAclEntries(final List<AclEntry> aclEntries) {
		this.aclEntries = aclEntries;
	}
}
