/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.component;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.genesys.blocks.security.model.AclAwareModel;
import org.genesys.blocks.security.service.CustomAclService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

/**
 * Using aspects to generate default ACL permissions when entities are persisted and remove permissions on delete.
 * 
 * @author Maxym Borodenko
 * @author Matija Obreza
 */
@Aspect
@Component
public class AclAssignerAspect {

	private static final Logger LOG = LoggerFactory.getLogger(AclAssignerAspect.class);

	@Autowired
	private CustomAclService aclService;

	public AclAssignerAspect() {
		LOG.warn("Enabling {}", getClass().getName());
	}

	/**
	 * Create owner permissions on persist
	 * 
	 * @param result
	 * @return
	 */
	@AfterReturning(pointcut = "execution(* org.springframework.data.repository.*.save(..)) || execution(* org.springframework.data.jpa.repository.*.save(..)) || execution(* org.springframework.data.jpa.repository.*.saveAndFlush(..))", returning = "result")
	public Object afterSaveAclObject(Object result) {
		final Authentication auth = SecurityContextHolder.getContext().getAuthentication();

		if (auth != null) {
			if (result instanceof AclAwareModel) {
				maybeAddCreatorPermissions(result);
			} else if (result instanceof Iterable) {
				// Handle collections of AclAwareModel
				Iterable<?> i = (Iterable<?>) result;
				for (Object o : i) {
					maybeAddCreatorPermissions(o);
				}
			} else {
				LOG.trace("{} is not instance of AclAwareModel", result);
			}
		}

		return result;
	}

	private void maybeAddCreatorPermissions(Object obj) {
		if (obj instanceof AclAwareModel) {
			aclService.addCreatorPermissions((AclAwareModel) obj);
		} else {
			LOG.trace("{} is not instance of AclAwareModel", obj);
		}
	}

	/**
	 * Remove permissions on delete
	 * 
	 * @param joinPoint
	 */
	@After("execution(* org.springframework.data.repository.*.delete(..)) || execution(* org.springframework.data.jpa.repository.*.deleteInBatch(..))")
	public void afterDeleteAclObject(JoinPoint joinPoint) {
		final Object arg0 = joinPoint.getArgs()[0];

		try {
			if (arg0 instanceof Long) {
				// NOOP
			} else if (arg0 instanceof AclAwareModel) {
				final AclAwareModel aclModel = (AclAwareModel) arg0;
				maybeRemovePermissions(aclModel);
			} else if (arg0 instanceof Iterable) {
				// Handle collections of AclAwareModel
				Iterable<?> i = (Iterable<?>) arg0;
				for (Object o : i) {
					maybeRemovePermissions(o);
				}
			} else {
				LOG.trace("{} is not instance of AclAwareModel", arg0);
			}
		} finally {
			// Nothing to do here
		}
	}

	private void maybeRemovePermissions(final Object obj) {
		if (obj instanceof AclAwareModel) {
			LOG.debug("Removing ACL entries for model={}", obj);
			aclService.removePermissions((AclAwareModel) obj);
		} else {
			LOG.trace("{} is not instance of AclAwareModel", obj);
		}
	}
}
