/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.service;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.stream.Collectors;

import org.genesys.blocks.oauth.model.AccessToken;
import org.genesys.blocks.oauth.model.OAuthClient;
import org.genesys.blocks.oauth.model.RefreshToken;
import org.genesys.blocks.oauth.persistence.AccessTokenRepository;
import org.genesys.blocks.oauth.persistence.OAuthClientRepository;
import org.genesys.blocks.oauth.persistence.RefreshTokenRepository;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.common.OAuth2RefreshToken;
import org.springframework.security.oauth2.common.util.SerializationUtils;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.ClientRegistrationException;
import org.springframework.security.oauth2.provider.NoSuchClientException;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.token.AuthenticationKeyGenerator;
import org.springframework.security.oauth2.provider.token.DefaultAuthenticationKeyGenerator;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
@Transactional(readOnly = false)
public class OAuthServiceImpl implements OAuthClientDetailsService, OAuthTokenStoreService {
	private static final Logger LOG = LoggerFactory.getLogger(OAuthServiceImpl.class);

	@Autowired
	private OAuthClientRepository oauthClientRepository;

	@Autowired
	private RefreshTokenRepository refreshTokenRepository;

	@Autowired
	private AccessTokenRepository accessTokenRepository;

	private AuthenticationKeyGenerator authenticationKeyGenerator = new DefaultAuthenticationKeyGenerator();

	public void setAuthenticationKeyGenerator(final AuthenticationKeyGenerator authenticationKeyGenerator) {
		this.authenticationKeyGenerator = authenticationKeyGenerator;
	}

	@Override
	@Cacheable(cacheNames = { "oauthclient" }, key = "#clientId", unless = "#result == null")
	public ClientDetails loadClientByClientId(final String clientId) throws ClientRegistrationException {
		final OAuthClient client = oauthClientRepository.findByClientId(clientId);
		if (client == null) {
			throw new NoSuchClientException(clientId);
		}
		client.getRoles().size();
		return client;
	}

	@Override
	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void storeAccessToken(final OAuth2AccessToken token, final OAuth2Authentication authentication) {
		String refreshToken = null;
		if (token.getRefreshToken() != null) {
			refreshToken = token.getRefreshToken().getValue();
		}

		if (readAccessToken(token.getValue()) != null) {
			removeAccessToken(token.getValue());
		}

		accessTokenRepository.deleteByAuthenticationId(authenticationKeyGenerator.extractKey(authentication));

		// "insert into oauth_access_token (token_id, token, authentication_id, user_name, client_id, authentication,
		// refresh_token)
		// values (?, ?, ?, ?, ?, ?, ?)";
		final AccessToken storedToken = new AccessToken();
		storedToken.setTokenId(extractTokenKey(token.getValue()));
		storedToken.setToken(serializeAccessToken(token));
		storedToken.setAuthenticationId(authenticationKeyGenerator.extractKey(authentication));
		storedToken.setUsername(authentication.isClientOnly() ? null : authentication.getName());
		storedToken.setClientId(authentication.getOAuth2Request().getClientId());
		storedToken.setAuthentication(serializeAuthentication(authentication));
		storedToken.setRefreshToken(extractTokenKey(refreshToken));

		accessTokenRepository.save(storedToken);
	}

	@Override
	@Cacheable(cacheNames = { "oauthaccesstoken" }, key = "#tokenValue", unless = "#result == null")
	public OAuth2AccessToken readAccessToken(final String tokenValue) {
		OAuth2AccessToken accessToken = null;

		LOG.trace("Reading access token value={} key={}", tokenValue, extractTokenKey(tokenValue));

		final AccessToken storedToken = accessTokenRepository.findOne(extractTokenKey(tokenValue));
		if (storedToken != null) {
			accessToken = deserializeAccessToken(storedToken.getToken());
		} else {
			if (LOG.isInfoEnabled()) {
				LOG.info("Failed to find access token for token " + tokenValue);
			}
		}

		return accessToken;
	}

	@Override
	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeAccessToken(final OAuth2AccessToken token) {
		removeAccessToken(token.getValue());
	}

	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeAccessToken(final String tokenValue) {
		accessTokenRepository.delete(extractTokenKey(tokenValue));
	}

	@Override
	@Cacheable(cacheNames = { "oauthaccesstokenauth" }, key = "#token.value", unless = "#result == null")
	public OAuth2Authentication readAuthentication(final OAuth2AccessToken token) {
		return readAuthentication(token.getValue());
	}

	@Override
	@Cacheable(cacheNames = { "oauthaccesstokenauth" }, key = "#tokenValue", unless = "#result == null")
	public OAuth2Authentication readAuthentication(final String tokenValue) {
		OAuth2Authentication authentication = null;
		LOG.trace("TokenValue={} key={}", tokenValue, extractTokenKey(tokenValue));
		final AccessToken storedToken = accessTokenRepository.findOne(extractTokenKey(tokenValue));
		if (storedToken != null) {
			authentication = deserializeAuthentication(storedToken.getAuthentication());
		} else {
			if (LOG.isInfoEnabled()) {
				LOG.info("Failed to find access token for token " + tokenValue);
			}
		}

		return authentication;
	}

	@Override
	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void storeRefreshToken(final OAuth2RefreshToken refreshToken, final OAuth2Authentication authentication) {
		// insert into oauth_refresh_token (token_id, token, authentication) values (?, ?, ?)
		final RefreshToken storedToken = new RefreshToken();
		storedToken.setTokenId(extractTokenKey(refreshToken.getValue()));
		storedToken.setToken(serializeRefreshToken(refreshToken));
		storedToken.setAuthentication(serializeAuthentication(authentication));

		refreshTokenRepository.save(storedToken);
	}

	@Override
	public OAuth2RefreshToken readRefreshToken(final String token) {
		OAuth2RefreshToken refreshToken = null;

		final RefreshToken storedToken = refreshTokenRepository.findOne(extractTokenKey(token));
		if (storedToken != null) {
			refreshToken = deserializeRefreshToken(storedToken.getToken());
		} else {
			if (LOG.isInfoEnabled()) {
				LOG.info("Failed to find refresh token for token " + token);
			}
		}

		return refreshToken;
	}

	@Override
	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeRefreshToken(final OAuth2RefreshToken token) {
		removeRefreshToken(token.getValue());
	}

	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeRefreshToken(final String token) {
		refreshTokenRepository.delete(extractTokenKey(token));
	}

	@Override
	public OAuth2Authentication readAuthenticationForRefreshToken(final OAuth2RefreshToken token) {
		return readAuthenticationForRefreshToken(token.getValue());
	}

	public OAuth2Authentication readAuthenticationForRefreshToken(final String token) {
		OAuth2Authentication authentication = null;

		final RefreshToken storedToken = refreshTokenRepository.findOne(extractTokenKey(token));
		if (storedToken != null) {
			authentication = deserializeAuthentication(storedToken.getAuthentication());
		} else {
			if (LOG.isInfoEnabled()) {
				LOG.info("Failed to find access token for token " + token);
			}
		}

		return authentication;
	}

	@Override
	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeAccessTokenUsingRefreshToken(final OAuth2RefreshToken refreshToken) {
		removeAccessTokenUsingRefreshToken(refreshToken.getValue());
	}

	@CacheEvict(cacheNames = { "oauthaccesstoken", "oauthaccesstokenauth" }, allEntries = true)
	public void removeAccessTokenUsingRefreshToken(final String refreshToken) {
		accessTokenRepository.deleteByRefreshToken(extractTokenKey(refreshToken));
	}

	@Override
	public OAuth2AccessToken getAccessToken(final OAuth2Authentication authentication) {
		OAuth2AccessToken accessToken = null;
		final String key = authenticationKeyGenerator.extractKey(authentication);

		LOG.trace("auth={} key={}", authentication, key);
		final AccessToken storedToken = accessTokenRepository.findByAuthenticationId(key);
		if (storedToken != null) {
			accessToken = deserializeAccessToken(storedToken.getToken());
			final OAuth2Authentication auth = readAuthentication(accessToken.getValue());

			if (accessToken != null && auth != null && !key.equals(authenticationKeyGenerator.extractKey(auth))) {
				removeAccessToken(accessToken.getValue());
				// Keep the store consistent (maybe the same user is represented by this
				// authentication but the details have
				// changed)
				storeAccessToken(accessToken, authentication);
			}
		} else {
			if (LOG.isDebugEnabled()) {
				LOG.debug("Failed to find access token for authentication={}", authentication);
			}
		}

		return accessToken;
	}

	@Override
	public Collection<OAuth2AccessToken> findTokensByClientIdAndUserName(final String clientId, final String username) {
		return accessTokenRepository.findByClientIdAndUsername(clientId, username).stream().filter(at -> at != null).map(at -> deserializeAccessToken(at.getToken())).collect(Collectors.toList());
	}

	@Override
	public Collection<OAuth2AccessToken> findTokensByClientId(final String clientId) {
		return accessTokenRepository.findByClientId(clientId).stream().filter(at -> at != null).map(at -> deserializeAccessToken(at.getToken())).collect(Collectors.toList());
	}

	protected String extractTokenKey(final String value) {
		if (value == null) {
			return null;
		}
		MessageDigest digest;
		try {
			digest = MessageDigest.getInstance("MD5");
		} catch (final NoSuchAlgorithmException e) {
			throw new IllegalStateException("MD5 algorithm not available.  Fatal (should be in the JDK).");
		}

		try {
			final byte[] bytes = digest.digest(value.getBytes("UTF-8"));
			return String.format("%032x", new BigInteger(1, bytes));
		} catch (final UnsupportedEncodingException e) {
			throw new IllegalStateException("UTF-8 encoding not available.  Fatal (should be in the JDK).");
		}
	}

	protected byte[] serializeAccessToken(final OAuth2AccessToken token) {
		try {
			return SerializationUtils.serialize(token);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

	protected byte[] serializeRefreshToken(final OAuth2RefreshToken token) {
		try {
			return SerializationUtils.serialize(token);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

	protected byte[] serializeAuthentication(final OAuth2Authentication authentication) {
		try {
			return SerializationUtils.serialize(authentication);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

	protected OAuth2AccessToken deserializeAccessToken(final byte[] token) {
		try {
			return SerializationUtils.deserialize(token);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

	protected OAuth2RefreshToken deserializeRefreshToken(final byte[] token) {
		try {
			return SerializationUtils.deserialize(token);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

	protected OAuth2Authentication deserializeAuthentication(final byte[] authentication) {
		try {
			return SerializationUtils.deserialize(authentication);
		} catch (final Throwable e) {
			LOG.warn(e.getMessage() + ". Returning null.");
			return null;
		}
	}

}
