/*
 * Copyright 2016 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.JoinColumn;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.Table;
import javax.persistence.Transient;

import org.apache.commons.lang3.StringUtils;
import org.genesys.blocks.model.AuditedVersionedModel;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.provider.ClientDetails;

@Entity
@Table(name = "oauthclient")
public class OAuthClient extends AuditedVersionedModel implements ClientDetails {

	private static final long serialVersionUID = -4204753722663196007L;

	@Column(unique = true, nullable = false, length = 100)
	private String clientId;

	@Column(nullable = true, length = 100)
	private String clientSecret;

	@Column(nullable = true, length = 200)
	private String resource;
	@Transient
	private final Set<String> resourceIds = new HashSet<>();

	@Column(nullable = false)
	boolean autoApprove = false;
	@Column(nullable = true, length = 200)
	private String autoApproveScope;
	@Transient
	private final Set<String> autoApproveScopes = new HashSet<>();

	@Column(nullable = true, length = 200)
	private String scope;
	@Transient
	private final Set<String> scopes = new HashSet<>();

	@Column(nullable = true, length = 200)
	private String grants;
	@Transient
	private final Set<String> grantTypes = new HashSet<>();

	@Column(nullable = true, length = 200)
	private String redirect;
	@Transient
	private final Set<String> redirectUris = new HashSet<>();

	@ElementCollection
	@Enumerated(EnumType.STRING)
	@CollectionTable(name = "clientrole", joinColumns = @JoinColumn(name = "clientId"))
	@Column(name = "oauthclientrole")
	private Collection<OAuthRole> roles = new ArrayList<>();

	@Transient
	private final Map<String, Object> additionalInformation = null;

	private Integer accessTokenValidity;
	private Integer refreshTokenValidity;

	@PrePersist
	private void flatten() {
		resource = resourceIds.stream().collect(Collectors.joining(";"));
		scope = scopes.stream().collect(Collectors.joining(";"));
		autoApproveScope = autoApproveScopes.stream().collect(Collectors.joining(";"));
		grants = grantTypes.stream().collect(Collectors.joining(";"));
		redirect = redirectUris.stream().collect(Collectors.joining(";"));
	}

	@PostLoad
	private void inflate() {
		if (resource != null) {
			Arrays.stream(StringUtils.split(resource, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(r -> resourceIds.add(r));
		}
		if (scope != null) {
			Arrays.stream(StringUtils.split(scope, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(s -> scopes.add(s));
		}
		if (autoApproveScope != null) {
			Arrays.stream(StringUtils.split(autoApproveScope, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(s -> autoApproveScopes.add(s));
		}
		if (grants != null) {
			Arrays.stream(StringUtils.split(grants, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(g -> grantTypes.add(g));
		}
		if (redirect != null) {
			Arrays.stream(StringUtils.split(redirect, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(u -> redirectUris.add(u));
		}
	}

	@Override
	public String getClientId() {
		return clientId;
	}

	public void setClientId(final String clientId) {
		this.clientId = clientId;
	}

	@Override
	public String getClientSecret() {
		return clientSecret;
	}

	public void setClientSecret(final String clientSecret) {
		this.clientSecret = clientSecret;
	}

	public String getResource() {
		return resource;
	}

	public void setResource(final String resource) {
		this.resource = resource;
	}

	public String getGrants() {
		return grants;
	}

	public void setGrants(final String grants) {
		this.grants = grants;
	}

	public String getRedirect() {
		return redirect;
	}

	public void setRedirect(final String redirect) {
		this.redirect = redirect;
	}

	public Collection<OAuthRole> getRoles() {
		return roles;
	}

	public void setRoles(final Collection<OAuthRole> roles) {
		this.roles = roles;
	}

	public void setScope(final String scope) {
		this.scope = scope;
	}

	public void setAutoApprove(final boolean autoApprove) {
		this.autoApprove = autoApprove;
	}

	public boolean getAutoApprove() {
		return autoApprove;
	}

	protected String getAutoApproveScope() {
		return autoApproveScope;
	}

	protected void setAutoApproveScope(final String autoApproveScope) {
		this.autoApproveScope = autoApproveScope;
	}

	public Set<String> getAutoApproveScopes() {
		return autoApproveScopes;
	}

	@Override
	public Set<String> getResourceIds() {
		return resourceIds;
	}

	/**
	 * Client secret is required when provided
	 */
	@Override
	public boolean isSecretRequired() {
		return clientSecret != null;
	}

	@Override
	public boolean isScoped() {
		return !scopes.isEmpty();
	}

	@Override
	public Set<String> getScope() {
		return scopes;
	}

	@Override
	public Set<String> getAuthorizedGrantTypes() {
		return grantTypes;
	}

	@Override
	public Set<String> getRegisteredRedirectUri() {
		return redirectUris;
	}

	@Override
	public Collection<GrantedAuthority> getAuthorities() {
		return roles.stream().collect(Collectors.toList());
	}

	@Override
	public Integer getAccessTokenValiditySeconds() {
		return accessTokenValidity;
	}

	public Integer getAccessTokenValidity() {
		return accessTokenValidity;
	}

	public void setAccessTokenValidity(final Integer accessTokenValidity) {
		this.accessTokenValidity = accessTokenValidity;
	}

	@Override
	public Integer getRefreshTokenValiditySeconds() {
		return refreshTokenValidity;
	}

	public Integer getRefreshTokenValidity() {
		return refreshTokenValidity;
	}

	public void setRefreshTokenValidity(final Integer refreshTokenValidity) {
		this.refreshTokenValidity = refreshTokenValidity;
	}

	@Override
	public boolean isAutoApprove(final String scope) {
		return autoApprove || autoApproveScopes.contains(scope);
	}

	@Override
	public Map<String, Object> getAdditionalInformation() {
		return additionalInformation;
	}

}
