/*
 * Decompiled with CFR 0.152.
 */
package com.apple.foundationdb.record.lucene.directory;

import com.apple.foundationdb.record.RecordCoreException;
import com.apple.foundationdb.record.logging.KeyValueLogMessage;
import com.apple.foundationdb.record.lucene.LuceneLogMessageKeys;
import com.apple.foundationdb.record.provider.common.CipherPool;
import com.apple.foundationdb.record.provider.common.CompressedAndEncryptedSerializerState;
import com.apple.foundationdb.record.provider.common.SerializationKeyManager;
import java.io.IOException;
import java.security.GeneralSecurityException;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import org.apache.lucene.codecs.compressing.CompressionMode;
import org.apache.lucene.codecs.compressing.Compressor;
import org.apache.lucene.codecs.compressing.Decompressor;
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.ByteBuffersDataOutput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.BytesRef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LuceneSerializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(LuceneSerializer.class);
    private static final int ENCODING_ENCRYPTED = 1;
    private static final int ENCODING_COMPRESSED = 2;
    private static final byte COMPRESSION_VERSION_FOR_HIGH_COMPRESSION = 0;
    private static final int ENCRYPTION_KEY_SHIFT = 3;
    private final boolean compressionEnabled;
    private final boolean encryptionEnabled;
    @Nullable
    private final SerializationKeyManager keyManager;

    public LuceneSerializer(boolean compressionEnabled, boolean encryptionEnabled, @Nullable SerializationKeyManager keyManager) {
        this.compressionEnabled = compressionEnabled;
        this.encryptionEnabled = encryptionEnabled;
        this.keyManager = keyManager;
    }

    public boolean isCompressionEnabled() {
        return this.compressionEnabled;
    }

    public boolean isEncryptionEnabled() {
        return this.encryptionEnabled;
    }

    @Nullable
    public SerializationKeyManager getKeyManager() {
        return this.keyManager;
    }

    @Nullable
    public byte[] encode(@Nullable byte[] data) {
        byte[] encoded;
        if (data == null) {
            return null;
        }
        CompressedAndEncryptedSerializerState state = new CompressedAndEncryptedSerializerState();
        long prefix = 0L;
        if (this.compressionEnabled) {
            prefix |= 2L;
            state.setCompressed(true);
        }
        if (this.encryptionEnabled) {
            if (this.keyManager == null) {
                throw new RecordCoreException("cannot encrypt Lucene blocks without keys", new Object[0]);
            }
            int key = this.keyManager.getSerializationKey();
            prefix |= 1L | ((long)key & 0xFFFFFFFFL) << 3;
            state.setEncrypted(true);
            state.setKeyNumber(key);
        }
        try {
            ByteBuffersDataOutput decodedDataOutput = new ByteBuffersDataOutput();
            decodedDataOutput.writeVLong(prefix);
            int prefixLength = (int)decodedDataOutput.size();
            encoded = LuceneSerializer.compressIfNeeded(state, decodedDataOutput, data, prefixLength);
            encoded = this.encryptIfNeeded(state, encoded, prefixLength);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new RecordCoreException("Lucene data encoding failure", (Throwable)ex);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(KeyValueLogMessage.of((String)"Encoded lucene data", (Object[])new Object[]{LuceneLogMessageKeys.COMPRESSION_SUPPOSED, this.compressionEnabled, LuceneLogMessageKeys.ENCRYPTION_SUPPOSED, this.encryptionEnabled, LuceneLogMessageKeys.COMPRESSED_EVENTUALLY, state.isCompressed(), LuceneLogMessageKeys.ENCRYPTED_EVENTUALLY, state.isEncrypted(), LuceneLogMessageKeys.ORIGINAL_DATA_SIZE, data.length, LuceneLogMessageKeys.ENCODED_DATA_SIZE, encoded.length}));
        }
        return encoded;
    }

    @Nullable
    public byte[] decode(@Nullable byte[] data) {
        byte[] decoded;
        if (data == null) {
            return null;
        }
        if (data.length < 2) {
            throw new RecordCoreException("Invalid data", new Object[0]).addLogInfo(new Object[]{LuceneLogMessageKeys.DATA_VALUE, data});
        }
        CompressedAndEncryptedSerializerState state = new CompressedAndEncryptedSerializerState();
        try {
            ByteArrayDataInput encodedDataInput = new ByteArrayDataInput(data);
            long prefix = encodedDataInput.readVLong();
            state.setCompressed((prefix & 2L) != 0L);
            state.setEncrypted((prefix & 1L) != 0L);
            state.setKeyNumber((int)(prefix >> 3));
            this.decryptIfNeeded(state, encodedDataInput);
            this.decompressIfNeeded(state, encodedDataInput);
            decoded = new byte[encodedDataInput.length() - encodedDataInput.getPosition()];
            encodedDataInput.readBytes(decoded, 0, decoded.length);
        }
        catch (IOException | GeneralSecurityException ex) {
            throw new RecordCoreException("Lucene data decoding failure", (Throwable)ex);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(KeyValueLogMessage.of((String)"Decoded lucene data", (Object[])new Object[]{LuceneLogMessageKeys.COMPRESSED_EVENTUALLY, state.isCompressed(), LuceneLogMessageKeys.ENCRYPTED_EVENTUALLY, state.isEncrypted(), LuceneLogMessageKeys.ENCODED_DATA_SIZE, data.length, LuceneLogMessageKeys.ORIGINAL_DATA_SIZE, decoded.length}));
        }
        return decoded;
    }

    @Nonnull
    private static byte[] compressIfNeeded(@Nonnull CompressedAndEncryptedSerializerState state, @Nonnull ByteBuffersDataOutput encodedDataOutput, @Nonnull byte[] uncompressedData, int prefixLength) throws IOException {
        if (!state.isCompressed()) {
            return LuceneSerializer.fallBackToUncompressed(state, uncompressedData, encodedDataOutput.toArrayCopy(), prefixLength);
        }
        try (Compressor compressor = CompressionMode.HIGH_COMPRESSION.newCompressor();){
            encodedDataOutput.writeByte((byte)0);
            encodedDataOutput.writeVInt(uncompressedData.length);
            compressor.compress(uncompressedData, 0, uncompressedData.length, (DataOutput)encodedDataOutput);
            byte[] compressedData = encodedDataOutput.toArrayCopy();
            if (compressedData.length < uncompressedData.length) {
                state.setCompressed(true);
                byte[] byArray = compressedData;
                return byArray;
            }
            byte[] byArray = LuceneSerializer.fallBackToUncompressed(state, uncompressedData, compressedData, prefixLength);
            return byArray;
        }
    }

    private static byte[] fallBackToUncompressed(@Nonnull CompressedAndEncryptedSerializerState state, @Nonnull byte[] originalData, @Nonnull byte[] encodedData, int prefixLength) {
        byte[] encoded = new byte[originalData.length + prefixLength];
        System.arraycopy(encodedData, 0, encoded, 0, prefixLength);
        encoded[0] = (byte)(encoded[0] & 0xFFFFFFFD);
        System.arraycopy(originalData, 0, encoded, prefixLength, originalData.length);
        state.setCompressed(false);
        return encoded;
    }

    private void decompressIfNeeded(@Nonnull CompressedAndEncryptedSerializerState state, @Nonnull ByteArrayDataInput encodedDataInput) throws IOException {
        if (!state.isCompressed()) {
            return;
        }
        byte version = encodedDataInput.readByte();
        if (version != 0) {
            throw new RecordCoreException("Un-supported compression version", new Object[0]).addLogInfo(new Object[]{LuceneLogMessageKeys.COMPRESSION_VERSION, version});
        }
        BytesRef ref = new BytesRef();
        int originalLength = encodedDataInput.readVInt();
        Decompressor decompressor = CompressionMode.HIGH_COMPRESSION.newDecompressor();
        decompressor.decompress((DataInput)encodedDataInput, originalLength, 0, originalLength, ref);
        encodedDataInput.reset(ref.bytes, ref.offset, ref.length);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] encryptIfNeeded(@Nonnull CompressedAndEncryptedSerializerState state, @Nonnull byte[] encoded, int prefixLength) throws GeneralSecurityException {
        byte[] encrypted;
        if (!state.isEncrypted()) {
            return encoded;
        }
        byte[] ivData = new byte[16];
        this.keyManager.getRandom(state.getKeyNumber()).nextBytes(ivData);
        IvParameterSpec iv = new IvParameterSpec(ivData);
        Cipher cipher = CipherPool.borrowCipher((String)this.keyManager.getCipher(state.getKeyNumber()));
        try {
            cipher.init(1, this.keyManager.getKey(state.getKeyNumber()), iv);
            encrypted = cipher.doFinal(encoded, prefixLength, encoded.length - prefixLength);
        }
        finally {
            CipherPool.returnCipher((Cipher)cipher);
        }
        int totalSize = prefixLength + 16 + encrypted.length;
        byte[] withIv = new byte[totalSize];
        System.arraycopy(encoded, 0, withIv, 0, prefixLength);
        System.arraycopy(iv.getIV(), 0, withIv, prefixLength, 16);
        System.arraycopy(encrypted, 0, withIv, prefixLength + 16, encrypted.length);
        return withIv;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void decryptIfNeeded(@Nonnull CompressedAndEncryptedSerializerState state, @Nonnull ByteArrayDataInput encodedDataInput) throws GeneralSecurityException {
        byte[] decrypted;
        if (!state.isEncrypted()) {
            return;
        }
        if (this.keyManager == null) {
            throw new RecordCoreException("cannot decrypt Lucene blocks without keys", new Object[0]);
        }
        byte[] ivData = new byte[16];
        encodedDataInput.readBytes(ivData, 0, 16);
        IvParameterSpec iv = new IvParameterSpec(ivData);
        byte[] encrypted = new byte[encodedDataInput.length() - encodedDataInput.getPosition()];
        encodedDataInput.readBytes(encrypted, 0, encrypted.length);
        Cipher cipher = CipherPool.borrowCipher((String)this.keyManager.getCipher(state.getKeyNumber()));
        try {
            cipher.init(2, this.keyManager.getKey(state.getKeyNumber()), iv);
            decrypted = cipher.doFinal(encrypted);
        }
        finally {
            CipherPool.returnCipher((Cipher)cipher);
        }
        encodedDataInput.reset(decrypted);
    }
}

