/*
 * Decompiled with CFR 0.152.
 */
package com.apple.foundationdb.record.lucene.directory;

import com.apple.foundationdb.async.AsyncUtil;
import com.apple.foundationdb.record.RecordCoreException;
import com.apple.foundationdb.record.logging.KeyValueLogMessage;
import com.apple.foundationdb.record.logging.LogMessageKeys;
import com.apple.foundationdb.record.lucene.LuceneEvents;
import com.apple.foundationdb.record.lucene.LuceneExceptions;
import com.apple.foundationdb.record.lucene.LuceneLogMessageKeys;
import com.apple.foundationdb.record.lucene.directory.AgilityContext;
import com.apple.foundationdb.record.lucene.directory.FDBDirectory;
import com.apple.foundationdb.record.provider.foundationdb.FDBRecordContext;
import com.apple.foundationdb.tuple.ByteArrayUtil2;
import com.apple.foundationdb.tuple.Tuple;
import com.apple.foundationdb.util.LoggableException;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import javax.annotation.Nonnull;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FDBDirectoryLockFactory
extends LockFactory {
    final FDBDirectory directory;
    final int timeWindowMilliseconds;

    public FDBDirectoryLockFactory(FDBDirectory directory, int timeWindowMilliseconds) {
        this.directory = directory;
        this.timeWindowMilliseconds = (long)timeWindowMilliseconds > TimeUnit.SECONDS.toMillis(10L) ? timeWindowMilliseconds : (int)TimeUnit.MINUTES.toMillis(10L);
    }

    public Lock obtainLock(Directory dir, String lockName) throws IOException {
        try {
            return new FDBDirectoryLock(this.directory.getAgilityContext(), lockName, this.directory.fileLockKey(lockName), this.timeWindowMilliseconds);
        }
        catch (RecordCoreException | FDBDirectoryLockException ex) {
            throw LuceneExceptions.toIoException((Throwable)ex, null);
        }
    }

    @VisibleForTesting
    public Lock obtainLock(AgilityContext agilityContext, byte[] fileLockKey, String lockName) {
        return new FDBDirectoryLock(agilityContext, lockName, fileLockKey, this.timeWindowMilliseconds);
    }

    protected static class FDBDirectoryLock
    extends Lock {
        private static final Logger LOGGER = LoggerFactory.getLogger(FDBDirectoryLock.class);
        private final AgilityContext agilityContext;
        private final String lockName;
        private final UUID selfStampUuid = UUID.randomUUID();
        private long timeStampMillis;
        private final int timeWindowMilliseconds;
        private final byte[] fileLockKey;
        private boolean closed;
        private final long lockStartTime;
        private FDBRecordContext closingContext = null;
        private final Object fileLockSetLock = new Object();

        private FDBDirectoryLock(AgilityContext agilityContext, String lockName, byte[] fileLockKey, int timeWindowMilliseconds) {
            this.agilityContext = agilityContext;
            this.lockName = lockName;
            this.fileLockKey = fileLockKey;
            this.timeWindowMilliseconds = timeWindowMilliseconds;
            this.logSelf("FileLock: Attempt to create a file Lock");
            this.lockStartTime = System.nanoTime();
            this.fileLockSet(false);
            agilityContext.flush();
            agilityContext.setCommitCheck(this::ensureValidIfNotClosed);
            this.logSelf("FileLock: Successfully created a file lock");
        }

        private CompletableFuture<Void> ensureValidIfNotClosed(FDBRecordContext context) {
            return this.closed ? AsyncUtil.DONE : this.fileLockSet(true, context);
        }

        public void ensureValid() throws IOException {
            if (this.closed) {
                throw new AlreadyClosedException("Lock instance already released. This=" + String.valueOf((Object)this));
            }
            long now = System.currentTimeMillis();
            if (now > this.timeStampMillis + (long)this.timeWindowMilliseconds) {
                throw new AlreadyClosedException("Lock is too old. This=" + String.valueOf((Object)this) + " now=" + now);
            }
            try {
                this.fileLockSet(true);
            }
            catch (RecordCoreException ex) {
                throw LuceneExceptions.toIoException(ex, null);
            }
        }

        private byte[] fileLockValue() {
            return Tuple.from((Object[])new Object[]{this.selfStampUuid, this.timeStampMillis}).pack();
        }

        private static long fileLockValueToTimestamp(byte[] value) {
            return value == null ? 0L : Tuple.fromBytes((byte[])value).getLong(1);
        }

        private static UUID fileLockValueToUuid(byte[] value) {
            return value == null ? null : Tuple.fromBytes((byte[])value).getUUID(0);
        }

        public void fileLockSet(boolean isHeartbeat) {
            this.agilityContext.asyncToSync(LuceneEvents.Waits.WAIT_LUCENE_FILE_LOCK_SET, this.agilityContext.apply(aContext -> this.fileLockSet(isHeartbeat, (FDBRecordContext)aContext)));
        }

        private CompletableFuture<Void> fileLockSet(boolean isHeartbeat, FDBRecordContext aContext) {
            long nowMillis = System.currentTimeMillis();
            return aContext.ensureActive().get(this.fileLockKey).thenAccept(val -> {
                Object object = this.fileLockSetLock;
                synchronized (object) {
                    if (isHeartbeat && aContext.equals(this.closingContext)) {
                        if (val != null) {
                            long existingTimeStamp = FDBDirectoryLock.fileLockValueToTimestamp(val);
                            UUID existingUuid = FDBDirectoryLock.fileLockValueToUuid(val);
                            throw new AlreadyClosedException("Lock file re-obtained by " + String.valueOf(existingUuid) + " at " + existingTimeStamp + ". This=" + String.valueOf((Object)this));
                        }
                    } else {
                        if (isHeartbeat) {
                            this.fileLockCheckHeartBeat((byte[])val);
                        } else {
                            this.fileLockCheckNewLock((byte[])val, nowMillis);
                        }
                        this.timeStampMillis = nowMillis;
                        byte[] value = this.fileLockValue();
                        aContext.ensureActive().set(this.fileLockKey, value);
                    }
                }
            });
        }

        private void fileLockCheckHeartBeat(byte[] val) {
            long existingTimeStamp = FDBDirectoryLock.fileLockValueToTimestamp(val);
            UUID existingUuid = FDBDirectoryLock.fileLockValueToUuid(val);
            if (existingTimeStamp == 0L || existingUuid == null) {
                throw new AlreadyClosedException("Lock file was deleted. This=" + String.valueOf((Object)this));
            }
            if (existingUuid.compareTo(this.selfStampUuid) != 0) {
                throw new AlreadyClosedException("Lock file changed by " + String.valueOf(existingUuid) + " at " + existingTimeStamp + ". This=" + String.valueOf((Object)this));
            }
        }

        private void fileLockCheckNewLock(byte[] val, long nowMillis) {
            long existingTimeStamp = FDBDirectoryLock.fileLockValueToTimestamp(val);
            UUID existingUuid = FDBDirectoryLock.fileLockValueToUuid(val);
            if (existingUuid == null || existingTimeStamp <= 0L) {
                return;
            }
            if (existingTimeStamp > nowMillis - (long)this.timeWindowMilliseconds && existingTimeStamp < nowMillis + (long)this.timeWindowMilliseconds) {
                throw new FDBDirectoryLockException("FileLock: Lock failed: already locked by another entity").addLogInfo(new Object[]{LuceneLogMessageKeys.LOCK_EXISTING_TIMESTAMP, existingTimeStamp, LuceneLogMessageKeys.LOCK_EXISTING_UUID, existingUuid, LuceneLogMessageKeys.LOCK_DIRECTORY, this});
            }
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn(KeyValueLogMessage.of((String)"FileLock: discarded an existing old lock", (Object[])new Object[]{LuceneLogMessageKeys.LOCK_EXISTING_TIMESTAMP, existingTimeStamp, LuceneLogMessageKeys.LOCK_EXISTING_UUID, existingUuid, LuceneLogMessageKeys.LOCK_DIRECTORY, this}));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void fileLockClearFlushAndClose(boolean isRecovery) {
            Function fileLockFunc = aContext -> aContext.ensureActive().get(this.fileLockKey).thenAccept(val -> {
                Object object = this.fileLockSetLock;
                synchronized (object) {
                    UUID existingUuid = FDBDirectoryLock.fileLockValueToUuid(val);
                    if (existingUuid != null && existingUuid.compareTo(this.selfStampUuid) == 0) {
                        aContext.ensureActive().clear(this.fileLockKey);
                        this.closingContext = aContext;
                        this.logSelf(isRecovery ? "FileLock: Cleared in Recovery path" : "FileLock: Cleared");
                    } else if (!isRecovery) {
                        throw new AlreadyClosedException("FileLock: Expected to be locked during close.This=" + String.valueOf((Object)this) + " existingUuid=" + String.valueOf(existingUuid));
                    }
                }
            });
            if (this.agilityContext.isClosed()) {
                this.agilityContext.asyncToSync(LuceneEvents.Waits.WAIT_LUCENE_FILE_LOCK_CLEAR, this.agilityContext.applyInRecoveryPath(fileLockFunc));
            } else {
                this.agilityContext.asyncToSync(LuceneEvents.Waits.WAIT_LUCENE_FILE_LOCK_CLEAR, this.agilityContext.apply(fileLockFunc));
            }
            this.agilityContext.recordEvent(LuceneEvents.Events.LUCENE_FILE_LOCK_DURATION, System.nanoTime() - this.lockStartTime);
            boolean flushed = false;
            try {
                this.closed = true;
                this.agilityContext.flush();
                flushed = true;
            }
            finally {
                this.closed = flushed;
                this.closingContext = null;
            }
        }

        protected void fileLockClearIfLocked() {
            if (this.closed) {
                return;
            }
            this.fileLockClearFlushAndClose(true);
        }

        public void close() throws IOException {
            if (this.closed) {
                throw new AlreadyClosedException("Lock file is already closed. This=" + String.valueOf((Object)this));
            }
            try {
                this.fileLockClearFlushAndClose(false);
            }
            catch (RecordCoreException ex) {
                throw LuceneExceptions.toIoException(ex, null);
            }
        }

        public String toString() {
            return "{FDBDirectoryLock: name=" + this.lockName + " uuid=" + String.valueOf(this.selfStampUuid) + " timeMillis=" + this.timeStampMillis + "}";
        }

        private void logSelf(String staticMessage) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(KeyValueLogMessage.of((String)staticMessage, (Object[])new Object[]{LogMessageKeys.TIME_LIMIT_MILLIS, this.timeWindowMilliseconds, LuceneLogMessageKeys.LOCK_TIMESTAMP, Duration.ofNanos(this.lockStartTime).toMillis(), LuceneLogMessageKeys.LOCK_UUID, this.selfStampUuid, LogMessageKeys.KEY, ByteArrayUtil2.loggable((byte[])this.fileLockKey)}));
            }
        }
    }

    public static class FDBDirectoryLockException
    extends LoggableException {
        public FDBDirectoryLockException(@Nonnull String msg) {
            super(msg);
        }
    }
}

