/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.mongo.transform;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.flowstep.core.FlowComponent;
import org.flowstep.core.context.FlowStepContext;
import org.flowstep.core.model.step.FlowStepComponent;
import org.flowstep.core.transform.FlowTransform;

import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.TimeZone;

@FlowComponent
public class DateFormat implements FlowTransform, FlowStepComponent {
    private String format = "yyyy-MM-dd HH:mm:ss.SSS'Z'";
    private String timeZone;
    @JsonIgnore
    private SimpleDateFormat simpleDateFormat;
    @JsonIgnore
    private DateTimeFormatter dateTimeFormatter;

    public String getFormat() {
        return format;
    }

    public DateFormat setFormat(String format) {
        this.format = format;
        return this;
    }

    public String getTimeZone() {
        return timeZone;
    }

    public DateFormat setTimeZone(String timeZone) {
        this.timeZone = timeZone;
        return this;
    }

    @Override
    public void postConfigure() {
        simpleDateFormat = new SimpleDateFormat(format);
        dateTimeFormatter = DateTimeFormatter.ofPattern(format);
    }

    @Override
    public Object transform(Object value, FlowStepContext context) {
        if (value instanceof Date) {
            if (timeZone != null)
                simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timeZone));
            return simpleDateFormat.format(value);
        } else if (value instanceof LocalDateTime) {
            if (timeZone != null)
                dateTimeFormatter = dateTimeFormatter.withZone(ZoneId.of(timeZone));
            return ((LocalDateTime) value).format(dateTimeFormatter);
        } else
            return value;
    }
}


