/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.mongo;

import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Aggregates;
import com.mongodb.client.model.BsonField;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Projections;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.flowstep.core.DataProviderException;
import org.flowstep.core.connection.EnvironmentConnection;
import org.flowstep.core.context.FlowNewStepContext;
import org.flowstep.core.context.FlowStepContext;
import org.flowstep.core.processor.FlowStepDataProcessor;
import org.flowstep.mongo.model.MongoEnvironment;
import org.flowstep.mongo.model.MongoStep;
import org.flowstep.mongo.model.StepSort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import static com.mongodb.client.model.Sorts.orderBy;

@Component
public class MongoNewStepProcessor implements FlowStepDataProcessor<MongoEnvironment, MongoStep> {
    private static final Logger logger = LoggerFactory.getLogger(MongoNewStepProcessor.class);

    @Override
    public EnvironmentConnection getConnection(MongoEnvironment environmentItemSettings) {
        return new EnvironmentConnection(mongoTemplate(environmentItemSettings), environmentItemSettings);
    }

    @Override
    public Iterator<Document> process(FlowNewStepContext<MongoStep> flowStepContext, EnvironmentConnection connection) {

        if (connection != null) {
            try {
                MongoEnvironment mongoEnvironment = (MongoEnvironment) connection.getEnvironmentItemSettings();
                MongoTemplate mongoTemplate = connection.getConnectionTemplate();
                MongoCollection<Document> collection = mongoTemplate.getCollection(mongoEnvironment.getCollection());
                FlowStepContext stepContext = new FlowStepContext(flowStepContext.getStepPackageContext(),
                        flowStepContext.getStepGroup(), flowStepContext.getStep());


                Bson filter = buildAggregateFilter(flowStepContext.getStep(), stepContext);
                Bson sort = buildFindSorts(flowStepContext.getStep());
                Bson projection = buildProjection(flowStepContext.getStep());

                return retrieveData(flowStepContext.getStep(), collection, filter, projection,
                        sort, flowStepContext.getStep().getLimit()).iterator();

            } catch (DataAccessException e) {
                throw new DataProviderException("Data Provider Exception", e);
            }

        } else
            logger.error("Error: couldn't find connection for environment: {}", flowStepContext.getStep().getEnvironment());

        return null;
    }

    private List<Bson> buildGroupPipeline(MongoStep step, Bson filter) {
        Document aggregateFields = new Document();
        step.getFields().forEach(field -> {
            String fieldName = field.getId().replace(".", "");
            String fieldKey = "$" + field.getId();
            aggregateFields.append(fieldName, fieldKey);
        });

        BsonField aggregator = step.getGroup().getGroupOperator();

        List<Bson> aggregatedParams = new ArrayList<>();
        aggregatedParams.add(Aggregates.match(filter));
        aggregatedParams.add(Aggregates.group(aggregateFields, aggregator));

        if (!step.getSort().isEmpty()) {
            aggregatedParams.add(Aggregates.sort(getAggregatedSorts(step)));
        }

        return aggregatedParams;
    }

    private Bson buildFindSorts(MongoStep step) {
        List<Bson> sorts = step.getSort().stream()
                .map(StepSort::getSort)
                .collect(Collectors.toList());

        return orderBy(sorts);
    }

    private Bson getAggregatedSorts(MongoStep step) {
        String prefix = "_id";
        String groupName = step.getGroup().getFieldName();

        List<Bson> sorts = step.getSort().stream()
                .map(sort -> {
                    String currentPrefix = sort.getFieldName().equalsIgnoreCase(groupName) ? "" : prefix;
                    return sort.getSort(currentPrefix);
                })
                .collect(Collectors.toList());

        return orderBy(sorts);
    }

    private Bson buildProjection(MongoStep step) {
        List<String> ids = step.getFields()
                .stream()
                .map(field -> (field.getId().contains(".[")) ? field.getId().substring(0, field.getId().indexOf(".[")) : field.getId())
                .collect(Collectors.toList());

        return Projections.fields(Projections.include(ids));
    }

    private Bson buildAggregateFilter(MongoStep step, FlowStepContext stepContext) {
        List<Bson> filterGroups = new ArrayList<>();

        step.getFilterGroups().forEach(group -> {
            Bson groupedFilter = group.getType()
                    .getOperatorFunction()
                    .apply(group.getFilters().stream()
                            .map(filter -> {
                                filter.processValueFromTransform(stepContext);
                                return filter.getCondition()
                                        .getFilter(filter);
                            })
                            .collect(Collectors.toList()));

            filterGroups.add(groupedFilter);
        });

        return filterGroups.size() > 1 ? Filters.or(filterGroups) : Filters.and(filterGroups);
    }

    private Iterable<Document> retrieveData(MongoStep step, MongoCollection<Document> collection, Bson filter, Bson projection, Bson sort, Integer limit) {

        if (step.getGroup() != null) {
            List<Bson> groupPipeline = buildGroupPipeline(step, filter);
            return collection.aggregate(groupPipeline).allowDiskUse(true);

        } else {

            FindIterable<Document> findResults = collection.find(filter)
                    .projection(projection)
                    .sort(sort)
                    .allowDiskUse(true)
                    .noCursorTimeout(true);

            if (limit > 0)
                findResults.limit(limit);

            return findResults;
        }
    }


    private MongoClient mongoClient(MongoEnvironment settings) {
        return MongoClients.create(settings.getUri());
    }

    private MongoTemplate mongoTemplate(MongoEnvironment settings) {
        return new MongoTemplate(mongoClient(settings), settings.getDatabase());
    }
}
