/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.mongo.model;

import com.mongodb.client.model.Filters;
import org.apache.commons.lang3.StringUtils;
import org.bson.conversions.Bson;
import org.flowstep.core.factory.FlowTransformFactory;

import java.util.List;
import java.util.function.Function;
import java.util.regex.Pattern;

public enum FilterCondition {

    /**
     * Greater than
     */
    GT(filter -> Filters.gt(filter.getFieldName(), filter.getFieldValue())),//ConverterFactory.FILTER_LONG.get().convert(filter.getFieldValue())

    /**
     * Greater or equal than
     */
    GTE(filter -> Filters.gte(filter.getFieldName(), filter.getFieldValue())),//ConverterFactory.FILTER_LONG.get().convert(filter.getFieldValue())

    /**
     * Exists
     */
    EXISTS(filter -> Filters.exists(filter.getFieldName(), (Boolean) FlowTransformFactory.FILTER_BOOLEAN.get().transform(filter.getFieldValue()))),

    /**
     * Less than
     */
    LT(filter -> Filters.lt(filter.getFieldName(), filter.getFieldValue())), //ConverterFactory.FILTER_LONG.get().convert(filter.getFieldValue())

    /**
     * Less or equal than
     */
    LTE(filter -> Filters.lte(filter.getFieldName(), filter.getFieldValue())), //ConverterFactory.FILTER_LONG.get().convert(filter.getFieldValue())

    /**
     * Not equal
     */
    NE(filter -> Filters.ne(filter.getFieldName(), filter.getFieldValue())),
    /**
     * In
     */
    IN(filter -> Filters.in(filter.getFieldName(),
            (List<?>) FlowTransformFactory.FILTER_SET_OF_INTEGER.get().transform(filter.getFieldValue()))),

    /**
     * Not in
     */
    NIN(filter -> Filters.nin(filter.getFieldName(),
            FlowTransformFactory.FILTER_SET_OF_STRING.get().transform(filter.getFieldValue()))),

    /**
     * Size
     */
    SIZE(filter -> Filters.size(filter.getFieldName(),
            (Integer) FlowTransformFactory.FILTER_INTEGER.get().transform(filter.getFieldValue()))),

    /**
     * Regular expression
     */
    REGEX(filter -> Filters.regex(filter.getFieldName(), Pattern.compile(filter.getFieldValue() != null ?
            String.valueOf(filter.getFieldValue()) : "", Pattern.CASE_INSENSITIVE))),

    /**
     * Equal
     */
    EQ(filter -> {
        if (StringUtils.equalsAnyIgnoreCase(String.valueOf(filter.getFieldValue()), "true", "false")) {
            return Filters.eq(filter.getFieldName(),
                    (Boolean) FlowTransformFactory.FILTER_BOOLEAN.get().transform(filter.getFieldValue()));
        } else
            return Filters.eq(filter.getFieldName(), (String) filter.getFieldValue());
    });

    private final Function<MongoStepFilter, Bson> filterFunction;

    FilterCondition(Function<MongoStepFilter, Bson> filterFunction) {
        this.filterFunction = filterFunction;
    }

    public Bson getFilter(MongoStepFilter filter) {
        return filterFunction.apply(filter);
    }
}
