/*
 *
 *  Copyright 2022 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.flowstep.mongo;

import org.flowstep.core.connection.EnvironmentConnection;
import org.flowstep.core.context.FlowPackageContext;
import org.flowstep.core.factory.FlowEnvironmentFactory;
import org.flowstep.core.model.step.FlowGroup;
import org.flowstep.core.model.step.FlowStep;
import org.flowstep.core.processor.FlowProcessor;
import org.flowstep.core.processor.FlowProcessorType;
import org.flowstep.mongo.model.MongoStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class MongoStepProcessor implements FlowProcessor {
    private static final Logger logger = LoggerFactory.getLogger(MongoStepProcessor.class);

    @Autowired
    private FlowEnvironmentFactory environmentFactory;

    @Override
    public FlowProcessorType getType() {
        return FlowProcessorType.PROCESSOR;
    }

    @Override
    public boolean canProcess(FlowStep step) {
        return step instanceof MongoStep;
    }

    @Override
    public boolean process(FlowGroup stepGroup, FlowStep step, FlowPackageContext stepPackageContext) {
        MongoStep mongoStep = (MongoStep) step;
        EnvironmentConnection connection = environmentFactory.getEnvironmentConnection(mongoStep.getEnvironment(),
                stepPackageContext.getStepPackage());

        if (connection != null) {
            new MongoDataProvider()
                    .setStepGroup(stepGroup)
                    .setStep(mongoStep)
                    .setConnection(connection)
                    .setStepPackageContext(stepPackageContext)
                    .build();

        } else
            logger.error("Error: couldn't find connection for environment: {}", mongoStep.getEnvironment());

        return true;
    }
}
