package org.flowstep.mongo.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.flowstep.core.CloneableList;
import org.flowstep.core.FilterDependencyLink;
import org.flowstep.core.model.step.FlowStepComponent;
import org.flowstep.core.model.step.FlowStepData;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
public class MongoStep extends FlowStepData implements HasStepFilters, FlowStepComponent {

    @JsonProperty("filterGroups")
    private CloneableList<StepFilterGroup> filterGroups;

    @JsonProperty("sort")
    private CloneableList<StepSort> sort;

    @JsonProperty("group")
    private Group group;

    public MongoStep() {
        super();
        filterGroups = new CloneableList<>();
        sort = new CloneableList<>();
    }

    public MongoStep(MongoStep step) {
        super(step);
        filterGroups = new CloneableList<>(step.filterGroups);

        if (step.sort != null)
            sort = new CloneableList<>(step.sort);

        if (step.group != null)
            group = new Group(step.group);
    }

    public CloneableList<StepFilterGroup> getFilterGroups() {
        return filterGroups;
    }

    public void setFilterGroups(CloneableList<StepFilterGroup> filterGroups) {
        this.filterGroups = filterGroups;
    }

    @Override
    public List<String> getStepDependencies() {
        return getAllFilters().map(filter -> new FilterDependencyLink(filter.getStep()).getStepName())
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toList());
    }

    public CloneableList<StepSort> getSort() {
        return sort;
    }

    public MongoStep setSort(CloneableList<StepSort> sort) {
        this.sort = sort;
        return this;
    }

    public Group getGroup() {
        return group;
    }

    public MongoStep setGroup(Group group) {
        this.group = group;
        return this;
    }

    @JsonIgnore
    public Stream<MongoStepFilter> getAllFilters() {
        return filterGroups.stream().flatMap(filterGroup -> filterGroup.getFilters().stream());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (o == null || getClass() != o.getClass()) return false;

        MongoStep mongoStep = (MongoStep) o;

        return new EqualsBuilder().appendSuper(super.equals(o)).append(filterGroups, mongoStep.filterGroups).append(sort, mongoStep.sort).append(group, mongoStep.group).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37).appendSuper(super.hashCode()).append(filterGroups).append(sort).append(group).toHashCode();
    }
}
