package org.flowstep.mongo;

import org.apache.commons.lang3.StringUtils;
import org.flowstep.core.FilterDependencyLink;
import org.flowstep.core.context.FlowPackageContext;
import org.flowstep.core.model.step.FlowPackage;
import org.flowstep.core.model.step.FlowStepData;
import org.flowstep.core.processor.FlowPackageProcessor;
import org.flowstep.core.processor.FlowProcessorType;
import org.flowstep.mongo.model.HasStepFilters;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.Set;

@Component
public class StepFilterDependenciesProcessor implements FlowPackageProcessor {

    @Override
    public FlowProcessorType getType() {
        return FlowProcessorType.PREPROCESSOR;
    }

    @Override
    public boolean process(FlowPackageContext context) {
        processPrimaryKeys(context.getStepPackage());
        return true;
    }

    public Set<String> getFilterDependencies(FlowPackage stepPackage) {
        Set<String> result = new HashSet<>();
        stepPackage.getStepGroups().forEach(stepGroup -> stepGroup.getSteps().forEach(step -> {

            if (step instanceof HasStepFilters) {
                HasStepFilters dataStep = (HasStepFilters) step;
                dataStep.getAllFilters()
                        .filter(filter -> StringUtils.isNotBlank(filter.getStep()))
                        .forEach(filter -> result.add(filter.getStep()));
            }

        }));

        return result;
    }

    private void processPrimaryKeys(FlowPackage stepPackage) {
        Set<String> filterDependencies = getFilterDependencies(stepPackage);

        if (!filterDependencies.isEmpty()) {
            filterDependencies.forEach(primaryKeyLink -> {
                FilterDependencyLink link = new FilterDependencyLink(primaryKeyLink);

                if (link.isCorrect()) {
                    stepPackage.getStepGroups()
                            .stream()
                            .flatMap(group -> group.getSteps().stream())
                            .forEach(step -> {
                                if (step instanceof FlowStepData) {
                                    trySetPrimaryKey((FlowStepData) step, link);
                                }
                            });
                }
            });
        }
    }

    private void trySetPrimaryKey(FlowStepData dataStep, FilterDependencyLink link) {
        if (dataStep.getId().equals(link.getStepName())) {
            dataStep.getFields()
                    .stream()
                    .filter(field -> field.getName().equals(link.getFieldName()))
                    .findFirst()
                    .ifPresent(field -> {
                        field.setPrimaryKey(true);
                        dataStep.setHasPrimaryKey(true);
                    });
        }
    }
}
