/*
 * Fabric3
 * Copyright (c) 2009-2015 Metaform Systems
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Portions originally based on Apache Tuscany 2007
 * licensed under the Apache 2.0 license.
 */
package org.fabric3.implementation.proxy.jdk.wire;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Map;

import org.fabric3.api.host.Fabric3Exception;
import org.fabric3.spi.container.invocation.Message;
import org.fabric3.spi.container.invocation.MessageCache;
import org.fabric3.spi.container.invocation.WorkContext;
import org.fabric3.spi.container.invocation.WorkContextCache;
import org.fabric3.spi.container.wire.Interceptor;
import org.fabric3.spi.container.wire.InvocationChain;
import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.ServiceRuntimeException;
import org.oasisopen.sca.ServiceUnavailableException;

/**
 * Dispatches from a proxy to a wire.
 */
@SuppressWarnings("NonSerializableFieldInSerializableClass")
public final class JDKInvocationHandler<B> implements InvocationHandler, ServiceReference<B> {
    private static final long serialVersionUID = -5841336280391145583L;
    private Class<B> interfaze;
    private B proxy;
    private Map<Method, InvocationChain> chains;
    private String callbackUri;

    /**
     * Constructor.
     *
     * @param interfaze   the proxy interface
     * @param callbackUri the callback uri or null if the wire is unidirectional
     * @param mapping     the method to invocation chain mappings for the wire
     */
    public JDKInvocationHandler(Class<B> interfaze, String callbackUri, Map<Method, InvocationChain> mapping) {
        this.callbackUri = callbackUri;
        this.interfaze = interfaze;
        this.chains = mapping;
    }

    public B getService() {
        if (proxy == null) {
            ClassLoader loader = interfaze.getClassLoader();
            this.proxy = interfaze.cast(Proxy.newProxyInstance(loader, new Class[]{interfaze}, this));
        }
        return proxy;
    }

    public Class<B> getBusinessInterface() {
        return interfaze;
    }

    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        InvocationChain chain = chains.get(method);
        if (chain == null) {
            return handleProxyMethod(method, args);
        }

        Interceptor headInterceptor = chain.getHeadInterceptor();
        assert headInterceptor != null;

        WorkContext workContext = WorkContextCache.getThreadWorkContext();

        if (callbackUri != null) {
            workContext.addCallbackReference(callbackUri);
        }

        Message message = MessageCache.getAndResetMessage();
        message.setBody(args);
        message.setWorkContext(workContext);
        try {
            // dispatch the invocation down the chain and get the response
            Message response;
            try {
                response = headInterceptor.invoke(message);
            } catch (ServiceRuntimeException e) {
                // simply rethrow ServiceRuntimeException
                throw e;
            } catch (RuntimeException e) {
                // wrap other exceptions raised by the runtime
                throw new ServiceUnavailableException(e);
            }

            // handle response from the application, returning or throwing an exception as appropriate
            Object body = response.getBody();
            boolean fault = response.isFault();
            if (fault) {
                throw (Throwable) body;
            } else {
                return body;
            }
        } finally {
            if (callbackUri != null) {
                workContext.popCallbackReference();
            }
            message.reset();
        }

    }

    public ServiceReference<B> getServiceReference() {
        return this;
    }

    private Object handleProxyMethod(Method method, Object[] args) throws Fabric3Exception {
        if (method.getParameterTypes().length == 0 && "toString".equals(method.getName())) {
            return "[Proxy - " + Integer.toHexString(hashCode()) + "]";
        } else if (method.getDeclaringClass().equals(Object.class) && "equals".equals(method.getName()) && args.length == 1) {
            return proxyEquals(args[0]);
        } else if (Object.class.equals(method.getDeclaringClass()) && "hashCode".equals(method.getName())) {
            return hashCode();
            // TODO better hash algorithm
        }
        String op = method.getName();
        throw new Fabric3Exception("Operation not configured: " + op);
    }

    private Object proxyEquals(Object other) {
        if (other == null) {
            return false;
        }
        if (!Proxy.isProxyClass(other.getClass())) {
            return false;
        }
        Object otherHandler = Proxy.getInvocationHandler(other);
        if (!(otherHandler instanceof JDKInvocationHandler)) {
            return false;
        }
        JDKInvocationHandler otherJDKHandler = (JDKInvocationHandler) otherHandler;
        return chains.equals(otherJDKHandler.chains);
    }
}
