package org.eu.sbin.commons.opc.ua.client.cmd;

import org.eclipse.milo.opcua.sdk.client.OpcUaClient;
import org.eclipse.milo.opcua.stack.client.security.DefaultClientCertificateValidator;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.security.DefaultTrustListManager;
import org.eclipse.milo.opcua.stack.core.types.builtin.DataValue;
import org.eclipse.milo.opcua.stack.core.types.builtin.LocalizedText;
import org.eclipse.milo.opcua.stack.core.types.builtin.NodeId;
import org.eclipse.milo.opcua.stack.core.types.builtin.Variant;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.UShort;
import org.eclipse.milo.opcua.stack.core.types.enumerated.TimestampsToReturn;
import org.eu.sbin.commons.opc.ua.client.IOpcUaClient;
import org.eu.sbin.commons.opc.ua.client.KeyStoreLoader;
import org.eu.vooo.commons.iot.cmd.AbstractCmdCollector;
import org.eu.vooo.commons.iot.common.RunStatus;
import org.eu.vooo.commons.iot.device.IDevice;
import org.eu.vooo.commons.iot.device.IIpParameter;
import org.eu.vooo.commons.iot.exception.XCollectDeviceException;
import org.eu.vooo.commons.iot.exception.XCollectSensorException;
import org.eu.vooo.commons.iot.exception.XCollectorInitException;
import org.eu.vooo.commons.iot.exception.XNetworkException;
import org.eu.vooo.commons.iot.sensor.ISensor;
import org.eu.vooo.commons.iot.sensor.IUASensor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import static org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned.uint;

/**
 * @author lxj990@gmail.com
 */
public class CmdOpcUaCollector<D extends IDevice> extends AbstractCmdCollector implements IOpcUaClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(CmdOpcUaCollector.class);

    private OpcUaClient opcUaClient;

    private UShort namespaceSpaceIndex;

    private final CompletableFuture<OpcUaClient> opcUaClientCompletableFuture = new CompletableFuture<>();

    public CmdOpcUaCollector(Builder<D> builder) {
        this.deviceHolder = builder.deviceHolder;
    }

    @Override
    protected void init() {
        try {
            run();
        } catch (Exception e) {
            LOGGER.warn("创建OPC UA CLIENT 失败");
            throw new XCollectorInitException("初始化 OPC UA CLIENT 失败");
        }
    }

    @Override
    public synchronized void tryConnect(IDevice device) {
        if (Objects.isNull(device.getIpParameter())) {
            throw new XNetworkException("连接参数为空");
        }
        try {
            if (Objects.nonNull(opcUaClient) && !RunStatus.ONLINE.equals(deviceHolder.getRunStatus())) {
                opcUaClient.connect().get(15, TimeUnit.SECONDS);
            }
        } catch (TimeoutException | InterruptedException | ExecutionException e) {
            throw new XNetworkException("连接失败");
        }
    }

    @Override
    public void read(ISensor sensor) {
        try {
            StringBuilder nodeIdPrepare = new StringBuilder();
            nodeIdPrepare.append("ns=");
            nodeIdPrepare.append(namespaceSpaceIndex.intValue());
            nodeIdPrepare.append(";");
            nodeIdPrepare.append("s=");
            nodeIdPrepare.append(((IUASensor) sensor).getUaChannel());
            nodeIdPrepare.append(".");
            nodeIdPrepare.append(((IUASensor) sensor).getUaDevice());
            nodeIdPrepare.append(".");
            nodeIdPrepare.append(((IUASensor) sensor).getUaTag());
            NodeId nodeId = NodeId.parse(nodeIdPrepare.toString());
            DataValue dataValue = opcUaClient.readValue(0.0, TimestampsToReturn.Neither, nodeId).get();
            final Variant variant = dataValue.getValue();
            final Object value = variant.getValue();
            sensor.setValue(value.toString());
        } catch (InterruptedException | ExecutionException e) {
            if (e.getCause() instanceof UaException) {
                throw new XCollectDeviceException("设备异常");
            } else {
                throw new XCollectSensorException("获取传感器异常");
            }
        }
    }

    @Override
    public <T extends ISensor> void read(List<T> sensorList) throws Exception {

    }

    @Override
    public synchronized void tryDisconnect(IDevice device) {
        this.opcUaClient.disconnect();
    }

    private OpcUaClient createClient() throws Exception {
        Path securityTempDir = Paths.get(System.getProperty("java.io.tmpdir"), "client", "security");
        Files.createDirectories(securityTempDir);
        if (!Files.exists(securityTempDir)) {
            throw new Exception("无法创建安全配置目录: " + securityTempDir);
        }
        File pkiDir = securityTempDir.resolve("pki").toFile();
        KeyStoreLoader loader = new KeyStoreLoader().load(securityTempDir);
        DefaultTrustListManager trustListManager = new DefaultTrustListManager(pkiDir);
        DefaultClientCertificateValidator certificateValidator =
                new DefaultClientCertificateValidator(trustListManager);
        StringBuilder endpointUrl = new StringBuilder();
        IIpParameter ipParameter = deviceHolder.getIpParameter();
        if (Objects.isNull(ipParameter)) {
            throw new XCollectorInitException("网络配置异常");
        }
        endpointUrl
                .append("opc.tcp://")
                .append(ipParameter.getHost())
                .append(":")
                .append(ipParameter.getPort());
        return OpcUaClient.create(
                endpointUrl.toString(),
                endpoints ->
                        endpoints.stream()
                                .filter(getEndpointFilter())
                                .findFirst(),
                configBuilder ->
                        configBuilder
                                .setApplicationName(LocalizedText.english("collector opc-ua client"))
                                .setApplicationUri("urn:org:eu:sbin:collector:client")
                                .setKeyPair(loader.getClientKeyPair())
                                .setCertificate(loader.getClientCertificate())
                                .setCertificateChain(loader.getClientCertificateChain())
                                .setCertificateValidator(certificateValidator)
                                .setIdentityProvider(getIdentityProvider())
                                .setRequestTimeout(uint(5000))
                                .build()
        );
    }

    @Override
    public void run() throws Exception {
        opcUaClient = createClient();
        this.run(opcUaClient, opcUaClientCompletableFuture);
        opcUaClientCompletableFuture.get(15, TimeUnit.SECONDS);
        opcUaClientCompletableFuture.whenCompleteAsync((c, ex) -> {
            if (Objects.nonNull(ex)) {
                deviceHolder.setRunStatus(RunStatus.EXCEPTION);
                deviceHolder.setRunStatusMsg(ex.getMessage());
            } else {
                deviceHolder.setRunStatus(RunStatus.ONLINE);
                deviceHolder.setRunStatusMsg(RunStatus.ONLINE_MSG);
            }
        });
    }

    @Override
    public void run(OpcUaClient client, CompletableFuture<OpcUaClient> future) throws Exception {
        client.connect().get();
        String namespace = deviceHolder.getIpParameter().getNameSpace();
        namespaceSpaceIndex = client.readNamespaceTable().getIndex(namespace);
        future.complete(client);
    }

    public static class Builder<D extends IDevice> extends CmdOpcUaCollectorBuilder<D> {

        @Override
        public CmdOpcUaCollector<D> build() {
            return new CmdOpcUaCollector<>(this);
        }

    }

}
