/*
 * Decompiled with CFR 0.152.
 */
package org.entur.netex.gtfs.export.producer;

import java.time.DayOfWeek;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.entur.netex.gtfs.export.exception.GtfsExportException;
import org.entur.netex.gtfs.export.model.GtfsService;
import org.entur.netex.gtfs.export.model.ServiceCalendarPeriod;
import org.entur.netex.gtfs.export.producer.GtfsServiceRepository;
import org.entur.netex.gtfs.export.repository.NetexDatasetRepository;
import org.rutebanken.netex.model.DayOfWeekEnumeration;
import org.rutebanken.netex.model.DayType;
import org.rutebanken.netex.model.DayTypeAssignment;
import org.rutebanken.netex.model.EntityStructure;
import org.rutebanken.netex.model.OperatingDay;
import org.rutebanken.netex.model.OperatingDayRefStructure;
import org.rutebanken.netex.model.OperatingPeriod;
import org.rutebanken.netex.model.PropertyOfDay;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultGtfsServiceRepository
implements GtfsServiceRepository {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultGtfsServiceRepository.class);
    private static final int MAX_SERVICE_ID_CHARS = 256;
    private static final Set<DayOfWeek> ALL_DAYS_OF_WEEKS = EnumSet.of(DayOfWeek.MONDAY, new DayOfWeek[]{DayOfWeek.TUESDAY, DayOfWeek.WEDNESDAY, DayOfWeek.THURSDAY, DayOfWeek.FRIDAY, DayOfWeek.SATURDAY, DayOfWeek.SUNDAY});
    private static final Set<DayOfWeek> WEEKDAYS = EnumSet.of(DayOfWeek.MONDAY, DayOfWeek.TUESDAY, DayOfWeek.WEDNESDAY, DayOfWeek.THURSDAY, DayOfWeek.FRIDAY);
    private static final Set<DayOfWeek> WEEKEND = EnumSet.of(DayOfWeek.SATURDAY, DayOfWeek.SUNDAY);
    private final String codespace;
    private final NetexDatasetRepository netexDatasetRepository;
    private final Map<String, GtfsService> gtfsServices;

    public DefaultGtfsServiceRepository(String codespace, NetexDatasetRepository netexDatasetRepository) {
        this.netexDatasetRepository = netexDatasetRepository;
        this.gtfsServices = new HashMap<String, GtfsService>();
        this.codespace = codespace;
    }

    @Override
    public Collection<GtfsService> getAllServices() {
        return this.gtfsServices.values();
    }

    @Override
    public GtfsService getServiceForDayTypes(Set<DayType> dayTypes) {
        String serviceId = this.getServiceIdForDayTypes(dayTypes);
        return this.gtfsServices.computeIfAbsent(serviceId, s -> this.createGtfsServiceForDayTypes(dayTypes, serviceId));
    }

    @Override
    public GtfsService getServiceForOperatingDays(Set<OperatingDay> operatingDays) {
        String serviceId = this.getServiceIdForOperatingDays(operatingDays);
        return this.gtfsServices.computeIfAbsent(serviceId, s -> DefaultGtfsServiceRepository.createGtfsServiceForOperatingDays(operatingDays, serviceId));
    }

    private String getServiceIdForDayTypes(Set<DayType> dayTypes) {
        Object serviceId = this.codespace + ":DayType:" + dayTypes.stream().map(EntityStructure::getId).map(DefaultGtfsServiceRepository::splitId).sorted().collect(Collectors.joining("-"));
        if (((String)serviceId).length() > 256) {
            serviceId = DefaultGtfsServiceRepository.truncateServiceId((String)serviceId);
        }
        return serviceId;
    }

    private String getServiceIdForOperatingDays(Set<OperatingDay> operatingDays) {
        Object serviceId = this.codespace + ":OperatingDay:" + operatingDays.stream().map(EntityStructure::getId).map(DefaultGtfsServiceRepository::splitId).sorted().collect(Collectors.joining("-"));
        if (((String)serviceId).length() > 256) {
            serviceId = DefaultGtfsServiceRepository.truncateServiceId((String)serviceId);
        }
        return serviceId;
    }

    private static String truncateServiceId(String serviceId) {
        String tooLongPart = serviceId.substring(246);
        serviceId = serviceId.replace(tooLongPart, StringUtils.truncate((String)("" + tooLongPart.hashCode()), (int)10));
        return serviceId;
    }

    private static String splitId(String id) {
        return id.split(":")[2];
    }

    private GtfsService createGtfsServiceForDayTypes(Set<DayType> dayTypes, String serviceId) {
        int nbPeriods = this.countPeriods(dayTypes);
        if (nbPeriods == 0) {
            return this.createGtfsServiceForIndividualDates(dayTypes, serviceId);
        }
        if (nbPeriods == 1) {
            return this.createGtfsServiceForOnePeriodAndIndividualDates(dayTypes, serviceId);
        }
        return this.createGtfsServiceForMultiplePeriodsAndIndividualDates(dayTypes, serviceId);
    }

    private static GtfsService createGtfsServiceForOperatingDays(Set<OperatingDay> operatingDays, String serviceId) {
        LOGGER.debug("Creating GTFS Service for operating days for serviceId {}", (Object)serviceId);
        GtfsService gtfsService = new GtfsService(serviceId);
        operatingDays.forEach(operatingDay -> gtfsService.addIncludedDate(operatingDay.getCalendarDate()));
        return gtfsService;
    }

    private GtfsService createGtfsServiceForIndividualDates(Set<DayType> dayTypes, String serviceId) {
        LOGGER.debug("Creating GTFS Service for individual dates for serviceId {}", (Object)serviceId);
        GtfsService gtfsService = new GtfsService(serviceId);
        dayTypes.stream().map(this.netexDatasetRepository::getDayTypeAssignmentsByDayType).flatMap(Collection::stream).forEach(dayTypeAssignment -> this.addIndividualDate(gtfsService, (DayTypeAssignment)dayTypeAssignment));
        gtfsService.removeIncludedDates(gtfsService.getExcludedDates());
        gtfsService.removeAllExcludedDates();
        return gtfsService;
    }

    private GtfsService createGtfsServiceForOnePeriodAndIndividualDates(Set<DayType> dayTypes, String serviceId) {
        LOGGER.debug("Creating GTFS Service for one period and individual dates for serviceId {}", (Object)serviceId);
        GtfsService gtfsService = new GtfsService(serviceId);
        DayTypeAssignment dayTypeAssignmentWithPeriod = dayTypes.stream().map(this.netexDatasetRepository::getDayTypeAssignmentsByDayType).flatMap(Collection::stream).filter(dta -> dta.getOperatingPeriodRef() != null).findFirst().orElseThrow(() -> new GtfsExportException("Could not find DayTypeAssignment without operating period for serviceId " + serviceId));
        DayType dayTypeWithAPeriod = this.netexDatasetRepository.getDayTypeByDayTypeAssignment(dayTypeAssignmentWithPeriod);
        OperatingPeriod operatingPeriod = this.netexDatasetRepository.getOperatingPeriodByDayTypeAssignment(dayTypeAssignmentWithPeriod);
        Set<DayOfWeek> daysOfWeek = DefaultGtfsServiceRepository.getDaysOfWeek(dayTypeWithAPeriod);
        ServiceCalendarPeriod serviceCalendarPeriod = new ServiceCalendarPeriod(this.getOperatingPeriodStartDate(operatingPeriod), this.getOperatingPeriodEndDate(operatingPeriod), daysOfWeek);
        gtfsService.setServiceCalendarPeriod(serviceCalendarPeriod);
        dayTypes.stream().map(this.netexDatasetRepository::getDayTypeAssignmentsByDayType).flatMap(Collection::stream).filter(dta -> dta.getOperatingPeriodRef() == null).forEach(dayTypeAssignment -> this.addIndividualDate(gtfsService, (DayTypeAssignment)dayTypeAssignment));
        gtfsService.removeIncludedDates(gtfsService.getExcludedDates());
        return gtfsService;
    }

    private GtfsService createGtfsServiceForMultiplePeriodsAndIndividualDates(Set<DayType> dayTypes, String serviceId) {
        LOGGER.debug("Creating GTFS Service for multiple periods and individual dates  for serviceId {}", (Object)serviceId);
        GtfsService gtfsService = new GtfsService(serviceId);
        for (DayType dayType : dayTypes) {
            Set<DayOfWeek> daysOfWeek = DefaultGtfsServiceRepository.getDaysOfWeek(dayType);
            for (DayTypeAssignment dayTypeAssignment2 : this.netexDatasetRepository.getDayTypeAssignmentsByDayType(dayType)) {
                if (dayTypeAssignment2.getOperatingPeriodRef() == null) continue;
                OperatingPeriod operatingPeriod = this.netexDatasetRepository.getOperatingPeriodByDayTypeAssignment(dayTypeAssignment2);
                LocalDateTime operatingPeriodStartDate = this.getOperatingPeriodStartDate(operatingPeriod);
                LocalDateTime operatingPeriodEndDate = this.getOperatingPeriodEndDate(operatingPeriod);
                LocalDateTime date = operatingPeriodStartDate;
                while (date.isBefore(operatingPeriodEndDate) || date.equals(operatingPeriodEndDate)) {
                    if (DefaultGtfsServiceRepository.isActiveDate(date, daysOfWeek)) {
                        gtfsService.addIncludedDate(date);
                    }
                    date = date.plusDays(1L);
                }
            }
        }
        dayTypes.stream().map(this.netexDatasetRepository::getDayTypeAssignmentsByDayType).flatMap(Collection::stream).filter(dta -> dta.getOperatingPeriodRef() == null).forEach(dayTypeAssignment -> this.addIndividualDate(gtfsService, (DayTypeAssignment)dayTypeAssignment));
        gtfsService.removeIncludedDates(gtfsService.getExcludedDates());
        gtfsService.removeAllExcludedDates();
        return gtfsService;
    }

    private void addIndividualDate(GtfsService gtfsService, DayTypeAssignment dayTypeAssignment) {
        LocalDateTime date;
        if (dayTypeAssignment.getOperatingDayRef() != null) {
            OperatingDay operatingDay = this.netexDatasetRepository.getOperatingDayByDayTypeAssignment(dayTypeAssignment);
            date = operatingDay.getCalendarDate();
        } else if (dayTypeAssignment.getDate() != null) {
            date = dayTypeAssignment.getDate();
        } else {
            throw new GtfsExportException("Both Date and OperatingDay are undefined on DayTypeAssignment " + dayTypeAssignment.getId());
        }
        if (dayTypeAssignment.isIsAvailable() != null && !dayTypeAssignment.isIsAvailable().booleanValue()) {
            gtfsService.addExcludedDate(date);
        } else {
            gtfsService.addIncludedDate(date);
        }
    }

    private static List<DayOfWeekEnumeration> getNetexDaysOfWeek(DayType dayType) {
        if (dayType.getProperties() != null && dayType.getProperties().getPropertyOfDay() != null) {
            for (PropertyOfDay propertyOfDay : dayType.getProperties().getPropertyOfDay()) {
                if (propertyOfDay.getDaysOfWeek() == null || propertyOfDay.getDaysOfWeek().isEmpty()) continue;
                return propertyOfDay.getDaysOfWeek();
            }
        }
        return Collections.emptyList();
    }

    private static Set<DayOfWeek> getDaysOfWeek(DayType dayType) {
        List<DayOfWeekEnumeration> netexDaysOfWeek = DefaultGtfsServiceRepository.getNetexDaysOfWeek(dayType);
        if (netexDaysOfWeek.isEmpty() || netexDaysOfWeek.contains(DayOfWeekEnumeration.EVERYDAY)) {
            return ALL_DAYS_OF_WEEKS;
        }
        return netexDaysOfWeek.stream().map(dayOfWeekEnumeration -> {
            if (DayOfWeekEnumeration.MONDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.MONDAY);
            }
            if (DayOfWeekEnumeration.TUESDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.TUESDAY);
            }
            if (DayOfWeekEnumeration.WEDNESDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.WEDNESDAY);
            }
            if (DayOfWeekEnumeration.THURSDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.THURSDAY);
            }
            if (DayOfWeekEnumeration.FRIDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.FRIDAY);
            }
            if (DayOfWeekEnumeration.SATURDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.SATURDAY);
            }
            if (DayOfWeekEnumeration.SUNDAY == dayOfWeekEnumeration) {
                return Set.of(DayOfWeek.SUNDAY);
            }
            if (DayOfWeekEnumeration.WEEKDAYS == dayOfWeekEnumeration) {
                return WEEKDAYS;
            }
            if (DayOfWeekEnumeration.WEEKEND == dayOfWeekEnumeration) {
                return WEEKEND;
            }
            throw new GtfsExportException("Unsupported day of week: " + String.valueOf(dayOfWeekEnumeration));
        }).flatMap(Collection::stream).collect(Collectors.toSet());
    }

    private static boolean isActiveDate(LocalDateTime date, Set<DayOfWeek> daysOfWeek) {
        return daysOfWeek.contains(date.getDayOfWeek());
    }

    private int countPeriods(Set<DayType> dayTypes) {
        return dayTypes.stream().map(dayType -> this.netexDatasetRepository.getDayTypeAssignmentsByDayType((DayType)dayType).stream().filter(dayTypeAssignment -> dayTypeAssignment.getOperatingPeriodRef() != null).count()).mapToInt(Long::intValue).sum();
    }

    private LocalDateTime getOperatingPeriodStartDate(OperatingPeriod operatingPeriod) {
        if (operatingPeriod.getFromDate() != null) {
            return operatingPeriod.getFromDate();
        }
        if (operatingPeriod.getFromOperatingDayRef() != null) {
            return this.lookupOperatingDay(operatingPeriod.getFromOperatingDayRef());
        }
        throw new IllegalArgumentException("Missing start date for operating period " + operatingPeriod.getId());
    }

    private LocalDateTime getOperatingPeriodEndDate(OperatingPeriod operatingPeriod) {
        if (operatingPeriod.getToDate() != null) {
            return operatingPeriod.getToDate();
        }
        if (operatingPeriod.getToOperatingDayRef() != null) {
            return this.lookupOperatingDay(operatingPeriod.getToOperatingDayRef());
        }
        throw new IllegalArgumentException("Missing end date for operating period " + operatingPeriod.getId());
    }

    private LocalDateTime lookupOperatingDay(OperatingDayRefStructure operatingDayRef) {
        return this.netexDatasetRepository.getOperatingDayById(operatingDayRef.getRef()).getCalendarDate();
    }
}

