/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.sql.Timestamp;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.OMFContext;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreField;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.types.ObjectStringConverter;
import org.joda.time.Duration;
import org.joda.time.Interval;

/**
 * SCO Mapping for org.joda.time.Interval type.
 * Can be persisted using either
 * <ul>
 * <li>Single column using a String mapping.</li>
 * <li>Two columns using Timestamp mappings for the start/end components</li>
 * </ul>
 * See http://joda-time.sourceforge.net/
 */
public class JodaIntervalMapping extends SingleFieldMultiMapping
{
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize(AbstractMemberMetaData, DatastoreContainerObject, ClassLoaderResolver)
     */
    public void initialize(AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
        super.initialize(fmd, container, clr);
        addDatastoreFields();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapped.mapping.JavaTypeMapping#initialize(MappedStoreManager, java.lang.String)
     */
    public void initialize(MappedStoreManager storeMgr, String type)
    {
        super.initialize(storeMgr, type);
        addDatastoreFields();
    }

    protected void addDatastoreFields()
    {
        ColumnMetaData[] colmds = getColumnMetaDataForMember(mmd, roleForMember);
        if (colmds != null && colmds.length == 1)
        {
            // Stored as single column, so use String based
            MappingManager mmgr = datastoreContainer.getStoreManager().getMappingManager();
            DatastoreField col = mmgr.createDatastoreField(this, ClassNameConstants.JAVA_LANG_STRING, 0);
            mmgr.createDatastoreMapping(this, mmd, 0, col);
        }
        else
        {
            // Store as 2 TIMESTAMP columns
            addDatastoreField(ClassNameConstants.JAVA_SQL_TIMESTAMP); // start
            addDatastoreField(ClassNameConstants.JAVA_SQL_TIMESTAMP); // end
        }
    }

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        if (datastoreMappings != null && datastoreMappings.length == 1 && datastoreMappings[0].isStringBased())
        {
            // STRING
            return ClassNameConstants.JAVA_LANG_STRING;
        }
        else
        {
            // TIMESTAMP
            return ClassNameConstants.JAVA_SQL_TIMESTAMP;
        }
    }

    public Class getJavaType()
    {
        return Duration.class;
    }

    /**
     * Method to return the value to be stored in the specified datastore index given the overall
     * value for this java type.
     * @param index The datastore index
     * @param value The overall value for this java type
     * @return The value for this datastore index
     */
    public Object getValueForDatastoreMapping(OMFContext omfCtx, int index, Object value)
    {
        Interval intvl = (Interval)value;
        if (getNumberOfDatastoreMappings() == 1)
        {
            return super.getValueForDatastoreMapping(omfCtx, index, value);
        }
        else if (index == 0)
        {
            return intvl.getStartMillis();
        }
        else if (index == 1)
        {
            return intvl.getEndMillis();
        }
        throw new IndexOutOfBoundsException();
    }

    public Object getObject(ExecutionContext ec, Object rs, int[] exprIndex)
    {
        if (getDatastoreMapping(0).getObject(rs, exprIndex[0]) == null)
        {
            return null;
        }

        if (datastoreMappings != null && datastoreMappings.length == 1 && datastoreMappings[0].isStringBased())
        {
            // String column
            Object datastoreValue = getDatastoreMapping(0).getObject(rs, exprIndex[0]);
            ObjectStringConverter conv = ec.getOMFContext().getTypeManager().getStringConverter(Interval.class);
            if (conv != null)
            {
                return conv.toObject((String)datastoreValue);
            }
            else
            {
                throw new NucleusUserException("This type doesn't support persistence as a String");
            }
        }
        else
        {
            // Timestamp columns
            Timestamp start = (Timestamp) getDatastoreMapping(0).getObject(rs, exprIndex[0]);
            Timestamp end = (Timestamp) getDatastoreMapping(1).getObject(rs, exprIndex[1]);
            return new Interval(start.getTime(), end.getTime());
        }
    }

    public void setObject(ExecutionContext ec, Object ps, int[] exprIndex, Object value)
    {
        Interval interval = (Interval) value;

        if (value == null)
        {
            getDatastoreMapping(0).setObject(ps, exprIndex[0], null);
        }
        else if (datastoreMappings != null && datastoreMappings.length == 1 && datastoreMappings[0].isStringBased())
        {
            // String column
            ObjectStringConverter conv = ec.getOMFContext().getTypeManager().getStringConverter(Interval.class);
            if (conv != null)
            {
                Object obj = conv.toString(value);
                getDatastoreMapping(0).setObject(ps, exprIndex[0], obj);
            }
            else
            {
                throw new NucleusUserException("This type doesn't support persistence as a String");
            }
        }
        else
        {
            // Timestamp columns
            if (interval == null)
            {
                getDatastoreMapping(0).setObject(ps, exprIndex[0], null);
                getDatastoreMapping(1).setObject(ps, exprIndex[1], null);
            }
            else
            {
                getDatastoreMapping(0).setObject(ps, exprIndex[0], new Timestamp(interval.getStartMillis()));
                getDatastoreMapping(1).setObject(ps, exprIndex[1], new Timestamp(interval.getEndMillis()));
            }
        }
    }
}