package org.daijie.jdbc.mybatis.transaction;

import static org.springframework.util.Assert.notNull;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.ibatis.logging.Log;
import org.apache.ibatis.logging.LogFactory;
import org.apache.ibatis.transaction.Transaction;
import org.daijie.jdbc.DbContextHolder;
import org.daijie.jdbc.interceptor.DefaultRoutingDataSource;
import org.springframework.jdbc.datasource.ConnectionHolder;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.transaction.support.TransactionSynchronizationManager;

/**
 * 每次访问mapper接口获取数据源连接
 * 动态切换数据源
 * @author daijie_jay
 * @since 2018年5月28日
 */
public class MybatisMultipleTransaction implements Transaction {

	private static final Log LOGGER = LogFactory.getLog(MybatisMultipleTransaction.class);

	private final DefaultRoutingDataSource dataSource;

	private Connection connection;

	private boolean isConnectionTransactional;

	private boolean autoCommit;

	public MybatisMultipleTransaction(DataSource dataSource) {
		notNull(dataSource, "没有初始化数据源");
		this.dataSource = (DefaultRoutingDataSource) dataSource;
	}

	@Override
	public Connection getConnection() throws SQLException {
		openConnection();
		return this.connection;
	}

	@Override
	public void commit() throws SQLException {
		if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug("提交事务 [" + this.connection + "]");
			}
			this.connection.commit();
		}
	}

	@Override
	public void rollback() throws SQLException {
		if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
			if (LOGGER.isDebugEnabled()) {
				LOGGER.debug("回滚事务 [" + this.connection + "]");
			}
			this.connection.rollback();
		}
	}

	@Override
	public void close() throws SQLException {
		DataSourceUtils.releaseConnection(this.connection, this.dataSource);
	}

	@Override
	public Integer getTimeout() throws SQLException {
		ConnectionHolder holder = (ConnectionHolder) TransactionSynchronizationManager.getResource(dataSource);
		if (holder != null && holder.hasTimeout()) {
			return holder.getTimeToLiveInSeconds();
		} 
		return null;
	}

	private void openConnection() throws SQLException {
		DataSource dataSource = (DataSource) this.dataSource.getTargetDataSources().get(DbContextHolder.getDataSourceName());
		if(dataSource == null){
			dataSource = this.dataSource;
		}
		this.connection = DataSourceUtils.getConnection(dataSource);
		this.isConnectionTransactional = DataSourceUtils.isConnectionTransactional(this.connection, dataSource);
		this.connection.setAutoCommit(!this.isConnectionTransactional);
		this.autoCommit = this.connection.getAutoCommit();
	}
}
