/*******************************************************************************
 * Copyright (c) 2016 Sebastian Stenzel and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the accompanying LICENSE.txt.
 *
 * Contributors:
 *     Sebastian Stenzel - initial API and implementation
 *******************************************************************************/
package org.cryptomator.cryptolib.common;

import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.SecureRandomSpi;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Wraps a fast CSPRNG, which gets reseeded automatically after a certain amount of bytes has been generated.<br>
 * 
 * <p>
 * Java 8 Example:
 * 
 * <pre>
 *  SecureRandom csprng = ReseedingSecureRandom.create(SecureRandom.getInstanceStrong());
 * </pre>
 */
public class ReseedingSecureRandom extends SecureRandom {

	private static final Logger LOG = LoggerFactory.getLogger(ReseedingSecureRandom.class);
	private static final Provider PROVIDER = new ReseedingSecureRandomProvider();

	/**
	 * @param seeder RNG for high-quality random numbers. E.g. <code>SecureRandom.getInstanceStrong()</code> in Java 8+ environments.
	 * @param csprng A fast csprng implementation, such as <code>SHA1PRNG</code>, that will be wrapped by this instance.
	 * @param reseedAfter How many bytes can be read from the <code>csprng</code>, before a new seed will be generated.
	 * @param seedLength Number of bytes generated by <code>seeder</code> in order to seed <code>csprng</code>.
	 */
	public ReseedingSecureRandom(SecureRandom seeder, SecureRandom csprng, long reseedAfter, int seedLength) {
		super(new ReseedingSecureRandomSpi(seeder, csprng, reseedAfter, seedLength), PROVIDER);
	}

	/**
	 * Creates a pre-configured automatically reseeding SHA1PRNG instance, reseeding itself with 440 bits from the given seeder after generating 2^30 bytes,
	 * thus satisfying recommendations by <a href="http://dx.doi.org/10.6028/NIST.SP.800-90Ar1">NIST SP 800-90A Rev 1</a>.
	 *
	 * @param seeder RNG for high-quality random numbers. E.g. <code>SecureRandom.getInstanceStrong()</code> in Java 8+ environments.
	 * @return An automatically reseeding SHA1PRNG suitable as CSPRNG for most applications.
	 */
	public static ReseedingSecureRandom create(SecureRandom seeder) {
		try {
			return new ReseedingSecureRandom(seeder, SecureRandom.getInstance("SHA1PRNG"), 1 << 30, 55);
		} catch (NoSuchAlgorithmException e) {
			throw new IllegalStateException("SHA1PRNG must exist in every Java platform implementation.", e);
		}
	}

	private static class ReseedingSecureRandomProvider extends Provider {

		protected ReseedingSecureRandomProvider() {
			super("ReseedingSecureRandomProvider", 1.0, "Provides ReseedingSecureRandom");
		}

	}

	private static class ReseedingSecureRandomSpi extends SecureRandomSpi {

		private final SecureRandom seeder;
		private final SecureRandom csprng;
		private final long reseedAfter;
		private final int seedLength;
		private long counter;

		public ReseedingSecureRandomSpi(SecureRandom seeder, SecureRandom csprng, long reseedAfter, int seedLength) {
			this.seeder = seeder;
			this.csprng = csprng;
			this.reseedAfter = reseedAfter;
			this.seedLength = seedLength;
			this.counter = reseedAfter; // trigger reseed during first "engineNextBytes(...)"
		}

		@Override
		protected void engineSetSeed(byte[] seed) {
			csprng.setSeed(seed);
		}

		@Override
		protected void engineNextBytes(byte[] bytes) {
			if (counter + bytes.length > reseedAfter) {
				reseed();
			}
			counter += bytes.length;
			csprng.nextBytes(bytes);
		}

		@Override
		protected byte[] engineGenerateSeed(int numBytes) {
			try {
				LOG.debug("Seeding CSPRNG with {} bytes...", numBytes);
				return seeder.generateSeed(numBytes);
			} finally {
				LOG.debug("Seeded CSPRNG.");
			}
		}

		private void reseed() {
			engineSetSeed(engineGenerateSeed(seedLength));
			counter = 0;
		}

	}

}
