/*
 * Copyright 2016 Daniel Vimont.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.commonvox.hbase_column_manager;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Date;
import java.util.Set;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVPrinter;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.util.Bytes;

/**
 *
 * @author Daniel Vimont
 */
class ColumnQualifierReport {

  static final CSVFormat CSV_FORMAT = CSVFormat.DEFAULT.withRecordSeparator("\n")
          .withCommentMarker('#').withHeader(ReportHeader.class);
  private final Repository repository;
  private final String sourceNamespace;
  private final TableName sourceTableName;
  private final byte[] sourceColFamily;
  private final File targetFile;
  private boolean recordsWrittenToReport = false;
  enum ReportType {QUALIFIER, LENGTH, VALUE};
  private String namespaceReportString;
  private String tableReportString;
  private String colFamilyReportString;
  private String colQualifierReportString;
  private String colMaxLengthReportString;


  ColumnQualifierReport(String namespace, TableName tableName, byte[] colFamily,
          Repository repository, File targetFile)
          throws IOException {
    this.sourceNamespace = namespace;
    this.sourceTableName = tableName;
    this.sourceColFamily = colFamily;
    this.repository = repository;
    this.targetFile = targetFile;
    outputReport();
  }

  enum ReportHeader {NAMESPACE, TABLE, COLUMN_FAMILY, COLUMN_QUALIFIER, COLUMN_MAX_LENGTH}

  private void outputReport() throws IOException {
    HBaseSchemaArchive hsa = new HBaseSchemaArchive(sourceNamespace, sourceTableName, repository);

    StringBuilder headerCommentsBuilder = new StringBuilder("Report on ColumnAuditors");
    if (sourceNamespace == null && sourceTableName == null) {
      headerCommentsBuilder.append(" in ALL ColumnManager-included Tables in ALL Namespaces");
    } else {
      if (sourceNamespace != null) {
        headerCommentsBuilder.append(" in Namespace <").append(sourceNamespace).append(">");
      }
      if (sourceTableName != null) {
         headerCommentsBuilder.append(" in Table <")
                 .append(sourceTableName.getNameAsString()).append(">");
      }
      if (sourceColFamily != null) {
        headerCommentsBuilder.append(" in Column Family <")
                .append(Bytes.toString(sourceColFamily)).append(">");
      }
      headerCommentsBuilder.append(" -- Generated by ").append(Repository.PRODUCT_NAME).append(":")
                            .append(this.getClass().getSimpleName());
    }
    try (CSVPrinter csvPrinter = CSV_FORMAT.withHeaderComments(headerCommentsBuilder, new Date())
            .print(new FileWriter(targetFile))) {

      namespaceReportString = "";
      tableReportString = "";
      colFamilyReportString = "";
      colQualifierReportString = "";
      colMaxLengthReportString = "";

      printReport(csvPrinter, hsa.getSchemaEntities());
    }
  }

  void printReport(CSVPrinter csvPrinter, Set<SchemaEntity> schemaEntities) throws IOException {
    for (SchemaEntity entity : schemaEntities) {
      switch (entity.getSchemaEntityType()) {
        case NAMESPACE:
          namespaceReportString = entity.getNameAsString();
          tableReportString = "";
          colFamilyReportString = "";
          colQualifierReportString = "";
          colMaxLengthReportString = "";
          printReport(csvPrinter, entity.getChildren());
          break;
        case TABLE:
          tableReportString = entity.getNameAsString();
          printReport(csvPrinter, entity.getChildren());
          break;
        case COLUMN_FAMILY:
          colFamilyReportString = entity.getNameAsString();
          if (this.sourceColFamily != null
                  && !colFamilyReportString.equals(Bytes.toString(sourceColFamily))) {
            continue;
          }
          printReport(csvPrinter, entity.getChildren());
          break;
        case COLUMN_AUDITOR:
          colQualifierReportString = entity.getNameAsString();
          colMaxLengthReportString = entity.getValue(ColumnAuditor.MAX_VALUE_LENGTH_KEY);
          csvPrinter.print(namespaceReportString);
          csvPrinter.print(tableReportString);
          csvPrinter.print(colFamilyReportString);
          csvPrinter.print(colQualifierReportString);
          csvPrinter.print(colMaxLengthReportString);
          csvPrinter.println();
          recordsWrittenToReport = true;
      }
    }
  }

  boolean isEmpty() {
    return !recordsWrittenToReport;
  }
}
