/*
 * Decompiled with CFR 0.152.
 */
package org.commonjava.storage.pathmapped.core;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang.StringUtils;
import org.commonjava.cdi.util.weft.NamedThreadFactory;
import org.commonjava.storage.pathmapped.config.PathMappedStorageConfig;
import org.commonjava.storage.pathmapped.core.PathDBOutputStream;
import org.commonjava.storage.pathmapped.spi.FileInfo;
import org.commonjava.storage.pathmapped.spi.PathDB;
import org.commonjava.storage.pathmapped.spi.PhysicalStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PathMappedFileManager
implements Closeable {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final PathDB pathDB;
    private final PhysicalStore physicalStore;
    private final PathMappedStorageConfig config;
    private ScheduledExecutorService gcThreadPool;
    private String deduplicatePattern;

    public PathMappedFileManager(PathMappedStorageConfig config, PathDB pathDB, PhysicalStore physicalStore) {
        this.pathDB = pathDB;
        this.physicalStore = physicalStore;
        this.config = config;
        int gcIntervalInMinutes = config.getGCIntervalInMinutes();
        if (gcIntervalInMinutes > 0) {
            this.logger.info("Start path-mapped GC thread, gcIntervalInMinutes: {}", (Object)gcIntervalInMinutes);
            int priority = 4;
            String name = "path-mapped-gc";
            this.gcThreadPool = Executors.newScheduledThreadPool(1, (ThreadFactory)new NamedThreadFactory(name, new ThreadGroup(name), true, priority));
            int initialDelay = gcIntervalInMinutes;
            this.gcThreadPool.scheduleAtFixedRate(() -> this.gc(), initialDelay, gcIntervalInMinutes, TimeUnit.MINUTES);
        }
        this.deduplicatePattern = config.getDeduplicatePattern();
    }

    public Set<String> getFileSystemContainingDirectory(Collection<String> candidates, String path) {
        if (!path.endsWith("/")) {
            path = path + "/";
        }
        return this.pathDB.getFileSystemContaining(candidates, path);
    }

    public Set<String> getFileSystemContaining(Collection<String> candidates, String path) {
        return this.pathDB.getFileSystemContaining(candidates, path);
    }

    public String getFirstFileSystemContaining(List<String> candidates, String path) {
        return this.pathDB.getFirstFileSystemContaining(candidates, path);
    }

    public InputStream openInputStream(String fileSystem, String path) throws IOException {
        String storageFile = this.pathDB.getStorageFile(fileSystem, path);
        if (storageFile == null) {
            throw new IOException(String.format("Could not open input stream to for path %s - %s: path-mapped file does not exist.", fileSystem, path));
        }
        InputStream stream = this.physicalStore.getInputStream(storageFile);
        if (stream == null) {
            throw new IOException(String.format("Could not open input stream to for path %s - %s: path-mapped physical file does not exist.", fileSystem, path));
        }
        return stream;
    }

    public OutputStream openOutputStream(String fileSystem, String path) throws IOException {
        return this.openOutputStream(fileSystem, path, 0L, TimeUnit.SECONDS);
    }

    public OutputStream openOutputStream(String fileSystem, String path, long timeout, TimeUnit timeoutUnit) throws IOException {
        FileInfo fileInfo = this.physicalStore.getFileInfo(fileSystem, path);
        String checksumAlgorithm = null;
        if (this.deduplicatePattern != null && fileSystem.matches(this.deduplicatePattern)) {
            checksumAlgorithm = this.config.getFileChecksumAlgorithm();
        }
        try {
            return new PathDBOutputStream(this.pathDB, this.physicalStore, fileSystem, path, fileInfo, this.physicalStore.getOutputStream(fileInfo), checksumAlgorithm, timeoutUnit.toMillis(timeout));
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException("Error: checksum checking not correct", e);
        }
    }

    public boolean delete(String fileSystem, String path) {
        return this.pathDB.delete(fileSystem, path);
    }

    public void cleanupCurrentThread() {
    }

    public void startReporting() {
    }

    public void stopReporting() {
    }

    public String[] list(String fileSystem, String path) {
        return this.list(fileSystem, path, PathDB.FileType.all);
    }

    public String[] list(String fileSystem, String path, boolean recursive, int limit) {
        return this.list(fileSystem, path, recursive, limit, PathDB.FileType.all);
    }

    public String[] list(String fileSystem, String path, PathDB.FileType fileType) {
        return this.list(fileSystem, path, false, 0, fileType);
    }

    public String[] list(String fileSystem, String path, boolean recursive, int limit, PathDB.FileType fileType) {
        if (path == null) {
            return new String[0];
        }
        if (recursive) {
            List paths = this.pathDB.list(fileSystem, path, true, limit, fileType);
            return (String[])paths.stream().map(x -> {
                String cutParentPath = this.cutParentPath(path, x.getParentPath());
                if (StringUtils.isBlank((String)cutParentPath)) {
                    return x.getFilename();
                }
                return cutParentPath + "/" + x.getFilename();
            }).toArray(String[]::new);
        }
        List paths = this.pathDB.list(fileSystem, path, fileType);
        return (String[])paths.stream().map(x -> x.getFilename()).toArray(String[]::new);
    }

    private String cutParentPath(String rootPath, String parentPath) {
        String ret;
        if (!rootPath.startsWith("/")) {
            rootPath = "/" + rootPath;
        }
        if ((ret = parentPath.replaceFirst(rootPath, "")).startsWith("/")) {
            ret = ret.replaceFirst("/", "");
        }
        return ret;
    }

    public long getFileLength(String fileSystem, String path) {
        if (path == null) {
            return 0L;
        }
        return this.pathDB.getFileLength(fileSystem, path);
    }

    public long getFileLastModified(String fileSystem, String path) {
        if (path == null) {
            return -1L;
        }
        return this.pathDB.getFileLastModified(fileSystem, path);
    }

    public boolean exists(String fileSystem, String path) {
        if (StringUtils.isBlank((String)path)) {
            return false;
        }
        PathDB.FileType exist = this.pathDB.exists(fileSystem, path);
        if (exist != null) {
            if (exist == PathDB.FileType.dir) {
                return true;
            }
            String storageFile = this.pathDB.getStorageFile(fileSystem, path);
            if (storageFile != null) {
                if (this.physicalStore.exists(storageFile)) {
                    return true;
                }
                this.logger.warn("File in pathDB but physical file missing!, fileSystem: {}, path: {}, storageFile: {}", new Object[]{fileSystem, path, storageFile});
                return false;
            }
        }
        return false;
    }

    public boolean isDirectory(String fileSystem, String path) {
        if (path == null) {
            return false;
        }
        if ("/".equals(path)) {
            return true;
        }
        return this.pathDB.isDirectory(fileSystem, path);
    }

    public boolean isFile(String fileSystem, String path) {
        if (path == null) {
            return false;
        }
        return this.pathDB.isFile(fileSystem, path);
    }

    public void copy(String fromFileSystem, String fromPath, String toFileSystem, String toPath) {
        this.pathDB.copy(fromFileSystem, fromPath, toFileSystem, toPath);
    }

    public void makeDirs(String fileSystem, String path) {
        this.pathDB.makeDirs(fileSystem, path);
    }

    public String getFileStoragePath(String fileSystem, String path) {
        return this.pathDB.getStorageFile(fileSystem, path);
    }

    public Map<FileInfo, Boolean> gc() {
        HashMap<FileInfo, Boolean> gcResults = new HashMap<FileInfo, Boolean>();
        while (true) {
            int batchSize = this.config.getGCBatchSize();
            List reclaims = this.pathDB.listOrphanedFiles(batchSize);
            int size = reclaims.size();
            this.logger.debug("Get reclaims for GC, size: {}", (Object)size);
            if (size <= 0) break;
            if (batchSize > 0 && size < batchSize) {
                this.logger.debug("Get reclaims but less than batch size {}. Break.", (Object)batchSize);
                break;
            }
            reclaims.forEach(reclaim -> {
                FileInfo fileInfo = new FileInfo();
                fileInfo.setFileId(reclaim.getFileId());
                fileInfo.setFileStorage(reclaim.getStorage());
                boolean result = this.physicalStore.delete(fileInfo);
                if (result) {
                    this.logger.debug("Delete from physicalStore, fileInfo: {}", (Object)fileInfo);
                    this.pathDB.removeFromReclaim(reclaim);
                }
                gcResults.put(fileInfo, result);
            });
        }
        return gcResults;
    }

    @Override
    public void close() throws IOException {
        if (this.pathDB instanceof Closeable) {
            ((Closeable)this.pathDB).close();
        }
    }
}

