/*
 * Decompiled with CFR 0.152.
 */
package org.codelibs.saml2.core.test.settings;

import java.net.URL;
import java.util.Map;
import org.codelibs.saml2.core.model.Contact;
import org.codelibs.saml2.core.settings.IdPMetadataParser;
import org.codelibs.saml2.core.settings.Saml2Settings;
import org.codelibs.saml2.core.settings.SettingsBuilder;
import org.codelibs.saml2.core.util.Util;
import org.junit.Assert;
import org.junit.Test;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

public class IdPMetadataParserTest {
    @Test
    public void testParseFileXML() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/onelogin_metadata.xml");
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/645460", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://example.onelogin.com/trust/saml2/http-redirect/sso/645460", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://example.onelogin.com/trust/saml2/http-redirect/slo/645460", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp_metadata.xml");
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/383123", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://app.onelogin.com/trust/saml2/http-post/sso/383123", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEHjCCAwagAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbTAeFw0xMzA2MDUxNzE2MjBaFw0xODA2MDUxNzE2MjBaMGcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRUwEwYDVQQHDAxTYW50YSBNb25pY2ExETAPBgNVBAoMCE9uZUxvZ2luMRkwFwYDVQQDDBBhcHAub25lbG9naW4uY29tMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAse8rnep4qL2GmhH10pMQyJ2Jae+AQHyfgVjaQZ7Z0QQog5jX91vcJRSMi0XWJnUtOr6lF0dq1+yckjZ92wyLrH+7fvngNO1aV4Mjk9sTgf+iqMrae6y6fRxDt9PXrEFVjvd3vv7QTJf2FuIPy4vVP06Dt8EMkQIr8rmLmU0mTr1k2DkrdtdlCuNFTXuAu3QqfvNCRrRwfNObn9MP6JeOUdcGLJsBjGF8exfcN1SFzRF0JFr3dmOlx761zK5liD0T1sYWnDquatj/JD9fZMbKecBKni1NglH/LVd+b6aJUAr5LulERULUjLqYJRKW31u91/4Qazdo9tbvwqyFxaoUrwIDAQABo4HUMIHRMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFPWcXvQSlTXnzZD2xziuoUvrrDedMIGRBgNVHSMEgYkwgYaAFPWcXvQSlTXnzZD2xziuoUvrrDedoWukaTBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbYIBATAOBgNVHQ8BAf8EBAMCBPAwDQYJKoZIhvcNAQEFBQADggEBAB/8xe3rzqXQVxzHyAHuAuPa73ClDoL1cko0Fp8CGcqEIyj6Te9gx5z6wyfv+Lo8RFvBLlnB1lXqbC+fTGcVgG/4oKLJ5UwRFxInqpZPnOAudVNnd0PYOODn9FWs6u+OTIQIaIcPUv3MhB9lwHIJsTk/bs9xcru5TPyLIxLLd6ib/pRceKH2mTkzUd0DYk9CQNXXeoGx/du5B9nh3ClPTbVakRzl3oswgI5MQIphYxkW70SopEh4kOFSRE1ND31NNIq1YrXlgtkguQBFsZWuQOPR6cEwFZzP0tHTYbI839WgxX6hfhIUTUz6mLqq4+3P4BG3+1OXeVDg63y8Uh781sE="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        boolean throwedInvalidXPath = false;
        try {
            IdPMetadataParser.parseFileXML((String)"data/metadata/idp_metadata.xml", null, null, (String)"\"", (String)"\"");
        }
        catch (Exception e) {
            throwedInvalidXPath = true;
        }
        Assert.assertTrue((boolean)throwedInvalidXPath);
        boolean throwedFileNotFound = false;
        try {
            IdPMetadataParser.parseFileXML((String)"nonexistent.file", null, null, (String)"\"", (String)"\"");
        }
        catch (Exception e) {
            throwedFileNotFound = true;
        }
        Assert.assertTrue((boolean)throwedFileNotFound);
    }

    @Test
    public void testParseXML() throws Exception {
        Document xmlDocument = Util.parseXML((InputSource)new InputSource(this.getClass().getClassLoader().getResourceAsStream("data/metadata/onelogin_metadata.xml")));
        Map idpInfo = IdPMetadataParser.parseXML((Document)xmlDocument);
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/645460", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://example.onelogin.com/trust/saml2/http-redirect/sso/645460", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://example.onelogin.com/trust/saml2/http-redirect/slo/645460", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseFileXmlMultix509cert() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/metadata.xml");
        Assert.assertEquals((Object)"https://idp.examle.com/saml/metadata", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/sso", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/slo", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509certMulti.0"))), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xNzA0MTUxNjMzMThaFw0xODA0MTUxNjMzMThaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC6GLkl5lDUZdHNDAojp5i24OoPlqrt5TGXJIPqAZYT1hQvJW5nv17MFDHrjmtEnmW4ACKEy0fAX80QWIcHunZSkbEGHb+NG/6oTi5RipXMvmHnfFnPJJ0AdtiLiPE478CV856gXekV4Xx5u3KrylcOgkpYsp0GMIQBDzleMUXlYQIDAQABo1AwTjAdBgNVHQ4EFgQUnP8vlYPGPL2n6ZzDYij2kMDC8wMwHwYDVR0jBBgwFoAUnP8vlYPGPL2n6ZzDYij2kMDC8wMwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQAlQGAl+b8Cpot1g+65lLLjVoY7APJPWLW0klKQNlMU0s4MU+71Y3ExUEOXDAZgKcFoavb1fEOGMwEf38NaJAy1e/l6VNuixXShffq20ymqHQxOG0q8ujeNkgZF9k6XDfn/QZ3AD0o/IrCT7UMc/0QsfgIjWYxwCvp2syApc5CYfQ=="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseFileXmlDesiredBindings() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/testshib-providers.xml");
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/shibboleth", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/profile/SAML2/Redirect/SSO", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEDjCCAvagAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzEVMBMGA1UECBMMUGVubnN5bHZhbmlhMRMwEQYDVQQHEwpQaXR0c2J1cmdoMREwDwYDVQQKEwhUZXN0U2hpYjEZMBcGA1UEAxMQaWRwLnRlc3RzaGliLm9yZzAeFw0wNjA4MzAyMTEyMjVaFw0xNjA4MjcyMTEyMjVaMGcxCzAJBgNVBAYTAlVTMRUwEwYDVQQIEwxQZW5uc3lsdmFuaWExEzARBgNVBAcTClBpdHRzYnVyZ2gxETAPBgNVBAoTCFRlc3RTaGliMRkwFwYDVQQDExBpZHAudGVzdHNoaWIub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArYkCGuTmJp9eAOSGHwRJo1SNatB5ZOKqDM9ysg7CyVTDClcpu93gSP10nH4gkCZOlnESNgttg0r+MqL8tfJC6ybddEFB3YBo8PZajKSe3OQ01Ow3yT4I+Wdg1tsTpSge9gEz7SrC07EkYmHuPtd71CHiUaCWDv+xVfUQX0aTNPFmDixzUjoYzbGDrtAyCqA8f9CN2txIfJnpHE6q6CmKcoLADS4UrNPlhHSzd614kR/JYiks0K4kbRqCQF0Dv0P5Di+rEfefC6glV8ysC8dB5/9nb0yh/ojRuJGmgMWHgWk6h0ihjihqiu4jACovUZ7vVOCgSE5Ipn7OIwqd93zp2wIDAQABo4HEMIHBMB0GA1UdDgQWBBSsBQ869nh83KqZr5jArr4/7b+QazCBkQYDVR0jBIGJMIGGgBSsBQ869nh83KqZr5jArr4/7b+Qa6FrpGkwZzELMAkGA1UEBhMCVVMxFTATBgNVBAgTDFBlbm5zeWx2YW5pYTETMBEGA1UEBxMKUGl0dHNidXJnaDERMA8GA1UEChMIVGVzdFNoaWIxGTAXBgNVBAMTEGlkcC50ZXN0c2hpYi5vcmeCAQAwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQUFAAOCAQEAjR29PhrCbk8qLN5MFfSVk98t3CT9jHZoYxd8QMRLI4j7iYQxXiGJTT1FXs1nd4Rha9un+LqTfeMMYqISdDDI6tv8iNpkOAvZZUosVkUo93pv1T0RPz35hcHHYq2yee59HJOco2bFlcsH8JBXRSRrJ3Q7Eut+z9uo80JdGNJ4/SJy5UorZ8KazGj16lfJhOBXldgrhppQBb0Nq6HKHguqmwRfJ+WkxemZXzhediAjGeka8nz8JjwxpUjAiSWYKLtJhGEaTqCYxCCX2Dw+dOTqUzHOZ7WKv4JXPK5G/Uhr8K/qhmFT2nIQi538n6rVYLeWj8Bbnl+ev0peYzxFyF5sQA=="));
        Assert.assertEquals((Object)"urn:mace:shibboleth:1.0:nameIdentifier", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/testshib-providers.xml", (String)"https://idp.testshib.org/idp/shibboleth");
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/shibboleth", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/profile/SAML2/Redirect/SSO", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEDjCCAvagAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzEVMBMGA1UECBMMUGVubnN5bHZhbmlhMRMwEQYDVQQHEwpQaXR0c2J1cmdoMREwDwYDVQQKEwhUZXN0U2hpYjEZMBcGA1UEAxMQaWRwLnRlc3RzaGliLm9yZzAeFw0wNjA4MzAyMTEyMjVaFw0xNjA4MjcyMTEyMjVaMGcxCzAJBgNVBAYTAlVTMRUwEwYDVQQIEwxQZW5uc3lsdmFuaWExEzARBgNVBAcTClBpdHRzYnVyZ2gxETAPBgNVBAoTCFRlc3RTaGliMRkwFwYDVQQDExBpZHAudGVzdHNoaWIub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArYkCGuTmJp9eAOSGHwRJo1SNatB5ZOKqDM9ysg7CyVTDClcpu93gSP10nH4gkCZOlnESNgttg0r+MqL8tfJC6ybddEFB3YBo8PZajKSe3OQ01Ow3yT4I+Wdg1tsTpSge9gEz7SrC07EkYmHuPtd71CHiUaCWDv+xVfUQX0aTNPFmDixzUjoYzbGDrtAyCqA8f9CN2txIfJnpHE6q6CmKcoLADS4UrNPlhHSzd614kR/JYiks0K4kbRqCQF0Dv0P5Di+rEfefC6glV8ysC8dB5/9nb0yh/ojRuJGmgMWHgWk6h0ihjihqiu4jACovUZ7vVOCgSE5Ipn7OIwqd93zp2wIDAQABo4HEMIHBMB0GA1UdDgQWBBSsBQ869nh83KqZr5jArr4/7b+QazCBkQYDVR0jBIGJMIGGgBSsBQ869nh83KqZr5jArr4/7b+Qa6FrpGkwZzELMAkGA1UEBhMCVVMxFTATBgNVBAgTDFBlbm5zeWx2YW5pYTETMBEGA1UEBxMKUGl0dHNidXJnaDERMA8GA1UEChMIVGVzdFNoaWIxGTAXBgNVBAMTEGlkcC50ZXN0c2hpYi5vcmeCAQAwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQUFAAOCAQEAjR29PhrCbk8qLN5MFfSVk98t3CT9jHZoYxd8QMRLI4j7iYQxXiGJTT1FXs1nd4Rha9un+LqTfeMMYqISdDDI6tv8iNpkOAvZZUosVkUo93pv1T0RPz35hcHHYq2yee59HJOco2bFlcsH8JBXRSRrJ3Q7Eut+z9uo80JdGNJ4/SJy5UorZ8KazGj16lfJhOBXldgrhppQBb0Nq6HKHguqmwRfJ+WkxemZXzhediAjGeka8nz8JjwxpUjAiSWYKLtJhGEaTqCYxCCX2Dw+dOTqUzHOZ7WKv4JXPK5G/Uhr8K/qhmFT2nIQi538n6rVYLeWj8Bbnl+ev0peYzxFyF5sQA=="));
        Assert.assertEquals((Object)"urn:mace:shibboleth:1.0:nameIdentifier", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/testshib-providers.xml", null, null, (String)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", (String)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/shibboleth", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/profile/SAML2/Redirect/SSO", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEDjCCAvagAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzEVMBMGA1UECBMMUGVubnN5bHZhbmlhMRMwEQYDVQQHEwpQaXR0c2J1cmdoMREwDwYDVQQKEwhUZXN0U2hpYjEZMBcGA1UEAxMQaWRwLnRlc3RzaGliLm9yZzAeFw0wNjA4MzAyMTEyMjVaFw0xNjA4MjcyMTEyMjVaMGcxCzAJBgNVBAYTAlVTMRUwEwYDVQQIEwxQZW5uc3lsdmFuaWExEzARBgNVBAcTClBpdHRzYnVyZ2gxETAPBgNVBAoTCFRlc3RTaGliMRkwFwYDVQQDExBpZHAudGVzdHNoaWIub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArYkCGuTmJp9eAOSGHwRJo1SNatB5ZOKqDM9ysg7CyVTDClcpu93gSP10nH4gkCZOlnESNgttg0r+MqL8tfJC6ybddEFB3YBo8PZajKSe3OQ01Ow3yT4I+Wdg1tsTpSge9gEz7SrC07EkYmHuPtd71CHiUaCWDv+xVfUQX0aTNPFmDixzUjoYzbGDrtAyCqA8f9CN2txIfJnpHE6q6CmKcoLADS4UrNPlhHSzd614kR/JYiks0K4kbRqCQF0Dv0P5Di+rEfefC6glV8ysC8dB5/9nb0yh/ojRuJGmgMWHgWk6h0ihjihqiu4jACovUZ7vVOCgSE5Ipn7OIwqd93zp2wIDAQABo4HEMIHBMB0GA1UdDgQWBBSsBQ869nh83KqZr5jArr4/7b+QazCBkQYDVR0jBIGJMIGGgBSsBQ869nh83KqZr5jArr4/7b+Qa6FrpGkwZzELMAkGA1UEBhMCVVMxFTATBgNVBAgTDFBlbm5zeWx2YW5pYTETMBEGA1UEBxMKUGl0dHNidXJnaDERMA8GA1UEChMIVGVzdFNoaWIxGTAXBgNVBAMTEGlkcC50ZXN0c2hpYi5vcmeCAQAwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQUFAAOCAQEAjR29PhrCbk8qLN5MFfSVk98t3CT9jHZoYxd8QMRLI4j7iYQxXiGJTT1FXs1nd4Rha9un+LqTfeMMYqISdDDI6tv8iNpkOAvZZUosVkUo93pv1T0RPz35hcHHYq2yee59HJOco2bFlcsH8JBXRSRrJ3Q7Eut+z9uo80JdGNJ4/SJy5UorZ8KazGj16lfJhOBXldgrhppQBb0Nq6HKHguqmwRfJ+WkxemZXzhediAjGeka8nz8JjwxpUjAiSWYKLtJhGEaTqCYxCCX2Dw+dOTqUzHOZ7WKv4JXPK5G/Uhr8K/qhmFT2nIQi538n6rVYLeWj8Bbnl+ev0peYzxFyF5sQA=="));
        Assert.assertEquals((Object)"urn:mace:shibboleth:1.0:nameIdentifier", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/testshib-providers.xml", null, null, (String)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact", (String)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact");
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/shibboleth", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.testshib.org/idp/profile/Shibboleth/SSO", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:mace:shibboleth:1.0:profiles:AuthnRequest", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEDjCCAvagAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzEVMBMGA1UECBMMUGVubnN5bHZhbmlhMRMwEQYDVQQHEwpQaXR0c2J1cmdoMREwDwYDVQQKEwhUZXN0U2hpYjEZMBcGA1UEAxMQaWRwLnRlc3RzaGliLm9yZzAeFw0wNjA4MzAyMTEyMjVaFw0xNjA4MjcyMTEyMjVaMGcxCzAJBgNVBAYTAlVTMRUwEwYDVQQIEwxQZW5uc3lsdmFuaWExEzARBgNVBAcTClBpdHRzYnVyZ2gxETAPBgNVBAoTCFRlc3RTaGliMRkwFwYDVQQDExBpZHAudGVzdHNoaWIub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArYkCGuTmJp9eAOSGHwRJo1SNatB5ZOKqDM9ysg7CyVTDClcpu93gSP10nH4gkCZOlnESNgttg0r+MqL8tfJC6ybddEFB3YBo8PZajKSe3OQ01Ow3yT4I+Wdg1tsTpSge9gEz7SrC07EkYmHuPtd71CHiUaCWDv+xVfUQX0aTNPFmDixzUjoYzbGDrtAyCqA8f9CN2txIfJnpHE6q6CmKcoLADS4UrNPlhHSzd614kR/JYiks0K4kbRqCQF0Dv0P5Di+rEfefC6glV8ysC8dB5/9nb0yh/ojRuJGmgMWHgWk6h0ihjihqiu4jACovUZ7vVOCgSE5Ipn7OIwqd93zp2wIDAQABo4HEMIHBMB0GA1UdDgQWBBSsBQ869nh83KqZr5jArr4/7b+QazCBkQYDVR0jBIGJMIGGgBSsBQ869nh83KqZr5jArr4/7b+Qa6FrpGkwZzELMAkGA1UEBhMCVVMxFTATBgNVBAgTDFBlbm5zeWx2YW5pYTETMBEGA1UEBxMKUGl0dHNidXJnaDERMA8GA1UEChMIVGVzdFNoaWIxGTAXBgNVBAMTEGlkcC50ZXN0c2hpYi5vcmeCAQAwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQUFAAOCAQEAjR29PhrCbk8qLN5MFfSVk98t3CT9jHZoYxd8QMRLI4j7iYQxXiGJTT1FXs1nd4Rha9un+LqTfeMMYqISdDDI6tv8iNpkOAvZZUosVkUo93pv1T0RPz35hcHHYq2yee59HJOco2bFlcsH8JBXRSRrJ3Q7Eut+z9uo80JdGNJ4/SJy5UorZ8KazGj16lfJhOBXldgrhppQBb0Nq6HKHguqmwRfJ+WkxemZXzhediAjGeka8nz8JjwxpUjAiSWYKLtJhGEaTqCYxCCX2Dw+dOTqUzHOZ7WKv4JXPK5G/Uhr8K/qhmFT2nIQi538n6rVYLeWj8Bbnl+ev0peYzxFyF5sQA=="));
        Assert.assertEquals((Object)"urn:mace:shibboleth:1.0:nameIdentifier", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseRemoteXML() throws Exception {
        Map idpInfo = IdPMetadataParser.parseRemoteXML((URL)new URL("https://app.onelogin.com/saml/metadata/645460"));
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/645460", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://sgarcia-us-preprod.onelogin.com/trust/saml2/http-redirect/sso/645460", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://sgarcia-us-preprod.onelogin.com/trust/saml2/http-redirect/slo/645460", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        idpInfo = IdPMetadataParser.parseRemoteXML((URL)new URL("https://app.onelogin.com/saml/metadata/383123"));
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/383123", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://try-demo.onelogin.com/trust/saml2/http-redirect/sso/383123", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://try-demo.onelogin.com/trust/saml2/http-redirect/slo/383123", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIDGTCCAoKgAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbTAeFw0xMzA2MDUxNzE2MjBaFw0xODA2MDUxNzE2MjBaMGcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRUwEwYDVQQHDAxTYW50YSBNb25pY2ExETAPBgNVBAoMCE9uZUxvZ2luMRkwFwYDVQQDDBBhcHAub25lbG9naW4uY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCsalDL15zSKeEGy9c0Hao7+G02x6k/MlZuCwEvkPKUcl9QF/q0584lta735hmiZSuWOFQDNQ4VT53VevjAhOtzLJOsa8wcZ+SA1s3j4bNcpUIAHltb4Az6NC7U2/LatfnwscOazEJnVsfL4aaBdpIHBFQ6Ed0StD0AfB6Ci0hURwIDAQABo4HUMIHRMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFHm3fLi+Q1zMc3guMyHy5AHdQvdgMIGRBgNVHSMEgYkwgYaAFHm3fLi+Q1zMc3guMyHy5AHdQvdgoWukaTBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbYIBATAOBgNVHQ8BAf8EBAMCBPAwDQYJKoZIhvcNAQEFBQADgYEANZvzlB1Aq84AdOvsn2XKxBB/PmNZLqnM1VWRPaNcvjafx7eHd5qayXFNQz+bOLujENmgAm5padbydG89SeefpOGcY2TMsVt0RUzxTnN3Zq5G6Ja2fAKOEX01ejdoPPMmStqqSw8k1wPUU8uLYJG5wmjf0rCb8RVaeAwMc+wcEIA="));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseMultiCerts() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/idp_metadata_multi_certs.xml");
        Assert.assertEquals((Object)"https://idp.examle.com/saml/metadata", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/sso", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/slo", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509certMulti.0"))), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xODAxMTcxNTMzNDNaFw0yMTEwMTMxNTMzNDNaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCxejk/DNtB9TlB7PNek/Pds6txAhSbTSIEX6jjKgE170PXCncpkogIO9ae/r3psBll2nU+FbKpnml+Jv81I8nMazQceDg9R4CRnTUV5mwgZShW1DzpEuG3/8TzYcpA41HZQ7Wl7dT19h55speZ8egGptQEcOazMfWmLEI1QhHaowIDAQABo1AwTjAdBgNVHQ4EFgQUmTK9rvir0zDUxKg8iTSh3fMCirowHwYDVR0jBBgwFoAUmTK9rvir0zDUxKg8iTSh3fMCirowDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQBhFvvRdguCYT34NJl884UhcmyEBarSBEEajkn73YAvyqhh+yo4LhWIvam/yFLsNdaDzwo9R8wzAaj4XGMPqM4WwSA69RTIv+n5gSgsrgFSja7HhP7Epw8SxpDQiW0ijh/TUTBvWOuqEEhQQvYRwshyJW7n82+wtArH8pnpFUOFuA=="));
        Assert.assertNull(idpInfo.get("onelogin.saml2.idp.x509certMulti.1"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseMultiSigningCerts() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/idp_metadata_multi_signing_certs.xml");
        Assert.assertEquals((Object)"https://idp.examle.com/saml/metadata", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/sso", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/slo", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509certMulti.0"))), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xNzA0MTUxNjMzMThaFw0xODA0MTUxNjMzMThaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC6GLkl5lDUZdHNDAojp5i24OoPlqrt5TGXJIPqAZYT1hQvJW5nv17MFDHrjmtEnmW4ACKEy0fAX80QWIcHunZSkbEGHb+NG/6oTi5RipXMvmHnfFnPJJ0AdtiLiPE478CV856gXekV4Xx5u3KrylcOgkpYsp0GMIQBDzleMUXlYQIDAQABo1AwTjAdBgNVHQ4EFgQUnP8vlYPGPL2n6ZzDYij2kMDC8wMwHwYDVR0jBBgwFoAUnP8vlYPGPL2n6ZzDYij2kMDC8wMwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQAlQGAl+b8Cpot1g+65lLLjVoY7APJPWLW0klKQNlMU0s4MU+71Y3ExUEOXDAZgKcFoavb1fEOGMwEf38NaJAy1e/l6VNuixXShffq20ymqHQxOG0q8ujeNkgZF9k6XDfn/QZ3AD0o/IrCT7UMc/0QsfgIjWYxwCvp2syApc5CYfQ=="));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509certMulti.1"))), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xODAxMTcxNTMzNDNaFw0yMTEwMTMxNTMzNDNaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCxejk/DNtB9TlB7PNek/Pds6txAhSbTSIEX6jjKgE170PXCncpkogIO9ae/r3psBll2nU+FbKpnml+Jv81I8nMazQceDg9R4CRnTUV5mwgZShW1DzpEuG3/8TzYcpA41HZQ7Wl7dT19h55speZ8egGptQEcOazMfWmLEI1QhHaowIDAQABo1AwTjAdBgNVHQ4EFgQUmTK9rvir0zDUxKg8iTSh3fMCirowHwYDVR0jBBgwFoAUmTK9rvir0zDUxKg8iTSh3fMCirowDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQBhFvvRdguCYT34NJl884UhcmyEBarSBEEajkn73YAvyqhh+yo4LhWIvam/yFLsNdaDzwo9R8wzAaj4XGMPqM4WwSA69RTIv+n5gSgsrgFSja7HhP7Epw8SxpDQiW0ijh/TUTBvWOuqEEhQQvYRwshyJW7n82+wtArH8pnpFUOFuA=="));
        Assert.assertNull(idpInfo.get("onelogin.saml2.idp.x509certMulti.2"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:nameid-format:transient", idpInfo.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseMultiSameSigningAndEncryptCert() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/idp_metadata_same_sign_and_encrypt_cert.xml");
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/383123", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://app.onelogin.com/trust/saml2/http-post/sso/383123", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEHjCCAwagAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbTAeFw0xMzA2MDUxNzE2MjBaFw0xODA2MDUxNzE2MjBaMGcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRUwEwYDVQQHDAxTYW50YSBNb25pY2ExETAPBgNVBAoMCE9uZUxvZ2luMRkwFwYDVQQDDBBhcHAub25lbG9naW4uY29tMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAse8rnep4qL2GmhH10pMQyJ2Jae+AQHyfgVjaQZ7Z0QQog5jX91vcJRSMi0XWJnUtOr6lF0dq1+yckjZ92wyLrH+7fvngNO1aV4Mjk9sTgf+iqMrae6y6fRxDt9PXrEFVjvd3vv7QTJf2FuIPy4vVP06Dt8EMkQIr8rmLmU0mTr1k2DkrdtdlCuNFTXuAu3QqfvNCRrRwfNObn9MP6JeOUdcGLJsBjGF8exfcN1SFzRF0JFr3dmOlx761zK5liD0T1sYWnDquatj/JD9fZMbKecBKni1NglH/LVd+b6aJUAr5LulERULUjLqYJRKW31u91/4Qazdo9tbvwqyFxaoUrwIDAQABo4HUMIHRMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFPWcXvQSlTXnzZD2xziuoUvrrDedMIGRBgNVHSMEgYkwgYaAFPWcXvQSlTXnzZD2xziuoUvrrDedoWukaTBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbYIBATAOBgNVHQ8BAf8EBAMCBPAwDQYJKoZIhvcNAQEFBQADggEBAB/8xe3rzqXQVxzHyAHuAuPa73ClDoL1cko0Fp8CGcqEIyj6Te9gx5z6wyfv+Lo8RFvBLlnB1lXqbC+fTGcVgG/4oKLJ5UwRFxInqpZPnOAudVNnd0PYOODn9FWs6u+OTIQIaIcPUv3MhB9lwHIJsTk/bs9xcru5TPyLIxLLd6ib/pRceKH2mTkzUd0DYk9CQNXXeoGx/du5B9nh3ClPTbVakRzl3oswgI5MQIphYxkW70SopEh4kOFSRE1ND31NNIq1YrXlgtkguQBFsZWuQOPR6cEwFZzP0tHTYbI839WgxX6hfhIUTUz6mLqq4+3P4BG3+1OXeVDg63y8Uh781sE="));
        Assert.assertNull(idpInfo.get("onelogin.saml2.idp.x509certMulti.0"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress", idpInfo.get("onelogin.saml2.sp.nameidformat"));
        Map idpInfo2 = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/idp_metadata_different_sign_and_encrypt_cert.xml");
        Assert.assertEquals((Object)"https://app.onelogin.com/saml/metadata/383123", idpInfo2.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://app.onelogin.com/trust/saml2/http-post/sso/383123", idpInfo2.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo2.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo2.get("onelogin.saml2.idp.x509cert"))), (Object)Util.loadCert((String)"MIIEZTCCA02gAwIBAgIUPyy/A3bZAZ4m28PzEUUoT7RJhxIwDQYJKoZIhvcNAQEFBQAwcjELMAkGA1UEBhMCVVMxKzApBgNVBAoMIk9uZUxvZ2luIFRlc3QgKHNnYXJjaWEtdXMtcHJlcHJvZCkxFTATBgNVBAsMDE9uZUxvZ2luIElkUDEfMB0GA1UEAwwWT25lTG9naW4gQWNjb3VudCA4OTE0NjAeFw0xNjA4MDQyMjI5MzdaFw0yMTA4MDUyMjI5MzdaMHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDN6iqQGcLOCglNO42I2rkzE05UXSiMXT6c8ALThMMiaDw6qqzo3sd/tKK+NcNKWLIIC8TozWVyh5ykUiVZps+08xil7VsTU7E+wKu3kvmOsvw2wlRwtnoKZJwYhnr+RkBa+h1r3ZYUgXm1ZPeHMKj1g18KaWz9+MxYL6BhKqrOzfW/P2xxVRcFH7/pq+ZsDdgNzD2GD+apzY4MZyZj/N6BpBWJ0GlFsmtBegpbX3LBitJuFkk5L4/U/jjF1AJa3boBdCUVfATqO5G03H4XS1GySjBIRQXmlUF52rLjg6xCgWJ30/+t1X+IHLJeixiQ0vxyh6C4/usCEt94cgD1r8ADAgMBAAGjgfIwge8wDAYDVR0TAQH/BAIwADAdBgNVHQ4EFgQUPW0DcH0G3IwynWgi74co4wZ6n7gwga8GA1UdIwSBpzCBpIAUPW0DcH0G3IwynWgi74co4wZ6n7ihdqR0MHIxCzAJBgNVBAYTAlVTMSswKQYDVQQKDCJPbmVMb2dpbiBUZXN0IChzZ2FyY2lhLXVzLXByZXByb2QpMRUwEwYDVQQLDAxPbmVMb2dpbiBJZFAxHzAdBgNVBAMMFk9uZUxvZ2luIEFjY291bnQgODkxNDaCFD8svwN22QGeJtvD8xFFKE+0SYcSMA4GA1UdDwEB/wQEAwIHgDANBgkqhkiG9w0BAQUFAAOCAQEAQhB4q9jrycwbHrDSoYR1X4LFFzvJ9Us75wQquRHXpdyS9D6HUBXMGI6ahPicXCQrfLgN8vzMIiqZqfySXXv/8/dxe/X4UsWLYKYJHDJmxXD5EmWTa65chjkeP1oJAc8f3CKCpcP2lOBTthbnk2fEVAeLHR4xNdQO0VvGXWO9BliYPpkYqUIBvlm+Fg9mF7AM/Uagq2503XXIE1Lq//HON68P10vNMwLSKOtYLsoTiCnuIKGJqG37MsZVjQ1ZPRcO+LSLkq0i91gFxrOrVCrgztX4JQi5XkvEsYZGIXXjwHqxTVyt3adZWQO0LPxPqRiUqUzyhDhLo/xXNrHCu4VbMw=="));
        Assert.assertEquals((Object)Util.loadCert((String)((String)idpInfo2.get("onelogin.saml2.idp.x509certMulti.0"))), (Object)Util.loadCert((String)"MIIEHjCCAwagAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbTAeFw0xMzA2MDUxNzE2MjBaFw0xODA2MDUxNzE2MjBaMGcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRUwEwYDVQQHDAxTYW50YSBNb25pY2ExETAPBgNVBAoMCE9uZUxvZ2luMRkwFwYDVQQDDBBhcHAub25lbG9naW4uY29tMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAse8rnep4qL2GmhH10pMQyJ2Jae+AQHyfgVjaQZ7Z0QQog5jX91vcJRSMi0XWJnUtOr6lF0dq1+yckjZ92wyLrH+7fvngNO1aV4Mjk9sTgf+iqMrae6y6fRxDt9PXrEFVjvd3vv7QTJf2FuIPy4vVP06Dt8EMkQIr8rmLmU0mTr1k2DkrdtdlCuNFTXuAu3QqfvNCRrRwfNObn9MP6JeOUdcGLJsBjGF8exfcN1SFzRF0JFr3dmOlx761zK5liD0T1sYWnDquatj/JD9fZMbKecBKni1NglH/LVd+b6aJUAr5LulERULUjLqYJRKW31u91/4Qazdo9tbvwqyFxaoUrwIDAQABo4HUMIHRMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFPWcXvQSlTXnzZD2xziuoUvrrDedMIGRBgNVHSMEgYkwgYaAFPWcXvQSlTXnzZD2xziuoUvrrDedoWukaTBnMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEVMBMGA1UEBwwMU2FudGEgTW9uaWNhMREwDwYDVQQKDAhPbmVMb2dpbjEZMBcGA1UEAwwQYXBwLm9uZWxvZ2luLmNvbYIBATAOBgNVHQ8BAf8EBAMCBPAwDQYJKoZIhvcNAQEFBQADggEBAB/8xe3rzqXQVxzHyAHuAuPa73ClDoL1cko0Fp8CGcqEIyj6Te9gx5z6wyfv+Lo8RFvBLlnB1lXqbC+fTGcVgG/4oKLJ5UwRFxInqpZPnOAudVNnd0PYOODn9FWs6u+OTIQIaIcPUv3MhB9lwHIJsTk/bs9xcru5TPyLIxLLd6ib/pRceKH2mTkzUd0DYk9CQNXXeoGx/du5B9nh3ClPTbVakRzl3oswgI5MQIphYxkW70SopEh4kOFSRE1ND31NNIq1YrXlgtkguQBFsZWuQOPR6cEwFZzP0tHTYbI839WgxX6hfhIUTUz6mLqq4+3P4BG3+1OXeVDg63y8Uh781sE="));
        Assert.assertNull(idpInfo2.get("onelogin.saml2.idp.x509certMulti.2"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress", idpInfo2.get("onelogin.saml2.sp.nameidformat"));
    }

    @Test
    public void testParseSeparateSingleLogoutServiceResponseLocation() throws Exception {
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/metadata_slo_responselocation.xml");
        Assert.assertEquals((Object)"https://idp.examle.com/saml/metadata", idpInfo.get("onelogin.saml2.idp.entityid"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/sso", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_sign_on_service.binding"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/slo", idpInfo.get("onelogin.saml2.idp.single_logout_service.url"));
        Assert.assertEquals((Object)"https://idp.examle.com/saml/sloresp", idpInfo.get("onelogin.saml2.idp.single_logout_service.response.url"));
        Assert.assertEquals((Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect", idpInfo.get("onelogin.saml2.idp.single_logout_service.binding"));
    }

    @Test
    public void testInjectIntoSettings() throws Exception {
        Saml2Settings setting = new SettingsBuilder().fromFile("config/config.all.properties").build();
        Assert.assertEquals((Object)"http://idp.example.com/", (Object)setting.getIdpEntityId());
        Assert.assertEquals((Object)"http://idp.example.com/simplesaml/saml2/idp/SSOService.php", (Object)setting.getIdpSingleSignOnServiceUrl().toString());
        Assert.assertEquals((Object)setting.getIdpSingleSignOnServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)"http://idp.example.com/simplesaml/saml2/idp/SingleLogoutService.php", (Object)setting.getIdpSingleLogoutServiceUrl().toString());
        Assert.assertEquals((Object)setting.getIdpSingleLogoutServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)Util.loadCert((String)Util.getFileAsString((String)"certs/certificate1")), (Object)setting.getIdpx509cert());
        Assert.assertEquals((Object)setting.getSpNameIDFormat(), (Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:unspecified");
        Assert.assertEquals((Object)"http://localhost:8080/java-saml-jspsample/metadata.jsp", (Object)setting.getSpEntityId());
        Assert.assertEquals((Object)"http://www.w3.org/2001/04/xmldsig-more#rsa-sha512", (Object)setting.getSignatureAlgorithm());
        Assert.assertEquals((Object)"http://www.w3.org/2001/04/xmlenc#sha512", (Object)setting.getDigestAlgorithm());
        Assert.assertEquals((long)4L, (long)setting.getContacts().size());
        Assert.assertEquals((Object)"administrative@example.com", ((Contact)setting.getContacts().get(0)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"administrative2@example.com", ((Contact)setting.getContacts().get(0)).getEmailAddresses().get(1));
        Assert.assertEquals((Object)"info@example.com", ((Contact)setting.getContacts().get(1)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"technical@example.com", ((Contact)setting.getContacts().get(2)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"support@example.com", ((Contact)setting.getContacts().get(3)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"SP Java", (Object)setting.getOrganization().getOrgName());
        Assert.assertEquals((Object)"EXAMPLE", (Object)setting.getUniqueIDPrefix());
        Map idpInfo = IdPMetadataParser.parseFileXML((String)"data/metadata/idp/FederationMetadata.xml");
        setting = IdPMetadataParser.injectIntoSettings((Saml2Settings)setting, (Map)idpInfo);
        Assert.assertEquals((Object)"http://idp.adfs.example.com/adfs/services/trust", (Object)setting.getIdpEntityId());
        Assert.assertEquals((Object)"https://idp.adfs.example.com/adfs/ls/", (Object)setting.getIdpSingleSignOnServiceUrl().toString());
        Assert.assertEquals((Object)setting.getIdpSingleSignOnServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)"https://idp.adfs.example.com/adfs/ls/", (Object)setting.getIdpSingleLogoutServiceUrl().toString());
        Assert.assertEquals((Object)setting.getIdpSingleLogoutServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)setting.getIdpx509cert(), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xNzA0MTUxMjI3NTFaFw0yNzA0MTMxMjI3NTFaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCYtEZ7hGZiNp+NecbcQXosYl8TzVOdL44b3Nl+BxL26Bvnt8YNnE63xiQzo7xDdO6+1MWWO26mMxwMpooTToOJgrot9YhlIX1VHIUPbOEGczSmXzCCmMhS26vR/leoLNah8QqCF1UdCoNQejb0fDCy+Q1yEdMXYkBWsFGfDSHSSQIDAQABo1AwTjAdBgNVHQ4EFgQUT1g33aGN0f6BJPgpYbr1pHrMZrYwHwYDVR0jBBgwFoAUT1g33aGN0f6BJPgpYbr1pHrMZrYwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQB6233Ic9bb6OCMT6hE1mRzhoP+AbixeojtUuM1IUG4JI5YUGsjsym96VBw+/ciwDLuxNYg6ZWu++WxWNwF3LwVRZGQ8bDdxYldm6VorvIbps2tzyT5N32xgMAgzy/3SZf6YOihdotXJd5AZNVp/razVO17WrjsFvldAlKtk0SM7w=="));
        Assert.assertEquals(setting.getIdpx509certMulti().get(0), (Object)Util.loadCert((String)"MIIC9jCCAd6gAwIBAgIQI/B8CLE676pCR2/QaKih9TANBgkqhkiG9w0BAQsFADA3MTUwMwYDVQQDEyxBREZTIFNpZ25pbmcgLSBsb2dpbnRlc3Qub3dlbnNib3JvaGVhbHRoLm9yZzAeFw0xNjEwMjUxNjI4MzhaFw0xNzEwMjUxNjI4MzhaMDcxNTAzBgNVBAMTLEFERlMgU2lnbmluZyAtIGxvZ2ludGVzdC5vd2Vuc2Jvcm9oZWFsdGgub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAjikmKRRVD5oK3fxm0xNfDqvWCujZIhtv2zeIwmoRKUAjo6KeUhauII4BHh5DclmbOFD4ruli3sNWGKgqVCX1AFW/p3m3/FtzeumFeZSmyfqeJEeOqAK5jAom/MfXxaQ85QHlGa0BTtdWdCuxhJz5G797o4s1Me/8QOQdmbkkwOHOVXRDW0QxBXvsRB1jPpIO+JvNcWFpvJrELccD0Fws91LH42j2C4gDNR8JLu5LrUGL6zAIq8NM7wfbwoax9n/0tIZKa6lo6szpXGqiMrDBJPpAqC5MSePyp5/SEX6jxwodQUGRgI5bKILQwOWDrkgfsK1MIeHfovtyqnDZj8e9VwIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQBKbK4qu7WTLYeQW7OcFAeWcT5D7ujo61QtPf+6eY8hpNntN8yF71vGm+5zdOjmw18igxUrf3W7dLk2wAogXK196WX34x9muorwmFK/HqmKuy0kWWzGcNzZHb0o4Md2Ux7QQVoHqD6dUSqUisOBs34ZPgT5R42LepJTGDEZSkvOxUv9V6fY5dYk8UaWbZ7MQAFi1CnOyybq2nVNjpuxWyJ6SsHQYKRhXa7XGurXFB2mlgcjVj9jxW0gO7djkgRD68b6PNpQmJkbKnkCtJg9YsSeOmuUjwgh4DlcIo5jZocKd5bnLbQ9XKJ3YQHRxFoZbP3BXKrfhVV3vqqzRxMwjZmK"));
        Assert.assertEquals((Object)setting.getSpNameIDFormat(), (Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress");
        Assert.assertEquals((Object)"http://localhost:8080/java-saml-jspsample/metadata.jsp", (Object)setting.getSpEntityId());
        Assert.assertEquals((Object)"http://www.w3.org/2001/04/xmldsig-more#rsa-sha512", (Object)setting.getSignatureAlgorithm());
        Assert.assertEquals((Object)"http://www.w3.org/2001/04/xmlenc#sha512", (Object)setting.getDigestAlgorithm());
        Assert.assertEquals((long)4L, (long)setting.getContacts().size());
        Assert.assertEquals((Object)"administrative@example.com", ((Contact)setting.getContacts().get(0)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"administrative2@example.com", ((Contact)setting.getContacts().get(0)).getEmailAddresses().get(1));
        Assert.assertEquals((Object)"info@example.com", ((Contact)setting.getContacts().get(1)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"technical@example.com", ((Contact)setting.getContacts().get(2)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"support@example.com", ((Contact)setting.getContacts().get(3)).getEmailAddresses().get(0));
        Assert.assertEquals((Object)"SP Java", (Object)setting.getOrganization().getOrgName());
        Assert.assertEquals((Object)"EXAMPLE", (Object)setting.getUniqueIDPrefix());
        Saml2Settings setting2 = new SettingsBuilder().fromFile("config/config.min.properties").build();
        Assert.assertEquals((Object)"http://idp.example.com/", (Object)setting2.getIdpEntityId());
        Assert.assertEquals((Object)"http://idp.example.com/simplesaml/saml2/idp/SSOService.php", (Object)setting2.getIdpSingleSignOnServiceUrl().toString());
        Assert.assertEquals((Object)setting2.getIdpSingleSignOnServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)"http://idp.example.com/simplesaml/saml2/idp/SingleLogoutService.php", (Object)setting2.getIdpSingleLogoutServiceUrl().toString());
        Assert.assertEquals((Object)setting2.getIdpSingleLogoutServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)Util.loadCert((String)Util.getFileAsString((String)"certs/certificate1")), (Object)setting2.getIdpx509cert());
        Assert.assertEquals((Object)setting2.getSpNameIDFormat(), (Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:unspecified");
        Assert.assertEquals((Object)"http://localhost:8080/java-saml-jspsample/metadata.jsp", (Object)setting2.getSpEntityId());
        Assert.assertEquals((Object)"http://www.w3.org/2000/09/xmldsig#rsa-sha1", (Object)setting2.getSignatureAlgorithm());
        Assert.assertEquals((Object)"http://www.w3.org/2000/09/xmldsig#sha1", (Object)setting2.getDigestAlgorithm());
        Assert.assertEquals((long)0L, (long)setting2.getContacts().size());
        Assert.assertNull((Object)setting2.getOrganization());
        Assert.assertEquals((Object)"ONELOGIN_", (Object)setting2.getUniqueIDPrefix());
        setting2 = IdPMetadataParser.injectIntoSettings((Saml2Settings)setting2, (Map)idpInfo);
        Assert.assertEquals((Object)"http://idp.adfs.example.com/adfs/services/trust", (Object)setting2.getIdpEntityId());
        Assert.assertEquals((Object)"https://idp.adfs.example.com/adfs/ls/", (Object)setting2.getIdpSingleSignOnServiceUrl().toString());
        Assert.assertEquals((Object)setting2.getIdpSingleSignOnServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)"https://idp.adfs.example.com/adfs/ls/", (Object)setting2.getIdpSingleLogoutServiceUrl().toString());
        Assert.assertEquals((Object)setting2.getIdpSingleLogoutServiceBinding(), (Object)"urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect");
        Assert.assertEquals((Object)setting2.getIdpx509cert(), (Object)Util.loadCert((String)"MIICZDCCAc2gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBPMQswCQYDVQQGEwJ1czEUMBIGA1UECAwLZXhhbXBsZS5jb20xFDASBgNVBAoMC2V4YW1wbGUuY29tMRQwEgYDVQQDDAtleGFtcGxlLmNvbTAeFw0xNzA0MTUxMjI3NTFaFw0yNzA0MTMxMjI3NTFaME8xCzAJBgNVBAYTAnVzMRQwEgYDVQQIDAtleGFtcGxlLmNvbTEUMBIGA1UECgwLZXhhbXBsZS5jb20xFDASBgNVBAMMC2V4YW1wbGUuY29tMIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCYtEZ7hGZiNp+NecbcQXosYl8TzVOdL44b3Nl+BxL26Bvnt8YNnE63xiQzo7xDdO6+1MWWO26mMxwMpooTToOJgrot9YhlIX1VHIUPbOEGczSmXzCCmMhS26vR/leoLNah8QqCF1UdCoNQejb0fDCy+Q1yEdMXYkBWsFGfDSHSSQIDAQABo1AwTjAdBgNVHQ4EFgQUT1g33aGN0f6BJPgpYbr1pHrMZrYwHwYDVR0jBBgwFoAUT1g33aGN0f6BJPgpYbr1pHrMZrYwDAYDVR0TBAUwAwEB/zANBgkqhkiG9w0BAQ0FAAOBgQB6233Ic9bb6OCMT6hE1mRzhoP+AbixeojtUuM1IUG4JI5YUGsjsym96VBw+/ciwDLuxNYg6ZWu++WxWNwF3LwVRZGQ8bDdxYldm6VorvIbps2tzyT5N32xgMAgzy/3SZf6YOihdotXJd5AZNVp/razVO17WrjsFvldAlKtk0SM7w=="));
        Assert.assertEquals(setting2.getIdpx509certMulti().get(0), (Object)Util.loadCert((String)"MIIC9jCCAd6gAwIBAgIQI/B8CLE676pCR2/QaKih9TANBgkqhkiG9w0BAQsFADA3MTUwMwYDVQQDEyxBREZTIFNpZ25pbmcgLSBsb2dpbnRlc3Qub3dlbnNib3JvaGVhbHRoLm9yZzAeFw0xNjEwMjUxNjI4MzhaFw0xNzEwMjUxNjI4MzhaMDcxNTAzBgNVBAMTLEFERlMgU2lnbmluZyAtIGxvZ2ludGVzdC5vd2Vuc2Jvcm9oZWFsdGgub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAjikmKRRVD5oK3fxm0xNfDqvWCujZIhtv2zeIwmoRKUAjo6KeUhauII4BHh5DclmbOFD4ruli3sNWGKgqVCX1AFW/p3m3/FtzeumFeZSmyfqeJEeOqAK5jAom/MfXxaQ85QHlGa0BTtdWdCuxhJz5G797o4s1Me/8QOQdmbkkwOHOVXRDW0QxBXvsRB1jPpIO+JvNcWFpvJrELccD0Fws91LH42j2C4gDNR8JLu5LrUGL6zAIq8NM7wfbwoax9n/0tIZKa6lo6szpXGqiMrDBJPpAqC5MSePyp5/SEX6jxwodQUGRgI5bKILQwOWDrkgfsK1MIeHfovtyqnDZj8e9VwIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQBKbK4qu7WTLYeQW7OcFAeWcT5D7ujo61QtPf+6eY8hpNntN8yF71vGm+5zdOjmw18igxUrf3W7dLk2wAogXK196WX34x9muorwmFK/HqmKuy0kWWzGcNzZHb0o4Md2Ux7QQVoHqD6dUSqUisOBs34ZPgT5R42LepJTGDEZSkvOxUv9V6fY5dYk8UaWbZ7MQAFi1CnOyybq2nVNjpuxWyJ6SsHQYKRhXa7XGurXFB2mlgcjVj9jxW0gO7djkgRD68b6PNpQmJkbKnkCtJg9YsSeOmuUjwgh4DlcIo5jZocKd5bnLbQ9XKJ3YQHRxFoZbP3BXKrfhVV3vqqzRxMwjZmK"));
        Assert.assertEquals((Object)setting2.getSpNameIDFormat(), (Object)"urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress");
        Assert.assertEquals((Object)"http://localhost:8080/java-saml-jspsample/metadata.jsp", (Object)setting2.getSpEntityId());
        Assert.assertEquals((Object)"http://www.w3.org/2000/09/xmldsig#rsa-sha1", (Object)setting2.getSignatureAlgorithm());
        Assert.assertEquals((Object)"http://www.w3.org/2000/09/xmldsig#sha1", (Object)setting2.getDigestAlgorithm());
        Assert.assertEquals((long)0L, (long)setting2.getContacts().size());
        Assert.assertNull((Object)setting2.getOrganization());
        Assert.assertEquals((Object)"ONELOGIN_", (Object)setting2.getUniqueIDPrefix());
    }
}

