/**
Copyright 2022 FRC Team 997

This program is free software: 
you can redistribute it and/or modify it under the terms of the 
GNU General Public License as published by the Free Software Foundation, 
either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of 
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with SpartanLib2. 
If not, see <https://www.gnu.org/licenses/>.
*/
package org.chsrobotics.lib.telemetry;

import edu.wpi.first.networktables.NetworkTable;
import edu.wpi.first.networktables.NetworkTableInstance;
import edu.wpi.first.util.datalog.DataLog;
import edu.wpi.first.util.sendable.Sendable;
import edu.wpi.first.util.sendable.SendableRegistry;
import edu.wpi.first.wpilibj.DataLogManager;
import edu.wpi.first.wpilibj.DriverStation;
import edu.wpi.first.wpilibj.Filesystem;
import edu.wpi.first.wpilibj.RobotBase;
import edu.wpi.first.wpilibj.RobotController;
import edu.wpi.first.wpilibj.Timer;
import edu.wpi.first.wpilibj.smartdashboard.SendableBuilderImpl;
import edu.wpi.first.wpilibj2.command.Command;
import edu.wpi.first.wpilibj2.command.CommandScheduler;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * Convenience wrapper class for telemetry/ logging with built-in logging for robot-agnostic data
 * like environment metadata, RoboRio status, and scheduled commands.
 *
 * <p>For the internally included loggers of system status to function, the method {@code
 * logPeriodic()} needs to be called once per robot loop cycle.
 */
public class HighLevelLogger {
    private static boolean hasStarted = false;
    private static final String commitDataFilename = "commit.txt";
    private static final String branchDataFilename = "branch.txt";

    private static HashMap<Command, Timer> commandTimeMap = new HashMap<>();

    private static final Logger<String[]> scheduledCommandsLogger =
            new Logger<>("scheduledCommands", "commandScheduler");

    private static final String subdirString = "System";

    private static final Logger<Boolean> isBrownedOutLogger =
            new Logger<>("isBrownedOut", subdirString);

    private static final Logger<Double> canUtilizationLogger =
            new Logger<>("canUtilizationPercent", subdirString);

    private static final Logger<Double> batteryVoltageLogger =
            new Logger<>("batteryVoltageVolts", subdirString);

    private static final Logger<Double> logger3p3vCurrent =
            new Logger<>("3p3vCurrentAmps", subdirString);
    private static final Logger<Double> logger5vCurrent =
            new Logger<>("5vCurrentAmps", subdirString);

    private static final Logger<Integer> brownoutCounterLogger =
            new Logger<>("brownoutCounter", subdirString);
    private static int brownoutCounter = 0;

    private static final NetworkTable sendables =
            NetworkTableInstance.getDefault().getTable("sendables");

    private static final Map<String, Sendable> tablesToData = new HashMap<>();

    /**
     * Starts the HighLevelLogger.
     *
     * <p>Upon startup, the logger attempts to read git commit/branch data from the files in the
     * deploy directory of the RoboRio specified in the source for this class (currently
     * "commit.txt" and "branch.txt").
     *
     * <p>These should be updated by the build.gradle of your functional robot code as specified by
     * <a
     * href="https://docs.wpilib.org/en/stable/docs/software/advanced-gradlerio/deploy-git-data.html">this</a>
     * documentation from WPI.
     */
    public static void startLogging() {
        if (!hasStarted) {
            CommandScheduler.getInstance().onCommandInitialize(HighLevelLogger::logCommandInit);
            CommandScheduler.getInstance().onCommandFinish(HighLevelLogger::logCommandFinished);
            CommandScheduler.getInstance()
                    .onCommandInterrupt(HighLevelLogger::logCommandInterrupted);

            hasStarted = true;
            DataLogManager.logNetworkTables(false);
            logMessage("Log init");

            logMessage("Real time: " + LocalDateTime.now().toString());
            logMessage("Robot is: " + (RobotBase.isReal() ? "real" : "simulated"));
            logMessage(
                    "Event: "
                            + (DriverStation.isFMSAttached()
                                    ? DriverStation.getEventName()
                                    : "N/A"));
            logMessage(
                    "Match type: "
                            + (DriverStation.isFMSAttached()
                                    ? DriverStation.getMatchType().toString()
                                    : "N/A"));
            logMessage(
                    "Match number: "
                            + (DriverStation.isFMSAttached()
                                    ? DriverStation.getMatchNumber()
                                    : "N/A"));
            try {
                File commitTxt = new File(Filesystem.getDeployDirectory(), commitDataFilename);
                File branchTxt = new File(Filesystem.getDeployDirectory(), branchDataFilename);

                logMessage("Git commit: " + Files.readString(commitTxt.toPath()));
                logMessage("Git branch: " + Files.readString(branchTxt.toPath()));
            } catch (IOException exc) {
                logMessage("Git branch / commit data could not be read!");
            }
        }
    }

    /**
     * Method to be called once per robot loop cycle, updating the various Loggers wrapped by this
     * class.
     */
    public static void logPeriodic() {
        ArrayList<String> commands = new ArrayList<>();

        for (Command command : commandTimeMap.keySet()) {
            commands.add(command.getName());
        }

        scheduledCommandsLogger.update(commands.toArray(new String[] {}));

        if (RobotController.getBatteryVoltage() < RobotController.getBrownoutVoltage()) {
            brownoutCounter++;
            isBrownedOutLogger.update(true);
        } else {
            isBrownedOutLogger.update(false);
        }

        brownoutCounterLogger.update(brownoutCounter);

        canUtilizationLogger.update(RobotController.getCANStatus().percentBusUtilization);

        batteryVoltageLogger.update(RobotController.getBatteryVoltage());

        logger3p3vCurrent.update(RobotController.getCurrent3V3());
        logger5vCurrent.update(RobotController.getCurrent5V());
    }

    /**
     * Returns the DataLog populated by this, for use in more-granular and robot-specific logging.
     *
     * @return The DataLog.
     */
    public static DataLog getLog() {
        if (!hasStarted) {
            startLogging();
        }
        return DataLogManager.getLog();
    }

    /**
     * Logs a String message (warning, state transition, startup information, etc.), and prints it
     * to the standard output (DriverStation console).
     *
     * @param message The message to log.
     */
    public static void logMessage(String message) {
        if (!hasStarted) {
            startLogging();
        }
        DataLogManager.log(message);
    }

    /**
     * Writes a warning, using the provided string, to the DriverStation console, and writes it to
     * the log.
     *
     * @param message The String message to associate with the warning.
     */
    public static void logWarning(String message) {
        logMessage("WARNING " + message);
        DriverStation.reportWarning(message, false);
    }

    /**
     * Writes an error, using the provided string, to the DriverStation console, and writes it to
     * the log.
     *
     * @param message The String message to associate with the error.
     */
    public static void logError(String message) {
        logMessage("ERROR " + message);
        DriverStation.reportError(message, false);
    }

    /**
     * Publishes a Sendable object to NetworkTables.
     *
     * @param key String key to associate with the object.
     * @param data Sendable object.
     */
    public static synchronized void publishSendable(String key, Sendable data) {
        Sendable sddata = tablesToData.get(key);
        if (sddata == null || sddata != data) {
            tablesToData.put(key, data);
            NetworkTable dataTable = sendables.getSubTable(key);
            SendableBuilderImpl builder = new SendableBuilderImpl();
            builder.setTable(dataTable);
            SendableRegistry.publish(data, builder);
            builder.startListeners();
            dataTable.getEntry(".name").setString(key);
        }
    }

    private static void logCommandInit(Command command) {
        logMessage("Command initialized: " + command.getName());

        Timer timer = new Timer();
        timer.reset();
        timer.start();

        commandTimeMap.put(command, timer);
    }

    private static void logCommandFinished(Command command) {
        logMessage(
                "Command finished after "
                        + commandTimeMap.get(command).get()
                        + " seconds: "
                        + command.getName());

        commandTimeMap.remove(command);
    }

    private static void logCommandInterrupted(Command command) {
        logMessage(
                "Command interrupted after "
                        + commandTimeMap.get(command).get()
                        + " seconds: "
                        + command.getName());

        commandTimeMap.remove(command);
    }
}
