/*
 * Decompiled with CFR 0.152.
 */
package org.cardanofoundation.hydra.test;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Duration;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.stringtemplate.v4.ST;
import org.testcontainers.containers.BindMode;
import org.testcontainers.containers.Container;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.containers.Network;
import org.testcontainers.containers.output.OutputFrame;
import org.testcontainers.containers.output.Slf4jLogConsumer;
import org.testcontainers.containers.startupcheck.OneShotStartupCheckStrategy;
import org.testcontainers.containers.wait.strategy.Wait;
import org.testcontainers.containers.wait.strategy.WaitStrategy;
import org.testcontainers.lifecycle.Startable;
import org.testcontainers.lifecycle.Startables;

public class HydraDevNetwork
implements Startable {
    private static final Logger log = LoggerFactory.getLogger(HydraDevNetwork.class);
    private static final String ISO_8601BASIC_DATE_PATTERN = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    private static final String INPUT_OUTPUT_CARDANO_NODE = "inputoutput/cardano-node:1.35.7";
    private static final String INPUT_OUTPUT_HYDRA_NODE = "ghcr.io/input-output-hk/hydra-node:0.10.0";
    protected static final ObjectMapper objectMapper = new ObjectMapper();
    private static final String MARKER_DATUM_HASH = "a654fb60d21c1fed48db2c320aa6df9737ec0204c0ba53b9b94a09fb40e757f3";
    public static final int CARDANO_REMOTE_PORT = 3001;
    public static final int HYDRA_API_REMOTE_PORT = 4001;
    private final boolean cardanoLogging;
    private final boolean hydraLogging;
    private final Map<String, Integer> initialFunds;
    protected GenericContainer<?> cardanoContainer;
    protected GenericContainer<?> aliceHydraContainer;
    protected GenericContainer<?> bobHydraContainer;

    public HydraDevNetwork(boolean withCardanoLogging, boolean withHydraLogging, Map<String, Integer> initialFunds) {
        this.cardanoLogging = withCardanoLogging;
        this.hydraLogging = withHydraLogging;
        this.initialFunds = initialFunds;
        this.cardanoContainer = this.createCardanoNodeContainer();
    }

    public HydraDevNetwork() {
        this(false, false, HydraDevNetwork.getInitialFunds());
    }

    public static Map<String, Integer> getInitialFunds() {
        LinkedHashMap<String, Integer> initialFunds = new LinkedHashMap<String, Integer>();
        initialFunds.put("alice", 1000);
        initialFunds.put("bob", 500);
        return initialFunds;
    }

    public void start() {
        log.info("Preparing devnet (cardano-node, hydra)...");
        this.prepareDevNet();
        log.info("Starting cardano node...");
        this.cardanoContainer.start();
        log.info("Seeding actors with initial funds...");
        this.seedActors(this.cardanoContainer);
        log.info("Publishing Hydra contract scripts to devnet cardano network.");
        String referenceScriptsTxId = this.publishReferenceScripts(this.cardanoContainer);
        log.info("ReferenceScriptsTxId:{}", (Object)referenceScriptsTxId);
        Network.NetworkImpl network = Network.builder().driver("bridge").build();
        log.info("Creating network:" + network);
        this.aliceHydraContainer = this.createAliceHydraNode(this.cardanoContainer, referenceScriptsTxId, (Network)network);
        this.bobHydraContainer = this.createBobHydraNode(this.cardanoContainer, referenceScriptsTxId, (Network)network);
        this.aliceHydraContainer.dependsOn(new Startable[]{this.cardanoContainer});
        this.bobHydraContainer.dependsOn(new Startable[]{this.cardanoContainer});
        log.info("Starting alice and bob hydra nodes in parallel...");
        Startables.deepStart((Startable[])new Startable[]{this.aliceHydraContainer, this.bobHydraContainer}).get();
    }

    public GenericContainer<?> getAliceHydraContainer() {
        return this.aliceHydraContainer;
    }

    public GenericContainer<?> getBobHydraContainer() {
        return this.bobHydraContainer;
    }

    public void stop() {
        log.info("Cleaning up container resources...");
        if (this.aliceHydraContainer != null && this.aliceHydraContainer.isRunning()) {
            this.aliceHydraContainer.stop();
            this.aliceHydraContainer = null;
        }
        if (this.bobHydraContainer != null && this.bobHydraContainer.isRunning()) {
            this.bobHydraContainer.stop();
            this.bobHydraContainer = null;
        }
        if (this.cardanoContainer != null && this.cardanoContainer.isRunning()) {
            this.cardanoContainer.stop();
            this.cardanoContainer = null;
        }
        String devnetPath = Resources.getResource((String)"devnet").getPath();
        Files.deleteIfExists(Paths.get(devnetPath, "node.socket"));
        Files.deleteIfExists(Paths.get(devnetPath, "genesis-byron.json"));
        Files.deleteIfExists(Paths.get(devnetPath, "genesis-shelley.json"));
    }

    public static String getHydraApiUrl(GenericContainer<?> container) {
        String host = container.getHost();
        Integer mappedPort = container.getMappedPort(4001);
        return String.format("ws://%s:%d", host, mappedPort);
    }

    private String publishReferenceScripts(GenericContainer<?> cardanoContainer) {
        StringBuilder commandOutputBuilder = new StringBuilder();
        try (GenericContainer hydraCliContainer = new GenericContainer(INPUT_OUTPUT_HYDRA_NODE);){
            hydraCliContainer.withVolumesFrom(cardanoContainer, BindMode.READ_WRITE).withStartupCheckStrategy(new OneShotStartupCheckStrategy().withTimeout(Duration.ofSeconds(60L))).withLogConsumer(new Slf4jLogConsumer(log).andThen(line -> {
                if (line.getType() == OutputFrame.OutputType.STDOUT) {
                    commandOutputBuilder.append(line.getUtf8String());
                }
            })).withCommand(new String[]{"publish-scripts", "--testnet-magic", "42", "--node-socket", "/devnet/node.socket", "--cardano-signing-key", "/devnet/credentials/faucet.sk"});
            if (this.cardanoLogging) {
                hydraCliContainer.withLogConsumer((Consumer)new Slf4jLogConsumer(log).withSeparateOutputStreams());
            }
            hydraCliContainer.start();
        }
        log.info("Publishing reference scripts...");
        return commandOutputBuilder.toString().replace("\n", "");
    }

    protected void prepareDevNet() throws IOException {
        String devnetPath = Resources.getResource((String)"devnet").getPath();
        String byronFile = "genesis-byron.json";
        String shelleyFile = "genesis-shelley.json";
        String vrfKeyFile = "vrf.skey";
        ST byronST = new ST(com.google.common.io.Files.toString((File)new File(devnetPath, byronFile + ".tmpl"), (Charset)StandardCharsets.UTF_8));
        ST shelleyST = new ST(com.google.common.io.Files.toString((File)new File(devnetPath, shelleyFile + ".tmpl"), (Charset)StandardCharsets.UTF_8));
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC);
        byronST.add("START_TIME", (Object)now.toLocalDateTime().toInstant(ZoneOffset.UTC).getEpochSecond());
        shelleyST.add("START_TIME", (Object)now.format(DateTimeFormatter.ofPattern(ISO_8601BASIC_DATE_PATTERN)));
        Files.deleteIfExists(Paths.get(devnetPath, byronFile));
        Files.deleteIfExists(Paths.get(devnetPath, shelleyFile));
        Path byron = Files.write(Paths.get(devnetPath, byronFile), byronST.render().getBytes(), new OpenOption[0]);
        Path shelley = Files.write(Paths.get(devnetPath, shelleyFile), shelleyST.render().getBytes(), new OpenOption[0]);
        Path vrfPath = Paths.get(devnetPath, vrfKeyFile);
        new File(byron.toUri()).setReadOnly();
        new File(shelley.toUri()).setReadOnly();
        Files.setPosixFilePermissions(vrfPath, PosixFilePermissions.fromString("rw-------"));
    }

    protected GenericContainer<?> createCardanoNodeContainer() {
        try (GenericContainer cardanoNode = new GenericContainer(INPUT_OUTPUT_CARDANO_NODE);){
            cardanoNode.withClasspathResourceMapping("/devnet", "/devnet", BindMode.READ_WRITE).withEnv(Map.of("CARDANO_BLOCK_PRODUCER", "true", "CARDANO_NODE_SOCKET_PATH", "/devnet/node.socket", "CARDANO_SOCKET_PATH", "/devnet/node.socket")).withLogConsumer((Consumer)new Slf4jLogConsumer(log).withPrefix("cardano-node").withSeparateOutputStreams()).withExposedPorts(new Integer[]{3001}).waitingFor((WaitStrategy)Wait.forListeningPort()).withCommand(new String[]{"run", "--config", "/devnet/cardano-node.json", "--topology", "/devnet/topology.json", "--database-path", "/tmp/cardano-db_" + System.currentTimeMillis(), "--shelley-kes-key", "/devnet/kes.skey", "--shelley-vrf-key", "/devnet/vrf.skey", "--shelley-operational-certificate", "/devnet/opcert.cert", "--byron-delegation-certificate", "/devnet/byron-delegation.cert", "--byron-signing-key", "/devnet/byron-delegate.key"});
            GenericContainer genericContainer = cardanoNode;
            return genericContainer;
        }
    }

    protected GenericContainer<?> createAliceHydraNode(GenericContainer<?> cardanoContainer, String scriptsTxId, Network network) {
        String containerName = "alice-hydra-node";
        try (GenericContainer aliceHydraNode = new GenericContainer(INPUT_OUTPUT_HYDRA_NODE);){
            aliceHydraNode.withExposedPorts(new Integer[]{4001}).withAccessToHost(true).withNetwork(network).withNetworkAliases(new String[]{containerName}).withClasspathResourceMapping("/keys", "/keys", BindMode.READ_ONLY).withVolumesFrom(cardanoContainer, BindMode.READ_WRITE).waitingFor(Wait.forLogMessage((String)".+bob-hydra-node.+PeerConnected.+", (int)1).withStartupTimeout(Duration.ofMinutes(10L))).withEnv(Map.of("HYDRA_SCRIPTS_TX_ID", scriptsTxId)).withCreateContainerCmdModifier(cmd -> cmd.withName(containerName).withHostName(containerName).withAliases(new String[]{containerName})).withCommand(new String[]{"--node-id", containerName, "--api-host", "0.0.0.0", "--monitoring-port", "6001", "--port", "5001", "--api-port", "4001", "--peer", "bob-hydra-node:5001", "--host", "0.0.0.0", "--hydra-scripts-tx-id", scriptsTxId, "--hydra-signing-key", "/keys/alice.sk", "--hydra-verification-key", "/keys/bob.vk", "--cardano-signing-key", "/devnet/credentials/alice.sk", "--cardano-verification-key", "/devnet/credentials/bob.vk", "--ledger-genesis", "/devnet/genesis-shelley.json", "--ledger-protocol-parameters", "/devnet/protocol-parameters.json", "--persistence-dir", "/tmp/alice-hydra-node_db" + System.currentTimeMillis(), "--testnet-magic", "42", "--node-socket", "/devnet/node.socket"});
            if (this.hydraLogging) {
                aliceHydraNode.withLogConsumer((Consumer)new Slf4jLogConsumer(log).withSeparateOutputStreams());
            }
            GenericContainer genericContainer = aliceHydraNode;
            return genericContainer;
        }
    }

    protected GenericContainer<?> createBobHydraNode(GenericContainer<?> cardanoContainer, String scriptsTxId, Network network) {
        String containerName = "bob-hydra-node";
        try (GenericContainer bobHydraNode = new GenericContainer(INPUT_OUTPUT_HYDRA_NODE);){
            bobHydraNode.withExposedPorts(new Integer[]{4001}).withAccessToHost(true).withNetwork(network).withNetworkAliases(new String[]{containerName}).withVolumesFrom(cardanoContainer, BindMode.READ_WRITE).withClasspathResourceMapping("/keys", "/keys", BindMode.READ_ONLY).waitingFor(Wait.forLogMessage((String)".+bob-hydra-node.+PeerConnected.+", (int)1).withStartupTimeout(Duration.ofMinutes(10L))).withEnv(Map.of("HYDRA_SCRIPTS_TX_ID", scriptsTxId)).withCreateContainerCmdModifier(cmd -> cmd.withName(containerName).withHostName(containerName).withAliases(new String[]{containerName})).withCommand(new String[]{"--node-id", containerName, "--api-host", "0.0.0.0", "--monitoring-port", "6001", "--api-port", "4001", "--port", "5001", "--peer", "alice-hydra-node:5001", "--host", "0.0.0.0", "--hydra-scripts-tx-id", scriptsTxId, "--hydra-signing-key", "/keys/bob.sk", "--hydra-verification-key", "/keys/alice.vk", "--cardano-signing-key", "/devnet/credentials/bob.sk", "--cardano-verification-key", "/devnet/credentials/alice.vk", "--ledger-genesis", "/devnet/genesis-shelley.json", "--ledger-protocol-parameters", "/devnet/protocol-parameters.json", "--persistence-dir", "/tmp/bob-hydra-node_db" + System.currentTimeMillis(), "--testnet-magic", "42", "--node-socket", "/devnet/node.socket"});
            if (this.hydraLogging) {
                bobHydraNode.withLogConsumer((Consumer)new Slf4jLogConsumer(log).withSeparateOutputStreams());
            }
            GenericContainer genericContainer = bobHydraNode;
            return genericContainer;
        }
    }

    protected void seedActor(GenericContainer<?> cardanoNodeContainer, String faucetAddress, String actor, int adaAmount, boolean marker) throws IOException, InterruptedException {
        Container.ExecResult txBuildExecResult;
        log.info(String.format("Seeding a UTXO from faucet to %s with %d ADA, faucet address:%s ", actor, adaAmount, faucetAddress));
        long actorLovelaces = (long)adaAmount * 1000000L;
        Container.ExecResult faucetUTxOExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "query", "utxo", "--testnet-magic", "42", "--address", faucetAddress, "--out-file", "/dev/stdout"});
        if (faucetUTxOExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to find faucet's UTxO, error:" + faucetUTxOExecResult);
        }
        String faucetUTxO = (String)objectMapper.readTree(faucetUTxOExecResult.getStdout()).fieldNames().next();
        log.info("Faucet utxo:{}", (Object)faucetUTxO);
        log.info("Fetching address for actor:{}...", (Object)actor);
        Container.ExecResult actorAddressExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "address", "build", "--testnet-magic", "42", "--payment-verification-key-file", String.format("/devnet/credentials/%s.vk", actor)});
        if (actorAddressExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to find actor's address, error:" + actorAddressExecResult);
        }
        String actorAddr = actorAddressExecResult.getStdout();
        log.info("Actor's:{} address:{}", (Object)actor, (Object)actorAddr);
        log.info("Seeding actor:{}...", (Object)actor);
        Container.ExecResult execResult = txBuildExecResult = marker ? cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "transaction", "build", "--testnet-magic", "42", "--babbage-era", "--cardano-mode", "--change-address", faucetAddress, "--tx-in", faucetUTxO, "--tx-out", String.format("%s+%d", actorAddr, actorLovelaces), "--tx-out-datum-hash", MARKER_DATUM_HASH, "--out-file", String.format("/tmp/seed-%s.draft", actor)}) : cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "transaction", "build", "--testnet-magic", "42", "--babbage-era", "--cardano-mode", "--change-address", faucetAddress, "--tx-in", faucetUTxO, "--tx-out", String.format("%s+%d", actorAddr, actorLovelaces), "--out-file", String.format("/tmp/seed-%s.draft", actor)});
        if (txBuildExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to build transaction for actor, error:" + txBuildExecResult);
        }
        Container.ExecResult txSignExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "transaction", "sign", "--testnet-magic", "42", "--tx-body-file", String.format("/tmp/seed-%s.draft", actor), "--signing-key-file", "/devnet/credentials/faucet.sk", "--out-file", String.format("/tmp/seed-%s.signed", actor)});
        if (txSignExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to sign transaction for actor, error:" + txSignExecResult);
        }
        Container.ExecResult txSeedIdExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "transaction", "txid", "--tx-file", String.format("/tmp/seed-%s.signed", actor)});
        if (txSeedIdExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to get transaction id for actor, error:" + txSeedIdExecResult);
        }
        String txSeedIn = txSeedIdExecResult.getStdout().replace("\n", "") + "#0";
        Container.ExecResult txSubmitExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "transaction", "submit", "--testnet-magic", "42", "--tx-file", String.format("/tmp/seed-%s.signed", actor)});
        if (txSubmitExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to submit transaction, error:" + txSubmitExecResult);
        }
        for (int i = 0; i < 10; ++i) {
            log.info("Checking if actor:{} got ADA.", (Object)actor);
            Thread.sleep(500L);
            Container.ExecResult txQueryExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "query", "utxo", "--testnet-magic", "42", "--tx-in", txSeedIn, "--out-file", "/dev/stdout"});
            boolean foundTx = objectMapper.readTree(txQueryExecResult.getStdout()).has(txSeedIn);
            if (foundTx) {
                log.info("Actor:{} got ADA, sitting in the utxo:{}", (Object)actor, (Object)txSeedIn);
                break;
            }
            log.info("Actor:{} didn't get ADA, didn't find transaction's utxo: {} yet, will try again in 500ms...", (Object)actor, (Object)txSeedIn);
        }
    }

    private static String getFaucetDetails(GenericContainer<?> cardanoNodeContainer) throws IOException, InterruptedException {
        log.info("Fetching address for faucet...");
        Container.ExecResult fauceutAddressExecResult = cardanoNodeContainer.execInContainer(new String[]{"cardano-cli", "address", "build", "--testnet-magic", "42", "--payment-verification-key-file", "/devnet/credentials/faucet.vk"});
        if (fauceutAddressExecResult.getExitCode() != 0) {
            throw new RuntimeException("Unable to find faucet's address, error:" + fauceutAddressExecResult);
        }
        return fauceutAddressExecResult.getStdout();
    }

    protected void seedActors(GenericContainer<?> cardanoContainer) throws IOException, InterruptedException {
        String faucetAddr = HydraDevNetwork.getFaucetDetails(cardanoContainer);
        log.info("Faucet address:{}", (Object)faucetAddr);
        for (Map.Entry<String, Integer> entry : this.initialFunds.entrySet()) {
            String actor = entry.getKey();
            Integer ada = entry.getValue();
            this.seedActor(cardanoContainer, faucetAddr, actor, ada, false);
            this.seedActor(cardanoContainer, faucetAddr, actor, ada, false);
        }
        this.seedActor(cardanoContainer, faucetAddr, "alice", 100, true);
        this.seedActor(cardanoContainer, faucetAddr, "bob", 100, true);
    }
}

