package org.cardanofoundation.lob.app.support.spring_web;

import lombok.Getter;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationConverter;
import org.springframework.security.web.SecurityFilterChain;

import org.cardanofoundation.lob.app.support.security.KeycloakRoleConverter;

@Configuration("securityConfig")
@EnableMethodSecurity
@Getter
@ConditionalOnProperty(value = "keycloak.enabled", havingValue = "true")
public class SecurityConfig {

    @Value("${keycloak.cert-url}")
    private String certUrl;

    @Value("${keycloak.roles.manager}")
    private String managerRole;

    @Value("${keycloak.roles.auditor}")
    private String auditorRole;

    @Value("${keycloak.roles.accountant}")
    private String accountantRole;

    @Value("${keycloak.roles.admin}")
    private String adminRole;

    @Value("${security.csp-policy}")
    private String cspPolicy;

    @Bean
    public JwtDecoder jwtDecoder() {
        return NimbusJwtDecoder.withJwkSetUri(certUrl)
                .build();
    }

    @Bean
    public SecurityFilterChain securityFilterChain(HttpSecurity http) throws Exception {
        http
                .headers(headers -> headers
                        .contentSecurityPolicy(csp -> csp.policyDirectives(cspPolicy)))
                .csrf(csrf -> csrf.disable())
                .authorizeHttpRequests(authorize -> authorize
                        .anyRequest().permitAll() // allowing all access
                )
                .oauth2ResourceServer(oauth2 -> oauth2.jwt(jwt ->
                        jwt.jwtAuthenticationConverter(jwtAuthenticationConverter())
                ));
        return http.build();
    }

    private JwtAuthenticationConverter jwtAuthenticationConverter() {
        JwtAuthenticationConverter converter = new JwtAuthenticationConverter();
        converter.setJwtGrantedAuthoritiesConverter(new KeycloakRoleConverter());
        return converter;
    }
}
