package org.boomevents.hubspot.domain.task

import org.boomevents.hubspot.Client
import org.boomevents.hubspot.ClientRequestCatalog
import org.boomevents.hubspot.domain.task.exceptions.TaskNotFoundException
import org.boomevents.hubspot.model.http.RequestMethod
import org.boomevents.hubspot.model.http.Requester
import org.boomevents.hubspot.model.http.exceptions.HttpRequestException
import org.boomevents.hubspot.model.mapper.Mapper
import java.math.BigInteger

class TaskClient(private val hubSpotClient: Client) {

    fun createTask(request: TaskCreateRequest): Task {
        val response = Requester.requestJson(hubSpotClient, RequestMethod.POST, ClientRequestCatalog.V3.TASKS, emptyMap(), request)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            throw RuntimeException(response.statusText)
        }
    }

    @Throws(
        TaskNotFoundException::class,
        HttpRequestException::class
    )
    fun findTask(taskId: BigInteger): Task {
        val requestUrl = ClientRequestCatalog.V3.TASKS_DETAIL.replace(
            "{taskId}", taskId.toString()
        )

        val response = Requester.requestJson(hubSpotClient, RequestMethod.GET, requestUrl)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            when (response.status) {
                404 -> throw TaskNotFoundException(taskId)
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }

    fun changeTask(taskId: BigInteger, request: TaskUpdateRequest): Task {
        val requestUrl = ClientRequestCatalog.V3.TASKS_DETAIL.replace(
            "{taskId}", taskId.toString()
        )

        val response = Requester.requestJson(hubSpotClient, RequestMethod.PATCH, requestUrl, emptyMap(), request)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            when (response.status) {
                404 -> throw TaskNotFoundException(taskId)
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }

    @Throws(
        HttpRequestException::class
    )
    fun removeTask(taskId: BigInteger) {
        val requestUrl = ClientRequestCatalog.V3.TASKS_DETAIL.replace(
            "{taskId}", taskId.toString()
        )

        // Unknown task returns HTTP code 204
        val response = Requester.requestVoid(hubSpotClient, RequestMethod.DELETE, requestUrl)

        if (!response.isSuccess) {
            when (response.status) {
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }
}
