package org.boomevents.hubspot.domain.company

import org.boomevents.hubspot.Client
import org.boomevents.hubspot.ClientRequestCatalog
import org.boomevents.hubspot.domain.company.exceptions.CompanyNotFoundException
import org.boomevents.hubspot.model.http.RequestMethod
import org.boomevents.hubspot.model.http.Requester
import org.boomevents.hubspot.model.http.exceptions.HttpRequestException
import org.boomevents.hubspot.model.mapper.Mapper
import java.math.BigInteger

class CompanyClient(private val hubSpotClient: Client) {

    fun <P> createCompany(request: CompanyRequest<P>): Company {
        val response = Requester.requestJson(hubSpotClient, RequestMethod.POST, ClientRequestCatalog.V3.COMPANIES, emptyMap(), request)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            throw RuntimeException(response.statusText)
        }
    }


    /**
     * Retrieves the details of a specific company by its ID from the HubSpot API.
     *
     * @param companyId The unique identifier of the company to be retrieved.
     * @param properties An optional list of property names to be included in the response.
     *                   If provided, these properties will be appended as query parameters
     *                   to the request URL to specify which fields to retrieve.
     * @return A Company object containing the details of the requested company.
     * @throws CompanyNotFoundException if the company with the specified ID is not found.
     * @throws HttpRequestException if the request to the HubSpot API fails for any other reason.
     */
    
    @Throws(
        CompanyNotFoundException::class,
        HttpRequestException::class
    )
    fun findCompany(companyId: BigInteger, properties: List<String>? = null): Company {
        val baseUrl = ClientRequestCatalog.V3.COMPANIES_DETAIL.replace(
            "{companyId}", companyId.toString()
        )

        val requestUrl = if (properties != null && properties.isNotEmpty()) {
            "$baseUrl?properties=${properties.joinToString(",")}"
        } else {
            baseUrl
        }

        val response = Requester.requestJson(hubSpotClient, RequestMethod.GET, requestUrl)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            when (response.status) {
                404 -> throw CompanyNotFoundException(companyId)
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }


    fun <P> changeCompany(companyId: BigInteger, request: CompanyRequest<P>): Company {
        val requestUrl = ClientRequestCatalog.V3.COMPANIES_DETAIL.replace(
            "{companyId}", companyId.toString()
        )

        val response = Requester.requestJson(hubSpotClient, RequestMethod.PATCH, requestUrl, emptyMap(), request)

        if (response.isSuccess) {
            return Mapper.mapToObject(response.body)
        } else {
            when (response.status) {
                404 -> throw CompanyNotFoundException(companyId)
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }

    @Throws(
        HttpRequestException::class
    )
    fun removeCompany(companyId: BigInteger) {
        val requestUrl = ClientRequestCatalog.V3.COMPANIES_DETAIL.replace(
            "{companyId}", companyId.toString()
        )

        // Unknown company returns HTTP code 204
        val response = Requester.requestVoid(hubSpotClient, RequestMethod.DELETE, requestUrl)

        if (!response.isSuccess) {
            when (response.status) {
                else -> throw HttpRequestException(response.status, response.statusText)
            }
        }
    }
}
