/*
 * Decompiled with CFR 0.152.
 */
package herddb.file;

import herddb.log.CommitLog;
import herddb.log.CommitLogResult;
import herddb.log.LogEntry;
import herddb.log.LogNotAvailableException;
import herddb.log.LogSequenceNumber;
import herddb.utils.ExtendedDataInputStream;
import herddb.utils.ExtendedDataOutputStream;
import herddb.utils.FileUtils;
import herddb.utils.ODirectFileOutputStream;
import herddb.utils.OpenFileUtils;
import herddb.utils.SimpleBufferedOutputStream;
import herddb.utils.SystemProperties;
import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;

public class FileCommitLog
extends CommitLog {
    private static final Logger LOGGER = Logger.getLogger(FileCommitLog.class.getName());
    private Path logDirectory;
    private final String tableSpaceName;
    private long currentLedgerId = 0L;
    private LogSequenceNumber recoveredLogSequence;
    private long maxLogFileSize = 0x100000L;
    private long writtenBytes = 0L;
    private volatile CommitFileWriter writer;
    private Thread spool;
    private final OpStatsLogger statsFsyncTime;
    private final AtomicInteger queueSize = new AtomicInteger();
    private final AtomicInteger pendingEntries = new AtomicInteger();
    private final OpStatsLogger statsEntryLatency;
    private final OpStatsLogger statsEntrySyncLatency;
    private final OpStatsLogger syncSize;
    private final OpStatsLogger syncBytes;
    private final Counter deferredSyncs;
    private final Counter newfiles;
    private final ExecutorService fsyncThreadPool;
    private final Consumer<FileCommitLog> onClose;
    private static final int WRITE_QUEUE_SIZE = SystemProperties.getIntSystemProperty((String)"herddb.file.writequeuesize", (int)10000000);
    private final BlockingQueue<LogEntryHolderFuture> writeQueue = new LinkedBlockingQueue<LogEntryHolderFuture>(WRITE_QUEUE_SIZE);
    private final int maxUnsyncedBatchSize;
    private final int maxUnsyncedBatchBytes;
    private final long maxSyncTime;
    private final boolean requireSync;
    private final boolean enableO_DIRECT;
    public static final String LOGFILEEXTENSION = ".txlog";
    private volatile boolean closed = false;
    private volatile boolean failed = false;
    private volatile boolean needsSync = false;
    static final byte ZERO_PADDING = 0;
    static final byte ENTRY_START = 13;
    static final byte ENTRY_END = 25;

    void backgroundSync() {
        if (this.needsSync) {
            try {
                this.getWriter().sync(true);
                this.deferredSyncs.inc();
                this.needsSync = false;
            }
            catch (Throwable t) {
                LOGGER.log(Level.SEVERE, "error background fsync on " + this.logDirectory, t);
            }
        }
    }

    private void openNewLedger() throws LogNotAvailableException {
        try {
            if (this.writer != null) {
                LOGGER.log(Level.FINEST, "closing actual file {0}", this.writer.filename);
                this.writer.close();
            }
            this.writer = new CommitFileWriter(++this.currentLedgerId, -1L);
            this.newfiles.inc();
        }
        catch (IOException err) {
            throw new LogNotAvailableException(err);
        }
    }

    public FileCommitLog(Path logDirectory, String tableSpaceName, long maxLogFileSize, ExecutorService fsyncThreadPool, StatsLogger statslogger, Consumer<FileCommitLog> onClose, int maxUnsynchedBatchSize, int maxUnsynchedBatchBytes, int maxSyncTime, boolean requireSync, boolean enableO_DIRECT) {
        this.maxUnsyncedBatchSize = maxUnsynchedBatchSize;
        this.maxUnsyncedBatchBytes = maxUnsynchedBatchBytes;
        this.maxSyncTime = TimeUnit.MILLISECONDS.toNanos(maxSyncTime);
        this.requireSync = requireSync;
        this.enableO_DIRECT = enableO_DIRECT && OpenFileUtils.isO_DIRECT_Supported();
        this.onClose = onClose;
        this.maxLogFileSize = maxLogFileSize;
        this.tableSpaceName = tableSpaceName;
        this.logDirectory = logDirectory.toAbsolutePath();
        this.spool = new Thread((Runnable)new SpoolTask(), "commitlog-" + tableSpaceName);
        this.spool.setDaemon(true);
        this.statsFsyncTime = statslogger.getOpStatsLogger("fsync");
        this.statsEntryLatency = statslogger.getOpStatsLogger("entryLatency");
        this.statsEntrySyncLatency = statslogger.getOpStatsLogger("entrySyncLatency");
        this.syncSize = statslogger.getOpStatsLogger("syncBatchSize");
        this.syncBytes = statslogger.getOpStatsLogger("syncBatchBytes");
        this.deferredSyncs = statslogger.getCounter("deferredSyncs");
        this.newfiles = statslogger.getCounter("newfiles");
        statslogger.registerGauge("queuesize", (Gauge)new Gauge<Integer>(){

            public Integer getDefaultValue() {
                return 0;
            }

            public Integer getSample() {
                return FileCommitLog.this.queueSize.get();
            }
        });
        statslogger.registerGauge("pendingentries", (Gauge)new Gauge<Integer>(){

            public Integer getDefaultValue() {
                return 0;
            }

            public Integer getSample() {
                return FileCommitLog.this.pendingEntries.get();
            }
        });
        this.fsyncThreadPool = fsyncThreadPool;
        LOGGER.log(Level.FINE, "tablespace {2}, logdirectory: {0}, maxLogFileSize {1} bytes", new Object[]{logDirectory, maxLogFileSize, tableSpaceName});
    }

    private int writeEntry(LogEntryHolderFuture entry) {
        try {
            CommitFileWriter writer = this.writer;
            if (writer == null) {
                throw new IOException("not yet writable");
            }
            long newSequenceNumber = ++writer.sequenceNumber;
            int written = writer.writeEntry(newSequenceNumber, entry.entry);
            if (this.writtenBytes > this.maxLogFileSize) {
                this.openNewLedger();
            }
            entry.done(new LogSequenceNumber(writer.ledgerId, newSequenceNumber));
            this.statsEntryLatency.registerSuccessfulEvent(System.currentTimeMillis() - entry.timestamp, TimeUnit.MILLISECONDS);
            return written;
        }
        catch (LogNotAvailableException | IOException err) {
            entry.error((Throwable)err);
            this.statsEntryLatency.registerFailedEvent(System.currentTimeMillis() - entry.timestamp, TimeUnit.MILLISECONDS);
            return 0;
        }
    }

    private void flush() throws IOException {
        CommitFileWriter _writer = this.writer;
        if (_writer == null) {
            return;
        }
        _writer.flush();
    }

    private void synch() throws IOException {
        CommitFileWriter _writer = this.writer;
        if (_writer == null) {
            return;
        }
        _writer.sync();
    }

    public int getQueueSize() {
        return this.queueSize.get();
    }

    @Override
    public CommitLogResult log(LogEntry edit, boolean sync) throws LogNotAvailableException {
        if (this.failed) {
            throw new LogNotAvailableException("file commit log is failed");
        }
        boolean hasListeners = this.isHasListeners();
        if (hasListeners) {
            sync = true;
        }
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, "log {0}", edit);
        }
        LogEntryHolderFuture future = new LogEntryHolderFuture(edit, sync);
        try {
            this.queueSize.incrementAndGet();
            this.pendingEntries.incrementAndGet();
            this.writeQueue.put(future);
            if (!sync) {
                return new CommitLogResult(future.ack, false, false);
            }
            if (hasListeners) {
                future.ack.thenAccept(pos -> this.notifyListeners((LogSequenceNumber)pos, edit));
            }
            return new CommitLogResult(future.ack, false, true);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new LogNotAvailableException(err);
        }
    }

    @Override
    public void recovery(LogSequenceNumber snapshotSequenceNumber, BiConsumer<LogSequenceNumber, LogEntry> consumer, boolean fencing) throws LogNotAvailableException {
        LOGGER.log(Level.INFO, "recovery {1}, snapshotSequenceNumber: {0}", new Object[]{snapshotSequenceNumber, this.tableSpaceName});
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.logDirectory);){
            ArrayList<Path> names = new ArrayList<Path>();
            for (Path path : stream) {
                if (!Files.isRegularFile(path, new LinkOption[0]) || !(path.getFileName() + "").endsWith(LOGFILEEXTENSION)) continue;
                names.add(path);
            }
            names.sort(Comparator.comparing(Path::toString));
            long offset = -1L;
            for (Path p : names) {
                LOGGER.log(Level.INFO, "tablespace {1}, logfile is {0}", new Object[]{p.toAbsolutePath(), this.tableSpaceName});
                String name = (p.getFileName() + "").replace(LOGFILEEXTENSION, "");
                long ledgerId = Long.parseLong(name, 16);
                this.currentLedgerId = Math.max(this.currentLedgerId, ledgerId);
                offset = -1L;
                try (CommitFileReader reader = new CommitFileReader(this.logDirectory, ledgerId);){
                    LogEntryWithSequenceNumber n = reader.nextEntry();
                    while (n != null) {
                        offset = n.logSequenceNumber.offset;
                        if (n.logSequenceNumber.after(snapshotSequenceNumber)) {
                            LOGGER.log(Level.FINE, "RECOVER ENTRY {0}, {1}", new Object[]{n.logSequenceNumber, n.entry});
                            consumer.accept(n.logSequenceNumber, n.entry);
                        } else {
                            LOGGER.log(Level.FINE, "SKIP ENTRY {0}, {1}", new Object[]{n.logSequenceNumber, n.entry});
                        }
                        n = reader.nextEntry();
                    }
                }
            }
            this.recoveredLogSequence = new LogSequenceNumber(this.currentLedgerId, offset);
            LOGGER.log(Level.INFO, "Tablespace {1}, max ledgerId is {0}", new Object[]{this.currentLedgerId, this.tableSpaceName});
        }
        catch (IOException | RuntimeException err) {
            this.failed = true;
            throw new LogNotAvailableException(err);
        }
    }

    @Override
    public void dropOldLedgers(LogSequenceNumber lastCheckPointSequenceNumber) throws LogNotAvailableException {
        LOGGER.log(Level.SEVERE, "dropOldLedgers {2} lastCheckPointSequenceNumber: {0}, currentLedgerId: {1}", new Object[]{lastCheckPointSequenceNumber, this.currentLedgerId, this.tableSpaceName});
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.logDirectory);){
            ArrayList<Path> names = new ArrayList<Path>();
            for (Path path : stream) {
                if (!Files.isRegularFile(path, new LinkOption[0]) || !(path.getFileName() + "").endsWith(LOGFILEEXTENSION)) continue;
                names.add(path);
            }
            names.sort(Comparator.comparing(Path::toString));
            Path last = names.isEmpty() ? null : (Path)names.get(names.size() - 1);
            int count = 0;
            long ledgerLimit = Math.min(lastCheckPointSequenceNumber.ledgerId, this.currentLedgerId);
            for (Path path : names) {
                boolean lastFile = path.equals(last);
                String name = (path.getFileName() + "").replace(LOGFILEEXTENSION, "");
                try {
                    long ledgerId = Long.parseLong(name, 16);
                    if (lastFile || ledgerId >= ledgerLimit) continue;
                    LOGGER.log(Level.SEVERE, "deleting logfile {0} for ledger {1}", new Object[]{path.toAbsolutePath(), ledgerId});
                    try {
                        Files.delete(path);
                    }
                    catch (IOException errorDelete) {
                        LOGGER.log(Level.SEVERE, "fatal error while deleting file " + path, errorDelete);
                        throw new LogNotAvailableException(errorDelete);
                    }
                    ++count;
                }
                catch (NumberFormatException numberFormatException) {}
            }
            LOGGER.log(Level.SEVERE, "Deleted logfiles: {0}", count);
        }
        catch (IOException err) {
            this.failed = true;
            throw new LogNotAvailableException(err);
        }
    }

    @Override
    public void startWriting(int expectedReplicaCount) throws LogNotAvailableException {
        this.ensureDirectories();
        this.spool.start();
    }

    private void ensureDirectories() throws LogNotAvailableException {
        try {
            if (!Files.isDirectory(this.logDirectory, new LinkOption[0])) {
                LOGGER.log(Level.INFO, "directory {0} does not exist. creating", this.logDirectory);
                Files.createDirectories(this.logDirectory, new FileAttribute[0]);
            }
        }
        catch (IOException err) {
            this.failed = true;
            throw new LogNotAvailableException(err);
        }
    }

    @Override
    public void close() throws LogNotAvailableException {
        this.closed = true;
        this.onClose.accept(this);
        try {
            if (this.writeQueue.isEmpty()) {
                this.writeQueue.add(new LogEntryHolderFuture(null, false));
            }
            this.spool.join();
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new LogNotAvailableException(err);
        }
        if (this.writer != null) {
            this.writer.close();
        }
    }

    @Override
    public boolean isClosed() {
        return this.closed;
    }

    @Override
    public boolean isFailed() {
        return this.failed;
    }

    @Override
    public void clear() throws LogNotAvailableException {
        try {
            FileUtils.cleanDirectory((Path)this.logDirectory);
        }
        catch (IOException err) {
            throw new LogNotAvailableException(err);
        }
    }

    CommitFileWriter getWriter() {
        return this.writer;
    }

    @Override
    public LogSequenceNumber getLastSequenceNumber() {
        CommitFileWriter _writer = this.writer;
        if (_writer == null) {
            return this.recoveredLogSequence == null ? new LogSequenceNumber(this.currentLedgerId, -1L) : this.recoveredLogSequence;
        }
        return new LogSequenceNumber(_writer.ledgerId, _writer.sequenceNumber);
    }

    private class LogEntryHolderFuture {
        final CompletableFuture<LogSequenceNumber> ack = new CompletableFuture();
        final LogEntry entry;
        final long timestamp;
        LogSequenceNumber sequenceNumber;
        Throwable error;
        final boolean sync;

        public LogEntryHolderFuture(LogEntry entry, boolean synch) {
            if (entry == null) {
                this.entry = null;
                this.timestamp = System.currentTimeMillis();
            } else {
                this.entry = entry;
                this.timestamp = entry.timestamp;
            }
            this.sync = synch;
        }

        public void error(Throwable error) {
            this.error = error;
            if (!this.sync) {
                this.syncDone();
            }
        }

        public void done(LogSequenceNumber sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            if (!this.sync) {
                this.syncDone();
            }
        }

        private void syncDone() {
            if (this.sequenceNumber == null && this.error == null) {
                throw new IllegalStateException();
            }
            FileCommitLog.this.pendingEntries.decrementAndGet();
            if (this.error != null) {
                this.ack.completeExceptionally(this.error);
            } else {
                this.ack.complete(this.sequenceNumber);
            }
        }
    }

    class CommitFileWriter
    implements AutoCloseable {
        final long ledgerId;
        long sequenceNumber;
        final Path filename;
        final FileChannel channel;
        final ExtendedDataOutputStream out;
        volatile boolean writerClosed;

        private CommitFileWriter(long ledgerId, long sequenceNumber) throws IOException {
            this.ledgerId = ledgerId;
            this.sequenceNumber = sequenceNumber;
            this.filename = FileCommitLog.this.logDirectory.resolve(String.format("%016x", ledgerId) + FileCommitLog.LOGFILEEXTENSION).toAbsolutePath();
            if (FileCommitLog.this.enableO_DIRECT) {
                Files.createFile(this.filename, new FileAttribute[0]);
                LOGGER.log(Level.FINE, "opening (O_DIRECT) new file {0} for tablespace {1}", new Object[]{this.filename, FileCommitLog.this.tableSpaceName});
                ODirectFileOutputStream oo = new ODirectFileOutputStream(this.filename);
                this.channel = oo.getFc();
                this.out = new ExtendedDataOutputStream((OutputStream)oo);
            } else {
                LOGGER.log(Level.FINE, "opening (no O_DIRECT) new file {0} for tablespace {1}", new Object[]{this.filename, FileCommitLog.this.tableSpaceName});
                this.channel = FileChannel.open(this.filename, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
                this.out = new ExtendedDataOutputStream((OutputStream)new SimpleBufferedOutputStream(Channels.newOutputStream(this.channel)));
            }
            FileCommitLog.this.writtenBytes = 0L;
        }

        private int writeEntry(long seqnumber, LogEntry entry) throws IOException {
            this.out.writeByte(13);
            this.out.writeLong(seqnumber);
            int written = entry.serialize(this.out);
            this.out.writeByte(25);
            int entrySize = 9 + written + 1;
            FileCommitLog.this.writtenBytes += entrySize;
            if (!FileCommitLog.this.requireSync) {
                FileCommitLog.this.needsSync = true;
            }
            return entrySize;
        }

        public void flush() throws IOException {
            this.out.flush();
        }

        public void sync() throws IOException {
            this.sync(false);
        }

        public void sync(boolean force) throws IOException {
            long now;
            block3: {
                if (!force && !FileCommitLog.this.requireSync) {
                    return;
                }
                now = System.nanoTime();
                try {
                    this.channel.force(false);
                }
                catch (ClosedChannelException closed) {
                    if (this.writerClosed) break block3;
                    throw closed;
                }
            }
            long time = System.nanoTime() - now;
            FileCommitLog.this.statsFsyncTime.registerSuccessfulEvent(time, TimeUnit.NANOSECONDS);
        }

        @Override
        public void close() throws LogNotAvailableException {
            try {
                try {
                    this.out.flush();
                    this.sync();
                }
                catch (IOException err) {
                    throw new LogNotAvailableException(err);
                }
            }
            finally {
                try {
                    this.writerClosed = true;
                    this.out.close();
                    this.channel.close();
                }
                catch (IOException err) {
                    throw new LogNotAvailableException(err);
                }
            }
        }
    }

    private class SpoolTask
    implements Runnable {
        private SpoolTask() {
        }

        @Override
        public void run() {
            try {
                FileCommitLog.this.openNewLedger();
                ArrayList<LogEntryHolderFuture> syncNeeded = new ArrayList<LogEntryHolderFuture>();
                long unsyncedBytes = 0L;
                int unsyncedCount = 0;
                while (!FileCommitLog.this.closed || !FileCommitLog.this.writeQueue.isEmpty()) {
                    LogEntryHolderFuture entry = (LogEntryHolderFuture)FileCommitLog.this.writeQueue.poll(FileCommitLog.this.maxSyncTime, TimeUnit.NANOSECONDS);
                    boolean timedOut = false;
                    if (entry != null) {
                        if (entry.entry == null) break;
                        FileCommitLog.this.queueSize.decrementAndGet();
                        int size = FileCommitLog.this.writeEntry(entry);
                        ++unsyncedCount;
                        unsyncedBytes += (long)size;
                        if (entry.sync) {
                            syncNeeded.add(entry);
                        }
                    } else {
                        timedOut = true;
                    }
                    if (!timedOut && unsyncedCount < FileCommitLog.this.maxUnsyncedBatchSize && unsyncedBytes < (long)FileCommitLog.this.maxUnsyncedBatchBytes || unsyncedCount <= 0) continue;
                    FileCommitLog.this.flush();
                    if (!syncNeeded.isEmpty()) {
                        SyncTask syncTask = new SyncTask(syncNeeded, unsyncedCount, unsyncedBytes);
                        syncNeeded = new ArrayList();
                        FileCommitLog.this.fsyncThreadPool.submit(syncTask);
                    }
                    unsyncedCount = 0;
                    unsyncedBytes = 0L;
                }
                if (unsyncedCount > 0) {
                    LOGGER.log(Level.INFO, "flushing last {0} entries", unsyncedCount);
                    FileCommitLog.this.flush();
                    if (!syncNeeded.isEmpty()) {
                        LOGGER.log(Level.INFO, "synching last {0} entries", unsyncedCount);
                        SyncTask syncTask = new SyncTask(syncNeeded, unsyncedCount, unsyncedBytes);
                        syncTask.run();
                    }
                }
            }
            catch (LogNotAvailableException | IOException | InterruptedException t) {
                FileCommitLog.this.failed = true;
                LOGGER.log(Level.SEVERE, "general commit log failure on " + FileCommitLog.this.logDirectory, (Throwable)t);
            }
        }
    }

    public static class CommitFileReader
    implements AutoCloseable {
        final ExtendedDataInputStream in;
        final long ledgerId;

        private CommitFileReader(ExtendedDataInputStream in, long ledgerId) {
            this.in = in;
            this.ledgerId = ledgerId;
        }

        public static CommitFileReader openForDescribeRawfile(Path filename) throws IOException {
            long ledgerId;
            ExtendedDataInputStream in = new ExtendedDataInputStream((InputStream)new BufferedInputStream(Files.newInputStream(filename, StandardOpenOption.READ), 65536));
            String lastPath = (filename.getFileName() + "").replace(FileCommitLog.LOGFILEEXTENSION, "");
            try {
                ledgerId = Long.valueOf(lastPath, 16);
            }
            catch (NumberFormatException err) {
                ledgerId = 0L;
            }
            return new CommitFileReader(in, ledgerId);
        }

        private CommitFileReader(Path logDirectory, long ledgerId) throws IOException {
            this.ledgerId = ledgerId;
            Path filename = logDirectory.resolve(String.format("%016x", ledgerId) + FileCommitLog.LOGFILEEXTENSION);
            this.in = new ExtendedDataInputStream((InputStream)new BufferedInputStream(Files.newInputStream(filename, StandardOpenOption.READ), 65536));
        }

        public LogEntryWithSequenceNumber nextEntry() throws IOException {
            try {
                byte entryStart;
                try {
                    entryStart = this.in.readByte();
                }
                catch (EOFException completeFileFinished) {
                    return null;
                }
                while (entryStart == 0) {
                    try {
                        entryStart = this.in.readByte();
                    }
                    catch (EOFException completeFileFinished) {
                        return null;
                    }
                }
                if (entryStart != 13) {
                    throw new IOException("corrupted txlog file");
                }
                long seqNumber = this.in.readLong();
                LogEntry edit = LogEntry.deserialize(this.in);
                byte entryEnd = this.in.readByte();
                if (entryEnd != 25) {
                    throw new IOException("corrupted txlog file, found a " + entryEnd + " instead of magic '" + 25 + "'");
                }
                return new LogEntryWithSequenceNumber(new LogSequenceNumber(this.ledgerId, seqNumber), edit);
            }
            catch (EOFException truncatedLog) {
                LOGGER.log(Level.SEVERE, "found unfinished entry in file " + this.ledgerId + ". entry was not acked. ignoring " + truncatedLog);
                return null;
            }
        }

        @Override
        public void close() throws IOException {
            this.in.close();
        }
    }

    public static final class LogEntryWithSequenceNumber {
        public final LogSequenceNumber logSequenceNumber;
        public final LogEntry entry;

        public LogEntryWithSequenceNumber(LogSequenceNumber logSequenceNumber, LogEntry entry) {
            this.logSequenceNumber = logSequenceNumber;
            this.entry = entry;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("LogEntryWithSequenceNumber [logSequenceNumber=").append(this.logSequenceNumber).append(", entry=").append(this.entry).append("]");
            return builder.toString();
        }
    }

    private class SyncTask
    implements Runnable {
        private List<LogEntryHolderFuture> syncNeeded;
        private final int unsyncedCount;
        private final long unsyncedBytes;

        public SyncTask(List<LogEntryHolderFuture> syncNeeded, int unsyncedCount, long unsyncedBytes) {
            this.syncNeeded = syncNeeded;
            this.unsyncedBytes = unsyncedBytes;
            this.unsyncedCount = unsyncedCount;
        }

        @Override
        public void run() {
            long now = System.currentTimeMillis();
            try {
                FileCommitLog.this.synch();
                FileCommitLog.this.syncSize.registerSuccessfulValue((long)this.unsyncedCount);
                FileCommitLog.this.syncBytes.registerSuccessfulValue(this.unsyncedBytes);
                for (LogEntryHolderFuture e : this.syncNeeded) {
                    FileCommitLog.this.statsEntrySyncLatency.registerSuccessfulEvent(now - e.timestamp, TimeUnit.MILLISECONDS);
                    e.syncDone();
                }
                this.syncNeeded = null;
            }
            catch (Throwable t) {
                FileCommitLog.this.failed = true;
                LOGGER.log(Level.SEVERE, "general commit log failure on " + FileCommitLog.this.logDirectory, t);
                for (LogEntryHolderFuture e : this.syncNeeded) {
                    FileCommitLog.this.statsEntrySyncLatency.registerFailedEvent(now - e.timestamp, TimeUnit.MILLISECONDS);
                }
            }
        }
    }
}

