/*
 * Decompiled with CFR 0.152.
 */
package herddb.cluster;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import herddb.cluster.BookkeeperCommitLogManager;
import herddb.cluster.ZookeeperMetadataStorageManager;
import herddb.core.HerdDBInternalException;
import herddb.core.MemoryManager;
import herddb.core.PostCheckpointAction;
import herddb.core.RecordSetFactory;
import herddb.file.FileRecordSetFactory;
import herddb.index.KeyToPageIndex;
import herddb.index.blink.BLinkKeyToPageIndex;
import herddb.log.LogSequenceNumber;
import herddb.model.Index;
import herddb.model.Record;
import herddb.model.Table;
import herddb.model.Transaction;
import herddb.storage.DataPageDoesNotExistException;
import herddb.storage.DataStorageManager;
import herddb.storage.DataStorageManagerException;
import herddb.storage.FullTableScanConsumer;
import herddb.storage.IndexStatus;
import herddb.storage.TableStatus;
import herddb.utils.ByteArrayCursor;
import herddb.utils.Bytes;
import herddb.utils.ExtendedDataInputStream;
import herddb.utils.ExtendedDataOutputStream;
import herddb.utils.FileUtils;
import herddb.utils.SimpleByteArrayInputStream;
import herddb.utils.SystemInstrumentation;
import herddb.utils.VisibleByteArrayOutputStream;
import herddb.utils.XXHash64Utils;
import io.netty.util.Recycler;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.api.BKException;
import org.apache.bookkeeper.client.api.DigestType;
import org.apache.bookkeeper.client.api.LedgerEntries;
import org.apache.bookkeeper.client.api.ReadHandle;
import org.apache.bookkeeper.client.api.WriteHandle;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Op;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.ZKUtil;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

public class BookKeeperDataStorageManager
extends DataStorageManager {
    private static final Logger LOGGER = Logger.getLogger(BookKeeperDataStorageManager.class.getName());
    private static final byte[] EMPTY_ARRAY = new byte[0];
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private final Path tmpDirectory;
    private final int swapThreshold;
    private final Counter zkWrites;
    private final Counter zkReads;
    private final Counter zkGetChildren;
    private final OpStatsLogger dataPageReads;
    private final OpStatsLogger dataPageWrites;
    private final OpStatsLogger indexPageReads;
    private final OpStatsLogger indexPageWrites;
    private final ZookeeperMetadataStorageManager zk;
    private final BookkeeperCommitLogManager bk;
    private final String nodeId;
    private final ConcurrentHashMap<String, TableSpacePagesMapping> tableSpaceMappings = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, Integer> tableSpaceExpectedReplicaCount = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, Integer> tableSpaceZkNodeVersion = new ConcurrentHashMap();
    private final String rootZkNode;
    public static final String FILEEXTENSION_PAGE = ".page";
    public static final String EXTENSION_TABLEORINDExCHECKPOINTINFOFILE = ".checkpoint";
    private static final Recycler<RecyclableByteArrayOutputStream> WRITE_BUFFERS_RECYCLER = new Recycler<RecyclableByteArrayOutputStream>(){

        protected RecyclableByteArrayOutputStream newObject(Recycler.Handle<RecyclableByteArrayOutputStream> handle) {
            return new RecyclableByteArrayOutputStream(handle);
        }
    };

    private TableSpacePagesMapping getTableSpacePagesMapping(String tableSpace) {
        return this.tableSpaceMappings.computeIfAbsent(tableSpace, s -> new TableSpacePagesMapping());
    }

    private void persistTableSpaceMapping(String tableSpace) {
        try {
            TableSpacePagesMapping mapping = this.getTableSpacePagesMapping(tableSpace);
            byte[] serialized = MAPPER.writeValueAsBytes((Object)mapping);
            String znode = this.getTableSpaceMappingZNode(tableSpace);
            LOGGER.log(Level.FINE, "persistTableSpaceMapping " + tableSpace + ", " + mapping + " to " + znode + " JSON " + new String(serialized, StandardCharsets.UTF_8));
            this.writeZNodeEnforceOwnership(tableSpace, znode, serialized, null);
        }
        catch (JsonProcessingException ex) {
            throw new RuntimeException(ex);
        }
    }

    private void loadTableSpacesAtBoot() throws DataStorageManagerException {
        List<String> list = this.zkGetChildren(this.rootZkNode, false);
        LOGGER.log(Level.INFO, "tableSpaces to load: {0}", list);
        for (String tablespace : list) {
            this.loadTableSpaceMapping(tablespace);
        }
    }

    private void loadTableSpaceMapping(String tableSpace) throws DataStorageManagerException {
        Stat stat;
        String tableSpaceNode = this.getTableSpaceZNode(tableSpace);
        byte[] exists = this.readZNode(tableSpaceNode, stat = new Stat());
        if (exists == null) {
            throw new DataStorageManagerException("znode " + tableSpace + " does not exist");
        }
        this.tableSpaceZkNodeVersion.put(tableSpace, stat.getVersion());
        LOGGER.log(Level.INFO, "loadTableSpaceMapping " + tableSpace + ", tableSpaceZkNodeVersion is " + stat.getVersion());
        String znode = this.getTableSpaceMappingZNode(tableSpace);
        byte[] serialized = this.readZNode(znode, null);
        if (serialized == null) {
            LOGGER.log(Level.INFO, "loadTableSpaceMapping " + tableSpace + ", from " + znode + " was not found");
            this.tableSpaceMappings.put(tableSpace, new TableSpacePagesMapping());
        } else {
            try {
                TableSpacePagesMapping read = (TableSpacePagesMapping)MAPPER.readValue(serialized, TableSpacePagesMapping.class);
                LOGGER.log(Level.INFO, "loadTableSpaceMapping " + tableSpace + ", " + read + " from " + znode + " JSON " + new String(serialized, StandardCharsets.UTF_8));
                this.tableSpaceMappings.put(tableSpace, read);
            }
            catch (IOException ex) {
                throw new DataStorageManagerException(ex);
            }
        }
    }

    private String getTableSpaceZNode(String tableSpaceUUID) {
        return this.rootZkNode + "/" + tableSpaceUUID;
    }

    public BookKeeperDataStorageManager(String nodeId, Path baseDirectory, ZookeeperMetadataStorageManager zk, BookkeeperCommitLogManager bk) {
        this(nodeId, baseDirectory.resolve("tmp"), 10000, zk, bk, (StatsLogger)new NullStatsLogger());
    }

    public BookKeeperDataStorageManager(String nodeId, Path tmpDirectory, int swapThreshold, ZookeeperMetadataStorageManager zk, BookkeeperCommitLogManager bk, StatsLogger logger) {
        this.nodeId = nodeId;
        this.tmpDirectory = tmpDirectory;
        this.swapThreshold = swapThreshold;
        StatsLogger scope = logger.scope("bkdatastore");
        this.dataPageReads = scope.getOpStatsLogger("data_pagereads");
        this.dataPageWrites = scope.getOpStatsLogger("data_pagewrites");
        this.indexPageReads = scope.getOpStatsLogger("index_pagereads");
        this.indexPageWrites = scope.getOpStatsLogger("index_pagewrites");
        this.zkReads = scope.getCounter("zkReads");
        this.zkWrites = scope.getCounter("zkWrites");
        this.zkGetChildren = scope.getCounter("zkGetChildren");
        this.zk = zk;
        this.bk = bk;
        this.rootZkNode = zk.getBasePath() + "/data";
    }

    @Override
    public void start() throws DataStorageManagerException {
        try {
            LOGGER.log(Level.FINE, "preparing tmp directory {0}", this.tmpDirectory.toAbsolutePath().toString());
            FileUtils.cleanDirectory((Path)this.tmpDirectory);
            Files.createDirectories(this.tmpDirectory, new FileAttribute[0]);
            LOGGER.log(Level.INFO, "preparing root znode " + this.rootZkNode);
            try {
                this.zk.ensureZooKeeper().create(this.rootZkNode, EMPTY_ARRAY, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
                LOGGER.log(Level.INFO, "first boot of this cluster, created " + this.rootZkNode);
            }
            catch (KeeperException.NodeExistsException nodeExistsException) {
                // empty catch block
            }
            this.loadTableSpacesAtBoot();
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void close() throws DataStorageManagerException {
        LOGGER.log(Level.FINE, "cleaning tmp directory {0}", this.tmpDirectory.toAbsolutePath().toString());
        try {
            FileUtils.cleanDirectory((Path)this.tmpDirectory);
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Cannot clean tmp directory", err);
        }
    }

    @Override
    public void eraseTablespaceData(String tableSpace) throws DataStorageManagerException {
        SystemInstrumentation.instrumentationPoint((String)"eraseTablespaceData", (Object[])new Object[]{tableSpace});
        String tableSpaceZNode = this.getTableSpaceZNode(tableSpace);
        LOGGER.log(Level.INFO, "erasing tablespace " + tableSpace + " znode {0}", tableSpaceZNode);
        try {
            ZKUtil.deleteRecursive((ZooKeeper)this.zk.ensureZooKeeper(), (String)tableSpaceZNode);
        }
        catch (IOException | KeeperException err) {
            LOGGER.log(Level.SEVERE, "Cannot clean znode for tablespace " + tableSpace, err);
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            LOGGER.log(Level.SEVERE, "Cannot clean znode for tablespace " + tableSpace, err);
            throw new DataStorageManagerException(err);
        }
    }

    private static boolean isTablespaceCheckPointInfoFile(String name) {
        return (name = BookKeeperDataStorageManager.getFilename(name)).startsWith("checkpoint.") && name.endsWith(EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    private String getTableSpaceMappingZNode(String tablespace) {
        return this.getTableSpaceZNode(tablespace) + "/pagemap";
    }

    private String getTablespaceCheckPointInfoFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTableSpaceZNode(tablespace) + "/" + "checkpoint." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + EXTENSION_TABLEORINDExCHECKPOINTINFOFILE;
    }

    private String getTablespaceTablesMetadataFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTableSpaceZNode(tablespace) + "/" + "tables." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tablesmetadata";
    }

    private static boolean isTablespaceTablesMetadataFile(String filename) {
        return (filename = BookKeeperDataStorageManager.getFilename(filename)) != null && filename.startsWith("tables.") && filename.endsWith(".tablesmetadata");
    }

    private static boolean isTablespaceIndexesMetadataFile(String filename) {
        return (filename = BookKeeperDataStorageManager.getFilename(filename)) != null && filename.startsWith("indexes.") && filename.endsWith(".tablesmetadata");
    }

    private String getTablespaceIndexesMetadataFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTableSpaceZNode(tablespace) + "/" + "indexes." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tablesmetadata";
    }

    private String getTablespaceTransactionsFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTableSpaceZNode(tablespace) + "/" + "transactions." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tx";
    }

    private static boolean isTransactionsFile(String filename) {
        return (filename = BookKeeperDataStorageManager.getFilename(filename)) != null && filename.startsWith("transactions.") && filename.endsWith(".tx");
    }

    private String getTableDirectory(String tablespace, String tablename) {
        return this.getTableSpaceZNode(tablespace) + "/" + tablename + ".table";
    }

    private String getIndexDirectory(String tablespace, String indexname) {
        return this.getTableSpaceZNode(tablespace) + "/" + indexname + ".index";
    }

    private static String getCheckPointsFile(String tableDirectory, LogSequenceNumber sequenceNumber) {
        return tableDirectory + "/" + sequenceNumber.ledgerId + "." + sequenceNumber.offset + EXTENSION_TABLEORINDExCHECKPOINTINFOFILE;
    }

    private static boolean isTableOrIndexCheckpointsFile(String filename) {
        return (filename = BookKeeperDataStorageManager.getFilename(filename)) != null && filename.endsWith(EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    @Override
    public void initIndex(String tableSpace, String uuid) throws DataStorageManagerException {
        String indexDir = this.getIndexDirectory(tableSpace, uuid);
        LOGGER.log(Level.FINE, "initIndex {0} {1} at {2}", new Object[]{tableSpace, uuid, indexDir});
        this.createZNode(tableSpace, indexDir, EMPTY_ARRAY, false);
    }

    @Override
    public void initTablespace(String tableSpace) throws DataStorageManagerException {
        LOGGER.log(Level.FINE, "initTablespace {0}", tableSpace);
        this.blindEnsureTablespaceNodeExists(tableSpace);
        this.loadTableSpaceMapping(tableSpace);
        this.persistTableSpaceMapping(tableSpace);
    }

    private void blindEnsureTablespaceNodeExists(String tableSpace) throws DataStorageManagerException {
        try {
            this.zk.ensureZooKeeper().create(this.getTableSpaceZNode(tableSpace), EMPTY_ARRAY, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void initTable(String tableSpace, String uuid) throws DataStorageManagerException {
        String tableDir = this.getTableDirectory(tableSpace, uuid);
        LOGGER.log(Level.FINE, "initTable {0} {1} at {2}", new Object[]{tableSpace, uuid, tableDir});
        this.createZNode(tableSpace, tableDir, EMPTY_ARRAY, false);
    }

    @Override
    public List<Record> readPage(String tableSpace, String tableName, Long pageId) throws DataStorageManagerException {
        List<Record> list;
        block20: {
            long _start = System.currentTimeMillis();
            TableSpacePagesMapping tableSpacePagesMapping = this.getTableSpacePagesMapping(tableSpace);
            Long ledgerId = tableSpacePagesMapping.getTablePagesMapping(tableName).getLedgerIdForPage(pageId);
            if (ledgerId == null) {
                throw new DataPageDoesNotExistException("No such page: " + tableSpace + "_" + tableName + "." + pageId);
            }
            ReadHandle read = (ReadHandle)FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newOpenLedgerOp().withLedgerId(ledgerId.longValue()).withPassword(EMPTY_ARRAY).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);
            try {
                byte[] data;
                try (LedgerEntries entries = read.readUnconfirmed(0L, 0L);){
                    data = entries.getEntry(0L).getEntryBytes();
                }
                List<Record> result = BookKeeperDataStorageManager.rawReadDataPage(data);
                long _stop = System.currentTimeMillis();
                long delta = _stop - _start;
                LOGGER.log(Level.FINE, "readPage {0}.{1} {2} ms", new Object[]{tableSpace, tableName, delta + ""});
                this.dataPageReads.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
                list = result;
                if (read == null) break block20;
            }
            catch (Throwable throwable) {
                try {
                    if (read != null) {
                        try {
                            read.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (BKException.BKNoSuchLedgerExistsException err) {
                    throw new DataStorageManagerException(err);
                }
                catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException err) {
                    throw new DataStorageManagerException(err);
                }
                catch (BKException err) {
                    throw new DataStorageManagerException(err);
                }
                catch (IOException err) {
                    throw new DataStorageManagerException(err);
                }
                catch (InterruptedException err) {
                    Thread.currentThread().interrupt();
                    throw new DataStorageManagerException(err);
                }
            }
            read.close();
        }
        return list;
    }

    private static List<Record> rawReadDataPage(byte[] dataPage) throws IOException, DataStorageManagerException {
        try (ByteArrayCursor dataIn = ByteArrayCursor.wrap((byte[])dataPage);){
            long version = dataIn.readVLong();
            long flags = dataIn.readVLong();
            if (version != 1L || flags != 0L) {
                throw new DataStorageManagerException("corrupted data");
            }
            int numRecords = dataIn.readInt();
            ArrayList<Record> result = new ArrayList<Record>(numRecords);
            for (int i = 0; i < numRecords; ++i) {
                Bytes key = dataIn.readBytesNoCopy();
                Bytes value = dataIn.readBytesNoCopy();
                result.add(new Record(key, value));
            }
            int pos = dataIn.getPosition();
            long hashFromFile = dataIn.readLong();
            long hashFromDigest = XXHash64Utils.hash((byte[])dataPage, (int)0, (int)pos);
            if (hashFromDigest != hashFromFile) {
                throw new DataStorageManagerException("Corrupted datafile. Bad hash " + hashFromFile + " <> " + hashFromDigest);
            }
            ArrayList<Record> arrayList = result;
            return arrayList;
        }
    }

    private static <X> X readIndexPage(byte[] dataPage, DataStorageManager.DataReader<X> reader) throws IOException, DataStorageManagerException {
        try (ByteArrayCursor dataIn = ByteArrayCursor.wrap((byte[])dataPage);){
            long version = dataIn.readVLong();
            long flags = dataIn.readVLong();
            if (version != 1L || flags != 0L) {
                throw new DataStorageManagerException("corrupted data file");
            }
            X result = reader.read(dataIn);
            int pos = dataIn.getPosition();
            long hashFromFile = dataIn.readLong();
            long hashFromDigest = XXHash64Utils.hash((byte[])dataPage, (int)0, (int)pos);
            if (hashFromDigest != hashFromFile) {
                throw new DataStorageManagerException("Corrupted datafile . Bad hash " + hashFromFile + " <> " + hashFromDigest);
            }
            X x = result;
            return x;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public <X> X readIndexPage(String tableSpace, String indexName, Long pageId, DataStorageManager.DataReader<X> reader) throws DataStorageManagerException {
        long _start = System.currentTimeMillis();
        TableSpacePagesMapping tableSpacePagesMapping = this.getTableSpacePagesMapping(tableSpace);
        Long ledgerId = tableSpacePagesMapping.getIndexPagesMapping(indexName).getLedgerIdForPage(pageId);
        if (ledgerId == null) {
            throw new DataPageDoesNotExistException("No such page for index : " + tableSpace + "_" + indexName + "." + pageId);
        }
        try (ReadHandle read = (ReadHandle)FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newOpenLedgerOp().withLedgerId(ledgerId.longValue()).withPassword(EMPTY_ARRAY).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);){
            byte[] data;
            try (LedgerEntries entries = read.readUnconfirmed(0L, 0L);){
                data = entries.getEntry(0L).getEntryBytes();
            }
            X result = BookKeeperDataStorageManager.readIndexPage(data, reader);
            long _stop = System.currentTimeMillis();
            long delta = _stop - _start;
            LOGGER.log(Level.FINE, "readIndexPage {0}.{1} {2} ms", new Object[]{tableSpace, indexName, delta + ""});
            this.indexPageReads.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
            X x = result;
            return x;
        }
        catch (BKException.BKNoSuchLedgerExistsException err) {
            throw new DataStorageManagerException(err);
        }
        catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException err) {
            throw new DataStorageManagerException(err);
        }
        catch (BKException err) {
            throw new DataStorageManagerException(err);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void fullTableScan(String tableSpace, String tableName, FullTableScanConsumer consumer) throws DataStorageManagerException {
        try {
            TableStatus status = this.getLatestTableStatus(tableSpace, tableName);
            this.fullTableScan(tableSpace, tableName, status, consumer);
        }
        catch (HerdDBInternalException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void fullTableScan(String tableSpace, String tableUuid, LogSequenceNumber sequenceNumber, FullTableScanConsumer consumer) throws DataStorageManagerException {
        try {
            TableStatus status = this.getTableStatus(tableSpace, tableUuid, sequenceNumber);
            this.fullTableScan(tableSpace, tableUuid, status, consumer);
        }
        catch (HerdDBInternalException err) {
            throw new DataStorageManagerException(err);
        }
    }

    private void fullTableScan(String tableSpace, String tableUuid, TableStatus status, FullTableScanConsumer consumer) {
        LOGGER.log(Level.FINE, "fullTableScan table {0}.{1}, status: {2}", new Object[]{tableSpace, tableUuid, status});
        consumer.acceptTableStatus(status);
        ArrayList<Long> activePages = new ArrayList<Long>(status.activePages.keySet());
        activePages.sort(null);
        Iterator iterator = activePages.iterator();
        while (iterator.hasNext()) {
            long idpage = (Long)iterator.next();
            List<Record> records = this.readPage(tableSpace, tableUuid, idpage);
            LOGGER.log(Level.FINE, "fullTableScan table {0}.{1}, page {2}, contains {3} records", new Object[]{tableSpace, tableUuid, idpage, records.size()});
            consumer.acceptPage(idpage, records);
        }
        consumer.endTable();
    }

    @Override
    public int getActualNumberOfPages(String tableSpace, String tableName) throws DataStorageManagerException {
        TableStatus latestStatus = this.getLatestTableStatus(tableSpace, tableName);
        return latestStatus.activePages.size();
    }

    @Override
    public IndexStatus getIndexStatus(String tableSpace, String indexName, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        String dir = this.getIndexDirectory(tableSpace, indexName);
        String checkpointFile = BookKeeperDataStorageManager.getCheckPointsFile(dir, sequenceNumber);
        this.checkExistsZNode(checkpointFile, "no such index checkpoint: " + checkpointFile);
        return this.readIndexStatusFromFile(checkpointFile);
    }

    private void checkExistsZNode(String checkpointFile, String message) throws DataStorageManagerException {
        try {
            if (this.zk.ensureZooKeeper().exists(checkpointFile, false) == null) {
                throw new DataStorageManagerException(message);
            }
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private byte[] readZNode(String checkpointFile, Stat stat) throws DataStorageManagerException {
        try {
            this.zkReads.inc();
            return this.zk.ensureZooKeeper().getData(checkpointFile, false, stat);
        }
        catch (KeeperException.NoNodeException err) {
            return null;
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private void deleteZNodeEnforceOwnership(String tableSpace, String path) throws DataStorageManagerException {
        try {
            this.zkWrites.inc();
            Op opUpdateRoot = this.createUpdateTableSpaceRootOp(tableSpace);
            Op opDelete = Op.delete((String)path, (int)-1);
            List multi = this.zk.ensureZooKeeper().multi(Arrays.asList(opUpdateRoot, opDelete));
            this.updateRootZkNodeVersion(tableSpace, (OpResult)multi.get(0));
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private void writeZNodeEnforceOwnership(String tableSpace, String path, byte[] content, Stat stat) throws DataStorageManagerException {
        try {
            this.zkWrites.inc();
            Op opUpdateRoot = this.createUpdateTableSpaceRootOp(tableSpace);
            Op opCreate = Op.create((String)path, (byte[])content, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, (CreateMode)CreateMode.PERSISTENT);
            try {
                List multi = this.zk.ensureZooKeeper().multi(Arrays.asList(opUpdateRoot, opCreate));
                this.updateRootZkNodeVersion(tableSpace, (OpResult)multi.get(0));
            }
            catch (KeeperException.NodeExistsException ok) {
                LOGGER.log(Level.FINE, "Node exists {0}", ok.getPath());
            }
            opUpdateRoot = this.createUpdateTableSpaceRootOp(tableSpace);
            Op opUpdate = Op.setData((String)path, (byte[])content, (int)(stat != null ? stat.getVersion() : -1));
            List multi = this.zk.ensureZooKeeper().multi(Arrays.asList(opUpdateRoot, opUpdate));
            this.updateRootZkNodeVersion(tableSpace, (OpResult)multi.get(0));
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private void createZNode(String tableSpace, String path, byte[] content, boolean errorIfExists) throws DataStorageManagerException {
        try {
            this.zkWrites.inc();
            Op opUpdateRoot = this.createUpdateTableSpaceRootOp(tableSpace);
            Op opCreate = Op.create((String)path, (byte[])content, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, (CreateMode)CreateMode.PERSISTENT);
            List multi = this.zk.ensureZooKeeper().multi(Arrays.asList(opUpdateRoot, opCreate));
            this.updateRootZkNodeVersion(tableSpace, (OpResult)multi.get(0));
        }
        catch (KeeperException.NodeExistsException err) {
            if (errorIfExists) {
                throw new DataStorageManagerException(err);
            }
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private Op createUpdateTableSpaceRootOp(String tableSpace) {
        String tableSpaceRootNode = this.getTableSpaceZNode(tableSpace);
        int tableSpaceRootNodeVersion = this.tableSpaceZkNodeVersion.get(tableSpace);
        Op opUpdateRoot = Op.setData((String)tableSpaceRootNode, (byte[])EMPTY_ARRAY, (int)tableSpaceRootNodeVersion);
        LOGGER.log(Level.FINE, "createUpdateTableSpaceRootOp " + tableSpaceRootNode + " version " + tableSpaceRootNodeVersion);
        return opUpdateRoot;
    }

    private void updateRootZkNodeVersion(String tableSpace, OpResult ensureLeadership) throws DataStorageManagerException {
        OpResult.SetDataResult res = (OpResult.SetDataResult)ensureLeadership;
        LOGGER.log(Level.FINE, "updateRootZkNodeVersion " + tableSpace + " newversion " + res.getStat().getVersion());
        this.tableSpaceZkNodeVersion.put(tableSpace, res.getStat().getVersion());
    }

    private List<String> ensureZNodeDirectoryAndReturnChildren(String tableSpace, String znode) throws DataStorageManagerException {
        try {
            if (this.zk.ensureZooKeeper().exists(znode, false) != null) {
                return this.zkGetChildren(znode);
            }
            this.createZNode(tableSpace, znode, EMPTY_ARRAY, true);
            return Collections.emptyList();
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    private List<String> zkGetChildren(String znode) throws DataStorageManagerException {
        return this.zkGetChildren(znode, true);
    }

    private List<String> zkGetChildren(String znode, boolean appendParent) throws DataStorageManagerException {
        try {
            this.zkGetChildren.inc();
            List res = this.zk.ensureZooKeeper().getChildren(znode, false);
            if (appendParent) {
                return res.stream().map(s -> znode + "/" + s).collect(Collectors.toList());
            }
            return res;
        }
        catch (KeeperException.NoNodeException err) {
            return Collections.emptyList();
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public TableStatus getTableStatus(String tableSpace, String tableUuid, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        try {
            String dir = this.getTableDirectory(tableSpace, tableUuid);
            String checkpointFile = BookKeeperDataStorageManager.getCheckPointsFile(dir, sequenceNumber);
            this.checkExistsZNode(checkpointFile, "no such table checkpoint: " + checkpointFile);
            return this.readTableStatusFromFile(checkpointFile);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public TableStatus getLatestTableStatus(String tableSpace, String tableName) throws DataStorageManagerException {
        try {
            String lastFile = this.getLastTableCheckpointFile(tableSpace, tableName);
            TableStatus latestStatus = lastFile == null ? TableStatus.buildTableStatusForNewCreatedTable(tableName) : this.readTableStatusFromFile(lastFile);
            return latestStatus;
        }
        catch (IOException | KeeperException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
    }

    public TableStatus readTableStatusFromFile(String checkpointsFile) throws IOException {
        byte[] fileContent = this.readZNode(checkpointsFile, new Stat());
        return this.readTableStatusFromFile(fileContent, checkpointsFile);
    }

    public TableStatus readTableStatusFromFile(byte[] fileContent, String znode) throws IOException {
        if (fileContent == null) {
            throw new IOException("Missing ZNode for TableStatus at " + znode);
        }
        XXHash64Utils.verifyBlockWithFooter((byte[])fileContent, (int)0, (int)fileContent.length);
        try (SimpleByteArrayInputStream input = new SimpleByteArrayInputStream(fileContent);){
            TableStatus tableStatus;
            try (ExtendedDataInputStream dataIn = new ExtendedDataInputStream((InputStream)input);){
                long version = dataIn.readVLong();
                long flags = dataIn.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table status file " + znode);
                }
                tableStatus = TableStatus.deserialize(dataIn);
            }
            return tableStatus;
        }
    }

    private String getLastTableCheckpointFile(String tableSpace, String tableName) throws IOException, KeeperException, InterruptedException {
        String dir = this.getTableDirectory(tableSpace, tableName);
        String result = this.getMostRecentCheckPointFile(tableSpace, dir);
        return result;
    }

    private String getMostRecentCheckPointFile(String tableSpace, String dir) throws IOException, KeeperException, InterruptedException {
        String result = null;
        long lastMod = -1L;
        List<String> children = this.ensureZNodeDirectoryAndReturnChildren(tableSpace, dir);
        for (String fullpath : children) {
            if (BookKeeperDataStorageManager.isTableOrIndexCheckpointsFile(fullpath)) {
                LOGGER.log(Level.INFO, "getMostRecentCheckPointFile on " + dir + " -> ACCEPT " + fullpath);
                Stat stat = new Stat();
                this.zk.ensureZooKeeper().exists(fullpath, false, null, (Object)stat);
                long ts = stat.getMtime();
                if (lastMod >= 0L && lastMod >= ts) continue;
                result = fullpath;
                lastMod = ts;
                continue;
            }
            LOGGER.log(Level.INFO, "getMostRecentCheckPointFile on " + dir + " -> SKIP " + fullpath);
        }
        LOGGER.log(Level.INFO, "getMostRecentCheckPointFile on " + dir + " -> " + result);
        return result;
    }

    public IndexStatus readIndexStatusFromFile(String checkpointsFile) throws DataStorageManagerException {
        byte[] fileContent = this.readZNode(checkpointsFile, new Stat());
        if (fileContent == null) {
            throw new DataStorageManagerException("Missing znode for " + checkpointsFile + " IndexStatusFile");
        }
        return this.readIndexStatusFromFile(fileContent, checkpointsFile);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public IndexStatus readIndexStatusFromFile(byte[] fileContent, String checkpointsFile) throws DataStorageManagerException {
        try {
            if (fileContent == null) {
                throw new DataStorageManagerException("Missing znode for " + checkpointsFile + " IndexStatusFile");
            }
            XXHash64Utils.verifyBlockWithFooter((byte[])fileContent, (int)0, (int)fileContent.length);
            try (SimpleByteArrayInputStream input = new SimpleByteArrayInputStream(fileContent);){
                IndexStatus indexStatus;
                try (ExtendedDataInputStream dataIn = new ExtendedDataInputStream((InputStream)input);){
                    long version = dataIn.readVLong();
                    long flags = dataIn.readVLong();
                    if (version != 1L || flags != 0L) {
                        throw new DataStorageManagerException("corrupted index status file " + checkpointsFile);
                    }
                    indexStatus = IndexStatus.deserialize(dataIn);
                }
                return indexStatus;
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public List<PostCheckpointAction> tableCheckpoint(String tableSpace, String tableName, TableStatus tableStatus, boolean pin) throws DataStorageManagerException {
        byte[] content;
        this.persistTableSpaceMapping(tableSpace);
        LogSequenceNumber logPosition = tableStatus.sequenceNumber;
        String dir = this.getTableDirectory(tableSpace, tableName);
        String checkpointFile = BookKeeperDataStorageManager.getCheckPointsFile(dir, logPosition);
        Stat stat = new Stat();
        try {
            TableStatus actualStatus;
            byte[] exists = this.readZNode(checkpointFile, stat);
            if (exists != null && (actualStatus = this.readTableStatusFromFile(checkpointFile)) != null && actualStatus.equals(tableStatus)) {
                LOGGER.log(Level.FINE, "tableCheckpoint " + tableSpace + ", " + tableName + ": " + tableStatus + " (pin:" + pin + ") already saved on file " + checkpointFile);
                return Collections.emptyList();
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        LOGGER.log(Level.FINE, "tableCheckpoint " + tableSpace + ", " + tableName + ": " + tableStatus + " (pin:" + pin + ") to file " + checkpointFile);
        try (ByteArrayOutputStream buffer = new ByteArrayOutputStream();
             XXHash64Utils.HashingOutputStream oo = new XXHash64Utils.HashingOutputStream((OutputStream)buffer);
             ExtendedDataOutputStream dataOutputKeys = new ExtendedDataOutputStream((OutputStream)oo);){
            dataOutputKeys.writeVLong(1L);
            dataOutputKeys.writeVLong(0L);
            tableStatus.serialize(dataOutputKeys);
            dataOutputKeys.writeLong(oo.hash());
            dataOutputKeys.flush();
            content = buffer.toByteArray();
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        this.writeZNodeEnforceOwnership(tableSpace, checkpointFile, content, stat);
        Map<Long, Integer> pins = this.pinTableAndGetPages(tableSpace, tableName, tableStatus, pin);
        Set<LogSequenceNumber> checkpoints = this.pinTableAndGetCheckpoints(tableSpace, tableName, tableStatus, pin);
        long maxPageId = tableStatus.activePages.keySet().stream().max(Comparator.naturalOrder()).orElse(Long.MAX_VALUE);
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        PagesMapping tableSpacePagesMapping = this.getTableSpacePagesMapping(tableSpace).getTablePagesMapping(tableName);
        for (Map.Entry<Long, Long> entry : tableSpacePagesMapping.pages.entrySet()) {
            long pageId = entry.getKey();
            long ledgerId = entry.getValue();
            LOGGER.log(Level.FINEST, "checkpoint pageId {0} ledgerId {1}", new Object[]{pageId, ledgerId});
            if (pageId <= 0L || pins.containsKey(pageId) || tableStatus.activePages.containsKey(pageId) || pageId >= maxPageId) continue;
            LOGGER.log(Level.FINEST, "checkpoint ledger " + ledgerId + " pageId " + pageId + ". will be deleted after checkpoint end");
            result.add(new DropLedgerForTableAction(tableSpace, tableName, "delete page " + pageId + " ledgerId " + ledgerId, pageId, ledgerId));
        }
        for (Long l : tableSpacePagesMapping.oldLedgers) {
            LOGGER.log(Level.FINEST, "checkpoint ledger " + l + " without page. will be deleted after checkpoint end");
            result.add(new DropLedgerForTableAction(tableSpace, tableName, "delete unused ledgerId " + l, Long.MAX_VALUE, l));
        }
        List<String> children = this.zkGetChildren(dir);
        try {
            for (String p : children) {
                if (!BookKeeperDataStorageManager.isTableOrIndexCheckpointsFile(p) || p.equals(checkpointFile)) continue;
                TableStatus status = this.readTableStatusFromFile(p);
                if (!logPosition.after(status.sequenceNumber) || checkpoints.contains(status.sequenceNumber)) continue;
                LOGGER.log(Level.FINEST, "checkpoint metadata znode " + p + ". will be deleted after checkpoint end");
                result.add(new DeleteZNodeAction(tableSpace, tableName, "delete checkpoint metadata znode " + p, p));
            }
        }
        catch (IOException iOException) {
            LOGGER.log(Level.SEVERE, "Could not list table dir " + dir, iOException);
        }
        return result;
    }

    @Override
    public List<PostCheckpointAction> indexCheckpoint(String tableSpace, String indexName, IndexStatus indexStatus, boolean pin) throws DataStorageManagerException {
        byte[] content;
        IndexStatus actualStatus;
        Stat stat;
        LogSequenceNumber logPosition;
        String dir = this.getIndexDirectory(tableSpace, indexName);
        String checkpointFile = BookKeeperDataStorageManager.getCheckPointsFile(dir, logPosition = indexStatus.sequenceNumber);
        byte[] exists = this.readZNode(checkpointFile, stat = new Stat());
        if (exists != null && (actualStatus = this.readIndexStatusFromFile(exists, checkpointFile)) != null && actualStatus.equals(indexStatus)) {
            LOGGER.log(Level.INFO, "indexCheckpoint " + tableSpace + ", " + indexName + ": " + indexStatus + " already saved on" + checkpointFile);
            return Collections.emptyList();
        }
        LOGGER.log(Level.FINE, "indexCheckpoint " + tableSpace + ", " + indexName + ": " + indexStatus + " to file " + checkpointFile);
        try (ByteArrayOutputStream buffer = new ByteArrayOutputStream();
             XXHash64Utils.HashingOutputStream oo = new XXHash64Utils.HashingOutputStream((OutputStream)buffer);
             ExtendedDataOutputStream dataOutputKeys = new ExtendedDataOutputStream((OutputStream)oo);){
            dataOutputKeys.writeVLong(1L);
            dataOutputKeys.writeVLong(0L);
            indexStatus.serialize(dataOutputKeys);
            dataOutputKeys.writeLong(oo.hash());
            dataOutputKeys.flush();
            content = buffer.toByteArray();
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        this.writeZNodeEnforceOwnership(tableSpace, checkpointFile, content, stat);
        Map<Long, Integer> pins = this.pinIndexAndGetPages(tableSpace, indexName, indexStatus, pin);
        Set<LogSequenceNumber> checkpoints = this.pinIndexAndGetCheckpoints(tableSpace, indexName, indexStatus, pin);
        long maxPageId = indexStatus.activePages.stream().max(Comparator.naturalOrder()).orElse(Long.MAX_VALUE);
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        PagesMapping tableSpacePagesMapping = this.getTableSpacePagesMapping(tableSpace).getIndexPagesMapping(indexName);
        for (Map.Entry<Long, Long> pages : tableSpacePagesMapping.pages.entrySet()) {
            long pageId = pages.getKey();
            long ledgerId = pages.getValue();
            LOGGER.log(Level.FINEST, "checkpoint pageId {0} ledgerId {1}", new Object[]{pageId, ledgerId});
            if (pageId <= 0L || pins.containsKey(pageId) || indexStatus.activePages.contains(pageId) || pageId >= maxPageId) continue;
            LOGGER.log(Level.FINEST, "checkpoint ledger " + ledgerId + " pageId " + pageId + ". will be deleted after checkpoint end");
            result.add(new DropLedgerForIndexAction(tableSpace, indexName, "delete index page " + pageId + " ledgerId " + ledgerId, pageId, ledgerId));
        }
        for (Long ledgerId : tableSpacePagesMapping.oldLedgers) {
            LOGGER.log(Level.FINEST, "checkpoint ledger " + ledgerId + " without page. will be deleted after checkpoint end");
            result.add(new DropLedgerForIndexAction(tableSpace, indexName, "delete unused ledgerId " + ledgerId, Long.MAX_VALUE, ledgerId));
        }
        List<String> children = this.zkGetChildren(dir);
        for (String p : children) {
            if (!BookKeeperDataStorageManager.isTableOrIndexCheckpointsFile(p) || p.equals(checkpointFile)) continue;
            IndexStatus status = this.readIndexStatusFromFile(p);
            if (!logPosition.after(status.sequenceNumber) || checkpoints.contains(status.sequenceNumber)) continue;
            LOGGER.log(Level.FINEST, "checkpoint metadata file " + p + ". will be deleted after checkpoint end");
            result.add(new DeleteZNodeAction(tableSpace, indexName, "delete checkpoint metadata file " + p, p));
        }
        return result;
    }

    private static String getFilename(String s) {
        int last = s.lastIndexOf("/");
        return s.substring(last + 1);
    }

    private static long getPageId(String p) {
        String filename = BookKeeperDataStorageManager.getFilename(p);
        if (filename.endsWith(FILEEXTENSION_PAGE)) {
            try {
                return Long.parseLong(filename.substring(0, filename.length() - FILEEXTENSION_PAGE.length()));
            }
            catch (NumberFormatException no) {
                return -1L;
            }
        }
        return -1L;
    }

    @Override
    public void cleanupAfterTableBoot(String tableSpace, String tableName, Set<Long> activePagesAtBoot) throws DataStorageManagerException {
        PagesMapping tablePagesMapping = this.getTableSpacePagesMapping(tableSpace).getTablePagesMapping(tableName);
        for (Map.Entry<Long, Long> next : tablePagesMapping.pages.entrySet()) {
            long pageId = next.getKey();
            long ledgerId = next.getValue();
            LOGGER.log(Level.FINER, "cleanupAfterTableBoot pageId " + pageId + " ledger id " + ledgerId);
            if (pageId <= 0L || activePagesAtBoot.contains(pageId)) continue;
            LOGGER.log(Level.INFO, "cleanupAfterTableBoot pageId " + pageId + " ledger id " + ledgerId + ". will be deleted");
            this.dropLedgerForTable(tableSpace, tableName, pageId, ledgerId, "cleanupAfterBoot " + tableSpace + "." + tableName + " pageId " + pageId);
        }
    }

    private static long writePage(Collection<Record> newPage, VisibleByteArrayOutputStream oo) throws IOException {
        try (ExtendedDataOutputStream dataOutput = new ExtendedDataOutputStream((OutputStream)oo);){
            dataOutput.writeVLong(1L);
            dataOutput.writeVLong(0L);
            dataOutput.writeInt(newPage.size());
            for (Record record : newPage) {
                dataOutput.writeArray(record.key);
                dataOutput.writeArray(record.value);
            }
            dataOutput.flush();
            long hash = XXHash64Utils.hash((byte[])oo.getBuffer(), (int)0, (int)oo.size());
            dataOutput.writeLong(hash);
            dataOutput.flush();
            long l = oo.size();
            return l;
        }
    }

    @Override
    public void writePage(String tableSpace, String tableName, long pageId, Collection<Record> newPage) throws DataStorageManagerException {
        long ledgerId;
        long size;
        long _start = System.currentTimeMillis();
        try (VisibleByteArrayOutputStream buffer = new VisibleByteArrayOutputStream();){
            size = BookKeeperDataStorageManager.writePage(newPage, buffer);
            HashMap<String, byte[]> metadata = new HashMap<String, byte[]>();
            metadata.put("tablespaceuuid", tableSpace.getBytes(StandardCharsets.UTF_8));
            metadata.put("node", this.nodeId.getBytes(StandardCharsets.UTF_8));
            metadata.put("application", "herddb".getBytes(StandardCharsets.UTF_8));
            metadata.put("component", "datastore".getBytes(StandardCharsets.UTF_8));
            metadata.put("table", tableName.getBytes(StandardCharsets.UTF_8));
            metadata.put("type", "datapage".getBytes(StandardCharsets.UTF_8));
            int expectedReplicaCount = this.tableSpaceExpectedReplicaCount.getOrDefault(tableSpace, 1);
            int actualEnsembleSize = Math.max(expectedReplicaCount, this.bk.getEnsemble());
            int actualWriteQuorumSize = Math.max(expectedReplicaCount, this.bk.getWriteQuorumSize());
            int actualAckQuorumSize = Math.max(expectedReplicaCount, this.bk.getAckQuorumSize());
            try (WriteHandle result = (WriteHandle)FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newCreateLedgerOp().withEnsembleSize(actualEnsembleSize).withWriteQuorumSize(actualWriteQuorumSize).withAckQuorumSize(actualAckQuorumSize).withPassword(EMPTY_ARRAY).withDigestType(DigestType.CRC32C).withCustomMetadata(metadata).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);){
                result.append(buffer.getBuffer(), 0, buffer.size());
                ledgerId = result.getId();
            }
        }
        catch (IOException | BKException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
        this.getTableSpacePagesMapping(tableSpace).getTablePagesMapping(tableName).writePageId(pageId, ledgerId);
        long now = System.currentTimeMillis();
        long delta = now - _start;
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.log(Level.FINER, "writePage {0} KBytes,{1} records, time {2} ms", new Object[]{size / 1024L + "", newPage.size(), delta + ""});
        }
        this.dataPageWrites.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
    }

    private static long writeIndexPage(DataStorageManager.DataWriter writer, VisibleByteArrayOutputStream stream) throws IOException {
        try (ExtendedDataOutputStream dataOutput = new ExtendedDataOutputStream((OutputStream)stream);){
            dataOutput.writeVLong(1L);
            dataOutput.writeVLong(0L);
            writer.write(dataOutput);
            dataOutput.flush();
            long hash = XXHash64Utils.hash((byte[])stream.getBuffer(), (int)0, (int)stream.size());
            dataOutput.writeLong(hash);
            dataOutput.flush();
            long l = stream.size();
            return l;
        }
    }

    @Override
    public void writeIndexPage(String tableSpace, String indexName, long pageId, DataStorageManager.DataWriter writer) throws DataStorageManagerException {
        long ledgerId;
        long size;
        long _start = System.currentTimeMillis();
        try (VisibleByteArrayOutputStream buffer = new VisibleByteArrayOutputStream();){
            size = BookKeeperDataStorageManager.writeIndexPage(writer, buffer);
            HashMap<String, byte[]> metadata = new HashMap<String, byte[]>();
            metadata.put("tablespaceuuid", tableSpace.getBytes(StandardCharsets.UTF_8));
            metadata.put("node", this.nodeId.getBytes(StandardCharsets.UTF_8));
            metadata.put("application", "herddb".getBytes(StandardCharsets.UTF_8));
            metadata.put("component", "datastore".getBytes(StandardCharsets.UTF_8));
            metadata.put("index", indexName.getBytes(StandardCharsets.UTF_8));
            metadata.put("type", "indexpage".getBytes(StandardCharsets.UTF_8));
            ledgerId = this.writeToLedger(tableSpace, metadata, buffer);
        }
        catch (IOException | BKException err) {
            throw new DataStorageManagerException(err);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new DataStorageManagerException(err);
        }
        LOGGER.log(Level.INFO, "writeIndexPage {0} pageId {1}, ledgerId {2}", new Object[]{indexName, pageId, ledgerId});
        this.getTableSpacePagesMapping(tableSpace).getIndexPagesMapping(indexName).writePageId(pageId, ledgerId);
        long now = System.currentTimeMillis();
        long delta = now - _start;
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.log(Level.FINER, "writePage {0} KBytes, time {2} ms", new Object[]{size / 1024L + "", delta + ""});
        }
        this.indexPageWrites.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
    }

    private long writeToLedger(String tableSpace, Map<String, byte[]> metadata, VisibleByteArrayOutputStream buffer) throws InterruptedException, BKException {
        long ledgerId;
        int expectedReplicaCount = this.tableSpaceExpectedReplicaCount.getOrDefault(tableSpace, 1);
        int actualEnsembleSize = Math.max(expectedReplicaCount, this.bk.getEnsemble());
        int actualWriteQuorumSize = Math.max(expectedReplicaCount, this.bk.getWriteQuorumSize());
        int actualAckQuorumSize = Math.max(expectedReplicaCount, this.bk.getAckQuorumSize());
        try (WriteHandle result = (WriteHandle)FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newCreateLedgerOp().withEnsembleSize(actualEnsembleSize).withWriteQuorumSize(actualWriteQuorumSize).withAckQuorumSize(actualAckQuorumSize).withPassword(EMPTY_ARRAY).withDigestType(DigestType.CRC32C).withCustomMetadata(metadata).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);){
            result.append(buffer.getBuffer(), 0, buffer.size());
            ledgerId = result.getId();
        }
        return ledgerId;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromTablesMetadataFile(String tableSpace, byte[] data, String znode) throws DataStorageManagerException {
        try (ByteArrayInputStream input = new ByteArrayInputStream(data);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table list znode" + znode);
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("znode " + znode + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromIndexMetadataFile(String tableSpace, byte[] data, String znode) throws DataStorageManagerException {
        try (ByteArrayInputStream input = new ByteArrayInputStream(data);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted index list znode " + znode);
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("znode " + znode + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public List<Table> loadTables(LogSequenceNumber sequenceNumber, String tableSpace) throws DataStorageManagerException {
        try {
            String file = this.getTablespaceTablesMetadataFile(tableSpace, sequenceNumber);
            LOGGER.log(Level.INFO, "loadTables for tableSpace " + tableSpace + " from " + file + ", sequenceNumber:" + sequenceNumber);
            byte[] content = this.readZNode(file, new Stat());
            if (content == null) {
                if (sequenceNumber.isStartOfTime()) {
                    LOGGER.log(Level.INFO, "zode " + file + " not found");
                    return Collections.emptyList();
                }
                throw new DataStorageManagerException("local table data not available for tableSpace " + tableSpace + ", recovering from sequenceNumber " + sequenceNumber);
            }
            return BookKeeperDataStorageManager.readTablespaceStructure(content, tableSpace, sequenceNumber);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    public static List<Table> readTablespaceStructure(byte[] file, String tableSpace, LogSequenceNumber sequenceNumber) throws IOException, DataStorageManagerException {
        try (ByteArrayInputStream input = new ByteArrayInputStream(file);){
            List<Table> list;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table list file");
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file is not for spablespace " + tableSpace + " but for " + readname);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                if (sequenceNumber != null && (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset)) {
                    throw new DataStorageManagerException("file is not for sequence number " + sequenceNumber);
                }
                int numTables = din.readInt();
                ArrayList<Table> res = new ArrayList<Table>();
                for (int i = 0; i < numTables; ++i) {
                    byte[] tableData = din.readArray();
                    Table table = Table.deserialize(tableData);
                    res.add(table);
                }
                list = Collections.unmodifiableList(res);
            }
            return list;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public List<Index> loadIndexes(LogSequenceNumber sequenceNumber, String tableSpace) throws DataStorageManagerException {
        try {
            String file = this.getTablespaceIndexesMetadataFile(tableSpace, sequenceNumber);
            LOGGER.log(Level.INFO, "loadIndexes for tableSpace " + tableSpace + " from " + file + ", sequenceNumber:" + sequenceNumber);
            byte[] content = this.readZNode(file, new Stat());
            if (content == null) {
                if (sequenceNumber.isStartOfTime()) {
                    LOGGER.log(Level.INFO, "file " + file + " not found");
                    return Collections.emptyList();
                }
                throw new DataStorageManagerException("local index data not available for tableSpace " + tableSpace + ", recovering from sequenceNumber " + sequenceNumber);
            }
            try (ByteArrayInputStream input = new ByteArrayInputStream(content);){
                List<Index> list;
                try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                    long version = din.readVLong();
                    long flags = din.readVLong();
                    if (version != 1L || flags != 0L) {
                        throw new DataStorageManagerException("corrupted index list file " + file);
                    }
                    String readname = din.readUTF();
                    if (!readname.equals(tableSpace)) {
                        throw new DataStorageManagerException("file " + file + " is not for tablespace " + tableSpace);
                    }
                    long ledgerId = din.readZLong();
                    long offset = din.readZLong();
                    if (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset) {
                        throw new DataStorageManagerException("file " + file + " is not for sequence number " + sequenceNumber);
                    }
                    int numTables = din.readInt();
                    ArrayList<Index> res = new ArrayList<Index>();
                    for (int i = 0; i < numTables; ++i) {
                        byte[] indexData = din.readArray();
                        Index table = Index.deserialize(indexData);
                        res.add(table);
                    }
                    list = Collections.unmodifiableList(res);
                }
                return list;
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public Collection<PostCheckpointAction> writeTables(String tableSpace, LogSequenceNumber sequenceNumber, List<Table> tables, List<Index> indexlist, boolean prepareActions) throws DataStorageManagerException {
        ExtendedDataOutputStream dout;
        VisibleByteArrayOutputStream buffer;
        if (sequenceNumber.isStartOfTime() && !tables.isEmpty()) {
            throw new DataStorageManagerException("impossible to write a non empty table list at start-of-time");
        }
        this.persistTableSpaceMapping(tableSpace);
        String tableSpaceDirectory = this.getTableSpaceZNode(tableSpace);
        String fileTables = this.getTablespaceTablesMetadataFile(tableSpace, sequenceNumber);
        String fileIndexes = this.getTablespaceIndexesMetadataFile(tableSpace, sequenceNumber);
        LOGGER.log(Level.FINE, "writeTables for tableSpace " + tableSpace + " sequenceNumber " + sequenceNumber + " to " + fileTables);
        try {
            buffer = new VisibleByteArrayOutputStream();
            try {
                dout = new ExtendedDataOutputStream((OutputStream)buffer);
                try {
                    dout.writeVLong(1L);
                    dout.writeVLong(0L);
                    dout.writeUTF(tableSpace);
                    dout.writeZLong(sequenceNumber.ledgerId);
                    dout.writeZLong(sequenceNumber.offset);
                    dout.writeInt(tables.size());
                    for (Table table : tables) {
                        byte[] tableSerialized = table.serialize();
                        dout.writeArray(tableSerialized);
                    }
                    dout.flush();
                    this.writeZNodeEnforceOwnership(tableSpace, fileTables, buffer.toByteArray(), null);
                }
                finally {
                    dout.close();
                }
            }
            finally {
                buffer.close();
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        try {
            buffer = new VisibleByteArrayOutputStream();
            try {
                dout = new ExtendedDataOutputStream((OutputStream)buffer);
                try {
                    dout.writeVLong(1L);
                    dout.writeVLong(0L);
                    dout.writeUTF(tableSpace);
                    dout.writeZLong(sequenceNumber.ledgerId);
                    dout.writeZLong(sequenceNumber.offset);
                    if (indexlist != null) {
                        dout.writeInt(indexlist.size());
                        for (Index index : indexlist) {
                            byte[] indexSerialized = index.serialize();
                            dout.writeArray(indexSerialized);
                        }
                    } else {
                        dout.writeInt(0);
                    }
                    dout.flush();
                    this.writeZNodeEnforceOwnership(tableSpace, fileIndexes, buffer.toByteArray(), null);
                }
                finally {
                    dout.close();
                }
            }
            finally {
                buffer.close();
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        if (prepareActions) {
            List<String> stream = this.zkGetChildren(tableSpaceDirectory);
            for (String string : stream) {
                LogSequenceNumber logPositionInFile;
                byte[] content;
                if (BookKeeperDataStorageManager.isTablespaceIndexesMetadataFile(string)) {
                    try {
                        content = this.readZNode(string, new Stat());
                        if (content == null || !sequenceNumber.after(logPositionInFile = BookKeeperDataStorageManager.readLogSequenceNumberFromIndexMetadataFile(tableSpace, content, string))) continue;
                        LOGGER.log(Level.FINEST, "indexes metadata file " + string + ". will be deleted after checkpoint end");
                        result.add(new DeleteZNodeAction(tableSpace, "indexes", "delete indexesmetadata file " + string, string));
                    }
                    catch (DataStorageManagerException ignore) {
                        LOGGER.log(Level.SEVERE, "Unparsable indexesmetadata file " + string, (Throwable)((Object)ignore));
                        result.add(new DeleteZNodeAction(tableSpace, "indexes", "delete unparsable indexesmetadata file " + string, string));
                    }
                    continue;
                }
                if (!BookKeeperDataStorageManager.isTablespaceTablesMetadataFile(string)) continue;
                try {
                    content = this.readZNode(string, new Stat());
                    if (content == null || !sequenceNumber.after(logPositionInFile = BookKeeperDataStorageManager.readLogSequenceNumberFromTablesMetadataFile(tableSpace, content, string))) continue;
                    LOGGER.log(Level.FINEST, "tables metadata file " + string + ". will be deleted after checkpoint end");
                    result.add(new DeleteZNodeAction(tableSpace, "tables", "delete tablesmetadata file " + string, string));
                }
                catch (DataStorageManagerException ignore) {
                    LOGGER.log(Level.SEVERE, "Unparsable tablesmetadata file " + string, (Throwable)((Object)ignore));
                    result.add(new DeleteZNodeAction(tableSpace, "transactions", "delete unparsable tablesmetadata file " + string, string));
                }
            }
        }
        return result;
    }

    @Override
    public Collection<PostCheckpointAction> writeCheckpointSequenceNumber(String tableSpace, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        this.persistTableSpaceMapping(tableSpace);
        String checkPointFile = this.getTablespaceCheckPointInfoFile(tableSpace, sequenceNumber);
        LOGGER.log(Level.INFO, "checkpoint for {0} at {1} to {2}", new Object[]{tableSpace, sequenceNumber, checkPointFile});
        try (VisibleByteArrayOutputStream buffer = new VisibleByteArrayOutputStream();
             ExtendedDataOutputStream dout = new ExtendedDataOutputStream((OutputStream)buffer);){
            dout.writeVLong(1L);
            dout.writeVLong(0L);
            dout.writeUTF(tableSpace);
            dout.writeZLong(sequenceNumber.ledgerId);
            dout.writeZLong(sequenceNumber.offset);
            dout.flush();
            this.writeZNodeEnforceOwnership(tableSpace, checkPointFile, buffer.toByteArray(), null);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        String tableSpaceDirectory = this.getTableSpaceZNode(tableSpace);
        List<String> stream = this.zkGetChildren(tableSpaceDirectory);
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        for (String p : stream) {
            if (!BookKeeperDataStorageManager.isTablespaceCheckPointInfoFile(p)) continue;
            try {
                LogSequenceNumber logPositionInFile;
                byte[] content = this.readZNode(p, new Stat());
                if (content == null || !sequenceNumber.after(logPositionInFile = BookKeeperDataStorageManager.readLogSequenceNumberFromCheckpointInfoFile(tableSpace, content, p))) continue;
                LOGGER.log(Level.FINEST, "checkpoint info file " + p + ". will be deleted after checkpoint end");
                result.add(new DeleteZNodeAction(tableSpace, "checkpoint", "delete checkpoint info file " + p, p));
            }
            catch (DataStorageManagerException | IOException ignore) {
                LOGGER.log(Level.SEVERE, "unparsable checkpoint info file " + p, (Throwable)ignore);
            }
        }
        return result;
    }

    @Override
    public void dropTable(String tablespace, String tableName) throws DataStorageManagerException {
        this.persistTableSpaceMapping(tablespace);
        String tableDir = this.getTableDirectory(tablespace, tableName);
        LOGGER.log(Level.INFO, "dropTable {0}.{1} in {2}", new Object[]{tablespace, tableName, tableDir});
        try {
            ZKUtil.deleteRecursive((ZooKeeper)this.zk.ensureZooKeeper(), (String)tableDir);
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    @Override
    public void truncateIndex(String tablespace, String name) throws DataStorageManagerException {
        this.persistTableSpaceMapping(tablespace);
        String tableDir = this.getIndexDirectory(tablespace, name);
        LOGGER.log(Level.INFO, "truncateIndex {0}.{1} in {2}", new Object[]{tablespace, name, tableDir});
        try {
            ZKUtil.deleteRecursive((ZooKeeper)this.zk.ensureZooKeeper(), (String)tableDir);
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    @Override
    public void dropIndex(String tablespace, String name) throws DataStorageManagerException {
        this.persistTableSpaceMapping(tablespace);
        String tableDir = this.getIndexDirectory(tablespace, name);
        LOGGER.log(Level.INFO, "dropIndex {0}.{1} in {2}", new Object[]{tablespace, name, tableDir});
        try {
            ZKUtil.deleteRecursive((ZooKeeper)this.zk.ensureZooKeeper(), (String)tableDir);
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    private static LogSequenceNumber readLogSequenceNumberFromCheckpointInfoFile(String tableSpace, byte[] data, String checkPointFile) throws DataStorageManagerException, IOException {
        try (ByteArrayInputStream input = new ByteArrayInputStream(data);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new IOException("corrupted checkpoint file");
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("zonde " + checkPointFile + " is not for spablespace " + tableSpace + " but for " + readname);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
    }

    @Override
    public LogSequenceNumber getLastcheckpointSequenceNumber(String tableSpace) throws DataStorageManagerException {
        try {
            String tableSpaceDirectory = this.getTableSpaceZNode(tableSpace);
            LogSequenceNumber max = LogSequenceNumber.START_OF_TIME;
            List<String> stream = this.zkGetChildren(tableSpaceDirectory);
            for (String p : stream) {
                if (!BookKeeperDataStorageManager.isTablespaceCheckPointInfoFile(p)) continue;
                try {
                    LogSequenceNumber logPositionInFile;
                    byte[] content = this.readZNode(p, new Stat());
                    if (content == null || !(logPositionInFile = BookKeeperDataStorageManager.readLogSequenceNumberFromCheckpointInfoFile(tableSpace, content, p)).after(max)) continue;
                    max = logPositionInFile;
                }
                catch (DataStorageManagerException ignore) {
                    LOGGER.log(Level.SEVERE, "unparsable checkpoint info file " + p, (Throwable)((Object)ignore));
                }
            }
            return max;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public KeyToPageIndex createKeyToPageMap(String tablespace, String name, MemoryManager memoryManager) throws DataStorageManagerException {
        return new BLinkKeyToPageIndex(tablespace, name, memoryManager, this);
    }

    @Override
    public void releaseKeyToPageMap(String tablespace, String name, KeyToPageIndex keyToPage) {
        if (keyToPage != null) {
            keyToPage.close();
        }
    }

    @Override
    public RecordSetFactory createRecordSetFactory() {
        return new FileRecordSetFactory(this.tmpDirectory, this.swapThreshold);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromTransactionsFile(String tableSpace, byte[] data, String file) throws DataStorageManagerException {
        try (ByteArrayInputStream input = new ByteArrayInputStream(data);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted transaction list znode " + file);
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("znode " + file + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void loadTransactions(LogSequenceNumber sequenceNumber, String tableSpace, Consumer<Transaction> consumer) throws DataStorageManagerException {
        try {
            String file = this.getTablespaceTransactionsFile(tableSpace, sequenceNumber);
            byte[] content = this.readZNode(file, new Stat());
            boolean exists = content != null;
            LOGGER.log(Level.INFO, "loadTransactions " + sequenceNumber + " for tableSpace " + tableSpace + " from file " + file + " (exists: " + exists + ")");
            if (!exists) {
                return;
            }
            try (ByteArrayInputStream input = new ByteArrayInputStream(content);
                 ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted transaction list file " + file);
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                if (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset) {
                    throw new DataStorageManagerException("file " + file + " is not for sequence number " + sequenceNumber);
                }
                int numTransactions = din.readInt();
                for (int i = 0; i < numTransactions; ++i) {
                    Transaction tx = Transaction.deserialize(tableSpace, din);
                    consumer.accept(tx);
                }
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void tableSpaceMetadataUpdated(String tableSpace, int expectedReplicaCount) {
        this.tableSpaceExpectedReplicaCount.put(tableSpace, expectedReplicaCount);
    }

    @Override
    public Collection<PostCheckpointAction> writeTransactionsAtCheckpoint(String tableSpace, LogSequenceNumber sequenceNumber, Collection<Transaction> transactions) throws DataStorageManagerException {
        if (sequenceNumber.isStartOfTime() && !transactions.isEmpty()) {
            throw new DataStorageManagerException("impossible to write a non empty transactions list at start-of-time");
        }
        String checkPointFile = this.getTablespaceTransactionsFile(tableSpace, sequenceNumber);
        LOGGER.log(Level.FINE, "writeTransactionsAtCheckpoint for tableSpace {0} sequenceNumber {1} to {2}, active transactions {3}", new Object[]{tableSpace, sequenceNumber, checkPointFile, transactions.size()});
        try (VisibleByteArrayOutputStream buffer = new VisibleByteArrayOutputStream();
             ExtendedDataOutputStream dout = new ExtendedDataOutputStream((OutputStream)buffer);){
            dout.writeVLong(1L);
            dout.writeVLong(0L);
            dout.writeUTF(tableSpace);
            dout.writeZLong(sequenceNumber.ledgerId);
            dout.writeZLong(sequenceNumber.offset);
            dout.writeInt(transactions.size());
            for (Transaction t : transactions) {
                t.serialize(dout);
            }
            dout.flush();
            this.writeZNodeEnforceOwnership(tableSpace, checkPointFile, buffer.toByteArray(), null);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        String tableSpaceDirectory = this.getTableSpaceZNode(tableSpace);
        List<String> stream = this.zkGetChildren(tableSpaceDirectory);
        for (String p : stream) {
            if (!BookKeeperDataStorageManager.isTransactionsFile(p)) continue;
            try {
                LogSequenceNumber logPositionInFile;
                byte[] content = this.readZNode(checkPointFile, new Stat());
                if (content == null || !sequenceNumber.after(logPositionInFile = BookKeeperDataStorageManager.readLogSequenceNumberFromTransactionsFile(tableSpace, content, p))) continue;
                LOGGER.log(Level.FINEST, "transactions metadata file " + p + ". will be deleted after checkpoint end");
                result.add(new DeleteZNodeAction(tableSpace, "transactions", "delete transactions file " + p, p));
            }
            catch (DataStorageManagerException ignore) {
                LOGGER.log(Level.SEVERE, "Unparsable transactions file " + p, (Throwable)((Object)ignore));
                result.add(new DeleteZNodeAction(tableSpace, "transactions", "delete unparsable transactions file " + p, p));
            }
        }
        return result;
    }

    private void dropLedgerForTable(String tableSpace, String tableName, long pageId, long ledgerId, String description) {
        try {
            LOGGER.log(Level.FINE, description);
            try {
                FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newDeleteLedgerOp().withLedgerId(ledgerId).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);
            }
            catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException err) {
                LOGGER.log(Level.SEVERE, "ledger " + ledgerId + " already dropped:" + (Object)((Object)err), err);
            }
            this.getTableSpacePagesMapping(tableSpace).getTablePagesMapping(tableName).removePageId(pageId);
        }
        catch (org.apache.bookkeeper.client.BKException err) {
            LOGGER.log(Level.SEVERE, "Could not delete ledger " + ledgerId + ":" + (Object)((Object)err), err);
        }
    }

    private void dropLedgerForIndex(String tableSpace, String indexName, long pageId, long ledgerId, String description) {
        try {
            LOGGER.log(Level.FINE, description);
            try {
                FutureUtils.result((CompletableFuture)this.bk.getBookKeeper().newDeleteLedgerOp().withLedgerId(ledgerId).execute(), (Function)org.apache.bookkeeper.client.BKException.HANDLER);
            }
            catch (BKException.BKNoSuchLedgerExistsOnMetadataServerException err) {
                LOGGER.log(Level.SEVERE, "ledger " + ledgerId + " already dropped:" + (Object)((Object)err), err);
            }
            this.getTableSpacePagesMapping(tableSpace).getIndexPagesMapping(indexName).removePageId(pageId);
        }
        catch (org.apache.bookkeeper.client.BKException err) {
            LOGGER.log(Level.SEVERE, "Could not delete ledger " + ledgerId + ":" + (Object)((Object)err), err);
        }
    }

    private static RecyclableByteArrayOutputStream getWriteBuffer() {
        RecyclableByteArrayOutputStream res = (RecyclableByteArrayOutputStream)((Object)WRITE_BUFFERS_RECYCLER.get());
        res.closed = false;
        res.reset();
        return res;
    }

    public static final class TableSpacePagesMapping {
        private ConcurrentHashMap<String, PagesMapping> tables = new ConcurrentHashMap();
        private ConcurrentHashMap<String, PagesMapping> indexes = new ConcurrentHashMap();

        private PagesMapping getTablePagesMapping(String tableName) {
            return this.tables.computeIfAbsent(tableName, t -> new PagesMapping());
        }

        private PagesMapping getIndexPagesMapping(String indexName) {
            return this.indexes.computeIfAbsent(indexName, t -> new PagesMapping());
        }

        public ConcurrentHashMap<String, PagesMapping> getTableMappings() {
            return this.tables;
        }

        public void setTableMappings(ConcurrentHashMap<String, PagesMapping> tableMappings) {
            this.tables = tableMappings;
        }

        public ConcurrentHashMap<String, PagesMapping> getIndexMappings() {
            return this.indexes;
        }

        public void setIndexMappings(ConcurrentHashMap<String, PagesMapping> indexMappings) {
            this.indexes = indexMappings;
        }

        public String toString() {
            return "TableSpacePagesMapping{tableMappings=" + this.tables + ", indexMappings=" + this.indexes + '}';
        }
    }

    private static final class PagesMapping {
        ConcurrentHashMap<Long, Long> pages = new ConcurrentHashMap();
        ConcurrentSkipListSet<Long> oldLedgers = new ConcurrentSkipListSet();

        private PagesMapping() {
        }

        private Long getLedgerIdForPage(Long pageId) {
            return this.pages.get(pageId);
        }

        private void removePageId(long pageId) {
            this.pages.remove(pageId);
        }

        private void writePageId(long pageId, long ledgerId) {
            Long oldLedger = this.pages.put(pageId, ledgerId);
            if (oldLedger != null) {
                this.oldLedgers.add(oldLedger);
            }
        }

        public ConcurrentHashMap<Long, Long> getPages() {
            return this.pages;
        }

        public void setPages(ConcurrentHashMap<Long, Long> pages) {
            this.pages = pages;
        }

        public ConcurrentSkipListSet<Long> getOldLedgers() {
            return this.oldLedgers;
        }

        public void setOldLedgers(ConcurrentSkipListSet<Long> oldLedgers) {
            this.oldLedgers = oldLedgers;
        }

        public String toString() {
            return "PagesMapping{pages=" + this.pages + ", oldLedgers=" + this.oldLedgers + '}';
        }
    }

    private class DropLedgerForTableAction
    extends PostCheckpointAction {
        private final long ledgerId;
        private final long pageId;

        public DropLedgerForTableAction(String tableSpace, String tableName, String description, long pageId, long ledgerId) {
            super(tableSpace, tableName, description);
            this.pageId = pageId;
            this.ledgerId = ledgerId;
        }

        @Override
        public void run() {
            BookKeeperDataStorageManager.this.dropLedgerForTable(this.tableSpace, this.tableName, this.pageId, this.ledgerId, this.description);
        }
    }

    private class DeleteZNodeAction
    extends PostCheckpointAction {
        private final String znode;

        public DeleteZNodeAction(String tableSpace, String tableName, String description, String znode) {
            super(tableSpace, tableName, description);
            this.znode = znode;
        }

        @Override
        public void run() {
            try {
                LOGGER.log(Level.FINE, this.description);
                BookKeeperDataStorageManager.this.deleteZNodeEnforceOwnership(this.tableSpace, this.znode);
            }
            catch (DataStorageManagerException err) {
                LOGGER.log(Level.SEVERE, "Could not delete znode " + this.znode + ":" + (Object)((Object)err), (Throwable)((Object)err));
            }
        }
    }

    private class DropLedgerForIndexAction
    extends PostCheckpointAction {
        private final long ledgerId;
        private final long pageId;

        public DropLedgerForIndexAction(String tableSpace, String tableName, String description, long pageId, long ledgerId) {
            super(tableSpace, tableName, description);
            this.pageId = pageId;
            this.ledgerId = ledgerId;
        }

        @Override
        public void run() {
            BookKeeperDataStorageManager.this.dropLedgerForIndex(this.tableSpace, this.tableName, this.pageId, this.ledgerId, this.description);
        }
    }

    private static class RecyclableByteArrayOutputStream
    extends VisibleByteArrayOutputStream {
        private static final int DEFAULT_INITIAL_SIZE = 0x100000;
        private final Recycler.Handle<RecyclableByteArrayOutputStream> handle;
        private boolean closed;

        RecyclableByteArrayOutputStream(Recycler.Handle<RecyclableByteArrayOutputStream> handle) {
            super(0x100000);
            this.handle = handle;
        }

        public void close() {
            if (!this.closed) {
                super.close();
                this.handle.recycle((Object)this);
                this.closed = true;
            }
        }
    }
}

