/*
 * Decompiled with CFR 0.152.
 */
package herddb.client;

import herddb.client.ClientSideConnectionPeer;
import herddb.client.ClientSideMetadataProviderException;
import herddb.client.DMLResult;
import herddb.client.GetResult;
import herddb.client.HDBClient;
import herddb.client.HDBException;
import herddb.client.NonMarshallingClientSideConnectionPeer;
import herddb.client.RoutedClientSideConnection;
import herddb.client.ScanResultSet;
import herddb.client.TableSpaceDumpReceiver;
import herddb.client.TableSpaceRestoreSource;
import herddb.client.impl.LeaderChangedException;
import herddb.client.impl.RetryRequestException;
import herddb.client.impl.UnreachableServerException;
import herddb.network.ServerHostData;
import herddb.utils.Futures;
import herddb.utils.QueryUtils;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.bookkeeper.stats.Counter;

public class HDBConnection
implements AutoCloseable {
    private static final AtomicLong IDGENERATOR = new AtomicLong();
    private final long id = IDGENERATOR.incrementAndGet();
    private final HDBClient client;
    private volatile boolean closed;
    private boolean discoverTablespaceFromSql = true;
    private Counter leaderChangedErrors;
    private final int maxConnectionsPerServer;
    private final Random random = new Random();
    private Map<String, ClientSideConnectionPeer[]> routes;
    private static final Logger LOGGER = Logger.getLogger(HDBConnection.class.getName());

    public HDBConnection(HDBClient client) {
        if (client == null) {
            throw new NullPointerException();
        }
        this.client = client;
        this.leaderChangedErrors = client.getStatsLogger().getCounter("leaderChangedErrors");
        this.maxConnectionsPerServer = client.getConfiguration().getInt("client.maxconnections.perserver", 10);
        this.routes = new ConcurrentHashMap<String, ClientSideConnectionPeer[]>();
    }

    public boolean isDiscoverTablespaceFromSql() {
        return this.discoverTablespaceFromSql;
    }

    public void setDiscoverTablespaceFromSql(boolean discoverTablespaceFromSql) {
        this.discoverTablespaceFromSql = discoverTablespaceFromSql;
    }

    public long getId() {
        return this.id;
    }

    public HDBClient getClient() {
        return this.client;
    }

    @Override
    public void close() {
        LOGGER.log(Level.FINER, "{0} close ", this);
        this.closed = true;
        this.routes.forEach((n, b) -> {
            for (ClientSideConnectionPeer cc : b) {
                cc.close();
            }
        });
        this.routes.clear();
        this.client.releaseConnection(this);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean waitForTableSpace(String tableSpace, int timeout) throws HDBException, ClientSideMetadataProviderException {
        long now;
        long start = System.currentTimeMillis();
        do {
            if (this.closed) return false;
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                try (ScanResultSet result = route.executeScan(tableSpace, "select * from systablespaces where tablespace_name=?", false, Arrays.asList(tableSpace), 0L, 1, 1, false);){
                    boolean ok = result.hasNext();
                    if (ok) {
                        LOGGER.log(Level.INFO, "table space {0} is up now: info {1}", new Object[]{tableSpace, result.consume().get(0)});
                        boolean bl = true;
                        return bl;
                    }
                }
            }
            catch (ClientSideMetadataProviderException | HDBException retry) {
                long now2 = System.currentTimeMillis();
                if (now2 - start > (long)timeout) {
                    return false;
                }
                LOGGER.log(Level.FINE, "tableSpace is still not up " + tableSpace, retry);
                this.handleRetryError(retry, 0);
            }
        } while ((now = System.currentTimeMillis()) - start <= (long)timeout);
        return false;
    }

    public long beginTransaction(String tableSpace) throws ClientSideMetadataProviderException, HDBException {
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                return route.beginTransaction(tableSpace);
            }
            catch (RetryRequestException retry) {
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public void rollbackTransaction(String tableSpace, long tx) throws ClientSideMetadataProviderException, HDBException {
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                route.rollbackTransaction(tableSpace, tx);
                return;
            }
            catch (RetryRequestException retry) {
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public void commitTransaction(String tableSpace, long tx) throws ClientSideMetadataProviderException, HDBException {
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                route.commitTransaction(tableSpace, tx);
                return;
            }
            catch (RetryRequestException retry) {
                LOGGER.log(Level.SEVERE, "error " + retry, retry);
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public DMLResult executeUpdate(String tableSpace, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<Object> params) throws ClientSideMetadataProviderException, HDBException {
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                return route.executeUpdate(tableSpace, query, tx, returnValues, usePreparedStatement, params);
            }
            catch (RetryRequestException retry) {
                LOGGER.log(Level.SEVERE, "error " + retry, retry);
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public CompletableFuture<DMLResult> executeUpdateAsync(String tableSpace, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<Object> params) {
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        if (this.closed) {
            return Futures.exception((Throwable)new HDBException("client is closed"));
        }
        CompletableFuture<DMLResult> res = new CompletableFuture<DMLResult>();
        AtomicInteger count = new AtomicInteger(0);
        this.executeStatementAsyncInternal(tableSpace, res, query, tx, returnValues, usePreparedStatement, params, count);
        return res;
    }

    private void executeStatementAsyncInternal(String tableSpace, CompletableFuture<DMLResult> res, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<Object> params, AtomicInteger count) {
        ClientSideConnectionPeer route;
        try {
            route = this.getRouteToTableSpace(tableSpace);
        }
        catch (ClientSideMetadataProviderException | HDBException err) {
            res.completeExceptionally(err);
            return;
        }
        route.executeUpdateAsync(tableSpace, query, tx, returnValues, usePreparedStatement, params).whenComplete((dmlresult, error) -> {
            if (error != null) {
                if (error instanceof RetryRequestException && !this.closed) {
                    try {
                        this.handleRetryError((Throwable)error, count.getAndIncrement());
                    }
                    catch (ClientSideMetadataProviderException | HDBException err) {
                        res.completeExceptionally(err);
                        return;
                    }
                    LOGGER.log(Level.INFO, "retry #{0} {1}: {2}", new Object[]{count, query, error});
                    this.executeStatementAsyncInternal(tableSpace, res, query, tx, returnValues, usePreparedStatement, params, count);
                } else {
                    res.completeExceptionally((Throwable)error);
                }
            } else {
                res.complete((DMLResult)dmlresult);
            }
        });
    }

    private void executeStatementsAsyncInternal(String tableSpace, CompletableFuture<List<DMLResult>> res, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<List<Object>> params, AtomicInteger count) {
        ClientSideConnectionPeer route;
        try {
            route = this.getRouteToTableSpace(tableSpace);
        }
        catch (ClientSideMetadataProviderException | HDBException err) {
            res.completeExceptionally(err);
            return;
        }
        route.executeUpdatesAsync(tableSpace, query, tx, returnValues, usePreparedStatement, params).whenComplete((dmlresult, error) -> {
            if (error != null) {
                if (error instanceof RetryRequestException && !this.closed) {
                    try {
                        this.handleRetryError((Throwable)error, count.getAndIncrement());
                    }
                    catch (ClientSideMetadataProviderException | HDBException err) {
                        res.completeExceptionally(err);
                        return;
                    }
                    LOGGER.log(Level.INFO, "retry #{0} {1}: {2}", new Object[]{count, query, error});
                    this.executeStatementsAsyncInternal(tableSpace, res, query, tx, returnValues, usePreparedStatement, params, count);
                } else {
                    res.completeExceptionally((Throwable)error);
                }
            } else {
                res.complete((List<DMLResult>)dmlresult);
            }
        });
    }

    public List<DMLResult> executeUpdates(String tableSpace, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<List<Object>> batch) throws ClientSideMetadataProviderException, HDBException {
        if (batch.isEmpty()) {
            return Collections.emptyList();
        }
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                return route.executeUpdates(tableSpace, query, tx, returnValues, usePreparedStatement, batch);
            }
            catch (RetryRequestException retry) {
                LOGGER.log(Level.SEVERE, "error " + retry, retry);
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public CompletableFuture<List<DMLResult>> executeUpdatesAsync(String tableSpace, String query, long tx, boolean returnValues, boolean usePreparedStatement, List<List<Object>> batch) {
        if (batch.isEmpty()) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        if (this.closed) {
            return Futures.exception((Throwable)new HDBException("client is closed"));
        }
        CompletableFuture<List<DMLResult>> res = new CompletableFuture<List<DMLResult>>();
        AtomicInteger count = new AtomicInteger(0);
        this.executeStatementsAsyncInternal(tableSpace, res, query, tx, returnValues, usePreparedStatement, batch, count);
        return res;
    }

    public GetResult executeGet(String tableSpace, String query, long tx, boolean usePreparedStatement, List<Object> params) throws ClientSideMetadataProviderException, HDBException {
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                return route.executeGet(tableSpace, query, tx, usePreparedStatement, params);
            }
            catch (RetryRequestException retry) {
                LOGGER.log(Level.SEVERE, "error " + retry, retry);
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    public ScanResultSet executeScan(String tableSpace, String query, boolean usePreparedStatement, List<Object> params, long tx, int maxRows, int fetchSize, boolean keepReadLocks) throws ClientSideMetadataProviderException, HDBException, InterruptedException {
        if (this.discoverTablespaceFromSql) {
            tableSpace = QueryUtils.discoverTablespace(tableSpace, query);
        }
        int trialCount = 0;
        while (!this.closed) {
            try {
                ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
                return route.executeScan(tableSpace, query, usePreparedStatement, params, tx, maxRows, fetchSize, keepReadLocks);
            }
            catch (RetryRequestException retry) {
                LOGGER.log(Level.INFO, "temporary error", retry);
                this.handleRetryError(retry, trialCount++);
            }
        }
        throw new HDBException("client is closed");
    }

    private void handleRetryError(Throwable retry, int trialCount) throws HDBException, ClientSideMetadataProviderException {
        LOGGER.log(Level.INFO, "retry #{0}:" + retry, trialCount);
        int sleepTimeout = this.client.getOperationRetryDelay();
        int maxTrials = this.client.getMaxOperationRetryCount();
        if (retry instanceof RetryRequestException) {
            RetryRequestException retryError = (RetryRequestException)retry;
            int errorMaxTrials = retryError.getMaxRetry();
            if (errorMaxTrials != -1) {
                maxTrials = errorMaxTrials;
            }
            if (retry instanceof LeaderChangedException) {
                this.leaderChangedErrors.inc();
            }
            if (trialCount > maxTrials) {
                throw new HDBException("Too many trials (" + trialCount + "/" + maxTrials + ") for " + retry, retry);
            }
            if (retryError.isRequireMetadataRefresh()) {
                this.requestMetadataRefresh(retryError);
            }
        }
        try {
            Thread.sleep((trialCount + 1) * sleepTimeout);
        }
        catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw new HDBException(err);
        }
    }

    public void dumpTableSpace(String tableSpace, TableSpaceDumpReceiver receiver, int fetchSize, boolean includeTransactionLog) throws ClientSideMetadataProviderException, HDBException, InterruptedException {
        ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
        route.dumpTableSpace(tableSpace, fetchSize, includeTransactionLog, receiver);
    }

    protected ClientSideConnectionPeer chooseConnection(ClientSideConnectionPeer[] all) {
        return all[this.random.nextInt(this.maxConnectionsPerServer)];
    }

    private ClientSideConnectionPeer getRouteToServer(String nodeId) throws ClientSideMetadataProviderException, HDBException {
        try {
            ClientSideConnectionPeer[] all = this.routes.computeIfAbsent(nodeId, n -> {
                try {
                    ServerHostData serverHostData = this.client.getClientSideMetadataProvider().getServerHostData(nodeId);
                    ClientSideConnectionPeer[] res = new ClientSideConnectionPeer[this.maxConnectionsPerServer];
                    for (int i = 0; i < this.maxConnectionsPerServer; ++i) {
                        RoutedClientSideConnection fullConnection = new RoutedClientSideConnection(this, nodeId, serverHostData);
                        res[i] = this.client.isLocalMode() ? new NonMarshallingClientSideConnectionPeer(fullConnection) : fullConnection;
                    }
                    return res;
                }
                catch (ClientSideMetadataProviderException err) {
                    throw new RuntimeException(err);
                }
            });
            return this.chooseConnection(all);
        }
        catch (RuntimeException err) {
            if (err.getCause() instanceof ClientSideMetadataProviderException) {
                throw (ClientSideMetadataProviderException)err.getCause();
            }
            throw new HDBException(err);
        }
    }

    protected ClientSideConnectionPeer getRouteToTableSpace(String tableSpace) throws ClientSideMetadataProviderException, HDBException {
        if (this.closed) {
            throw new HDBException("connection is closed");
        }
        if (tableSpace == null) {
            throw new HDBException("null tablespace");
        }
        String leaderId = this.client.getClientSideMetadataProvider().getTableSpaceLeader(tableSpace);
        if (leaderId == null) {
            throw new HDBException("no leader found on metadata for tablespace " + tableSpace);
        }
        return this.getRouteToServer(leaderId);
    }

    public boolean isClosed() {
        return this.closed;
    }

    void requestMetadataRefresh(Exception err) throws ClientSideMetadataProviderException {
        UnreachableServerException u;
        String nodeId;
        ClientSideConnectionPeer[] all;
        this.client.getClientSideMetadataProvider().requestMetadataRefresh(err);
        if (err instanceof UnreachableServerException && (all = this.routes.remove(nodeId = (u = (UnreachableServerException)err).getNodeId())) != null) {
            for (ClientSideConnectionPeer con : all) {
                con.close();
            }
        }
    }

    public void restoreTableSpace(String tableSpace, TableSpaceRestoreSource source) throws ClientSideMetadataProviderException, HDBException {
        ClientSideConnectionPeer route = this.getRouteToTableSpace(tableSpace);
        route.restoreTableSpace(tableSpace, source);
    }

    public String toString() {
        return "HDBConnection{routes=" + this.routes + ", id=" + this.id + '}';
    }

    public int hashCode() {
        int hash = 7;
        hash = 41 * hash + (int)(this.id ^ this.id >>> 32);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        HDBConnection other = (HDBConnection)obj;
        return this.id == other.id;
    }
}

