/*
 * Decompiled with CFR 0.152.
 */
package herddb.file;

import herddb.core.HerdDBInternalException;
import herddb.core.MemoryManager;
import herddb.core.PostCheckpointAction;
import herddb.core.RecordSetFactory;
import herddb.file.FileRecordSetFactory;
import herddb.index.KeyToPageIndex;
import herddb.index.blink.BLinkKeyToPageIndex;
import herddb.log.LogSequenceNumber;
import herddb.model.Index;
import herddb.model.Record;
import herddb.model.Table;
import herddb.model.Transaction;
import herddb.server.ServerConfiguration;
import herddb.storage.DataPageDoesNotExistException;
import herddb.storage.DataStorageManager;
import herddb.storage.DataStorageManagerException;
import herddb.storage.FullTableScanConsumer;
import herddb.storage.IndexStatus;
import herddb.storage.TableStatus;
import herddb.utils.ByteArrayCursor;
import herddb.utils.Bytes;
import herddb.utils.CleanDirectoryFileVisitor;
import herddb.utils.DeleteFileVisitor;
import herddb.utils.ExtendedDataInputStream;
import herddb.utils.ExtendedDataOutputStream;
import herddb.utils.FileUtils;
import herddb.utils.ManagedFile;
import herddb.utils.ODirectFileInputStream;
import herddb.utils.ODirectFileOutputStream;
import herddb.utils.OpenFileUtils;
import herddb.utils.SimpleBufferedOutputStream;
import herddb.utils.SimpleByteArrayInputStream;
import herddb.utils.SystemInstrumentation;
import herddb.utils.SystemProperties;
import herddb.utils.VisibleByteArrayOutputStream;
import herddb.utils.XXHash64Utils;
import io.netty.util.Recycler;
import java.io.BufferedInputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;

public class FileDataStorageManager
extends DataStorageManager {
    private static final Logger LOGGER = Logger.getLogger(FileDataStorageManager.class.getName());
    private final Path baseDirectory;
    private final Path tmpDirectory;
    private final int swapThreshold;
    private final boolean requirefsync;
    private final boolean pageodirect;
    private final boolean indexodirect;
    private final StatsLogger logger;
    private final OpStatsLogger dataPageReads;
    private final OpStatsLogger dataPageWrites;
    private final OpStatsLogger indexPageReads;
    private final OpStatsLogger indexPageWrites;
    public static final String FILEEXTENSION_PAGE = ".page";
    public static final int COPY_BUFFERS_SIZE = SystemProperties.getIntSystemProperty((String)"herddb.file.copybuffersize", (int)65536);
    public static final int O_DIRECT_BLOCK_BATCH = SystemProperties.getIntSystemProperty((String)"herddb.file.odirectblockbatch", (int)16);
    public static final String EXTENSION_TABLEORINDExCHECKPOINTINFOFILE = ".checkpoint";
    private static final Recycler<RecyclableByteArrayOutputStream> WRITE_BUFFERS_RECYCLER = new Recycler<RecyclableByteArrayOutputStream>(){

        protected RecyclableByteArrayOutputStream newObject(Recycler.Handle<RecyclableByteArrayOutputStream> handle) {
            return new RecyclableByteArrayOutputStream(handle);
        }
    };

    public FileDataStorageManager(Path baseDirectory) {
        this(baseDirectory, baseDirectory.resolve("tmp"), 10000, ServerConfiguration.PROPERTY_REQUIRE_FSYNC_DEFAULT, ServerConfiguration.PROPERTY_PAGE_USE_ODIRECT_DEFAULT, ServerConfiguration.PROPERTY_INDEX_USE_ODIRECT_DEFAULT, (StatsLogger)new NullStatsLogger());
    }

    public FileDataStorageManager(Path baseDirectory, Path tmpDirectory, int swapThreshold, boolean requirefsync, boolean pageodirect, boolean indexodirect, StatsLogger logger) {
        this.baseDirectory = baseDirectory;
        this.tmpDirectory = tmpDirectory;
        this.swapThreshold = swapThreshold;
        this.logger = logger;
        this.requirefsync = requirefsync;
        this.pageodirect = pageodirect && OpenFileUtils.isO_DIRECT_Supported();
        this.indexodirect = indexodirect && OpenFileUtils.isO_DIRECT_Supported();
        StatsLogger scope = logger.scope("filedatastore");
        this.dataPageReads = scope.getOpStatsLogger("data_pagereads");
        this.dataPageWrites = scope.getOpStatsLogger("data_pagewrites");
        this.indexPageReads = scope.getOpStatsLogger("index_pagereads");
        this.indexPageWrites = scope.getOpStatsLogger("index_pagewrites");
    }

    @Override
    public void start() throws DataStorageManagerException {
        try {
            LOGGER.log(Level.INFO, "ensuring directory {0}", this.baseDirectory.toAbsolutePath().toString());
            Files.createDirectories(this.baseDirectory, new FileAttribute[0]);
            LOGGER.log(Level.INFO, "preparing tmp directory {0}", this.tmpDirectory.toAbsolutePath().toString());
            FileUtils.cleanDirectory((Path)this.tmpDirectory);
            Files.createDirectories(this.tmpDirectory, new FileAttribute[0]);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void close() throws DataStorageManagerException {
        LOGGER.log(Level.INFO, "cleaning tmp directory {0}", this.tmpDirectory.toAbsolutePath().toString());
        try {
            FileUtils.cleanDirectory((Path)this.tmpDirectory);
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Cannot clean tmp directory", err);
        }
    }

    @Override
    public void eraseTablespaceData(String tableSpace) throws DataStorageManagerException {
        SystemInstrumentation.instrumentationPoint((String)"eraseTablespaceData", (Object[])new Object[]{tableSpace});
        Path tablespaceDirectory = this.getTablespaceDirectory(tableSpace);
        LOGGER.log(Level.INFO, "erasing tablespace " + tableSpace + " directory {0}", tablespaceDirectory.toAbsolutePath().toString());
        try {
            FileUtils.cleanDirectory((Path)tablespaceDirectory);
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Cannot clean directory for tablespace " + tableSpace, err);
            throw new DataStorageManagerException(err);
        }
    }

    private Path getTablespaceDirectory(String tablespace) {
        return this.baseDirectory.resolve(tablespace + ".tablespace");
    }

    private Path getTablespaceCheckPointInfoFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTablespaceDirectory(tablespace).resolve("checkpoint." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    private static boolean isTablespaceCheckPointInfoFile(Path path) {
        Path filename = path.getFileName();
        if (filename == null) {
            return false;
        }
        String name = filename.toString();
        return name.startsWith("checkpoint.") && name.endsWith(EXTENSION_TABLEORINDExCHECKPOINTINFOFILE) || name.equals(EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    private Path getTablespaceTablesMetadataFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTablespaceDirectory(tablespace).resolve("tables." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tablesmetadata");
    }

    private static boolean isTablespaceTablesMetadataFile(Path path) {
        Path filename = path.getFileName();
        return filename != null && filename.toString().startsWith("tables.") && filename.toString().endsWith(".tablesmetadata");
    }

    private static boolean isTablespaceIndexesMetadataFile(Path path) {
        Path filename = path.getFileName();
        return filename != null && filename.toString().startsWith("indexes.") && filename.toString().endsWith(".tablesmetadata");
    }

    private Path getTablespaceIndexesMetadataFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTablespaceDirectory(tablespace).resolve("indexes." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tablesmetadata");
    }

    private Path getTablespaceTransactionsFile(String tablespace, LogSequenceNumber sequenceNumber) {
        return this.getTablespaceDirectory(tablespace).resolve("transactions." + sequenceNumber.ledgerId + "." + sequenceNumber.offset + ".tx");
    }

    private static boolean isTransactionsFile(Path path) {
        Path filename = path.getFileName();
        return filename != null && filename.toString().startsWith("transactions.") && filename.toString().endsWith(".tx");
    }

    private Path getTableDirectory(String tableSpace, String tablename) {
        return this.getTablespaceDirectory(tableSpace).resolve(tablename + ".table");
    }

    private Path getIndexDirectory(String tableSpace, String indexname) {
        return this.getTablespaceDirectory(tableSpace).resolve(indexname + ".index");
    }

    private static Path getPageFile(Path tableDirectory, Long pageId) {
        return tableDirectory.resolve(pageId + FILEEXTENSION_PAGE);
    }

    private static Path getTableCheckPointsFile(Path tableDirectory, LogSequenceNumber sequenceNumber) {
        return tableDirectory.resolve(sequenceNumber.ledgerId + "." + sequenceNumber.offset + EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    private static boolean isTableOrIndexCheckpointsFile(Path path) {
        Path filename = path.getFileName();
        return filename != null && filename.toString().endsWith(EXTENSION_TABLEORINDExCHECKPOINTINFOFILE);
    }

    @Override
    public void initIndex(String tableSpace, String uuid) throws DataStorageManagerException {
        Path indexDir = this.getIndexDirectory(tableSpace, uuid);
        LOGGER.log(Level.FINE, "initIndex {0} {1} at {2}", new Object[]{tableSpace, uuid, indexDir});
        try {
            Files.createDirectories(indexDir, new FileAttribute[0]);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void initTable(String tableSpace, String uuid) throws DataStorageManagerException {
        Path tableDir = this.getTableDirectory(tableSpace, uuid);
        LOGGER.log(Level.FINE, "initTable {0} {1} at {2}", new Object[]{tableSpace, uuid, tableDir});
        try {
            Files.createDirectories(tableDir, new FileAttribute[0]);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public List<Record> readPage(String tableSpace, String tableName, Long pageId) throws DataStorageManagerException {
        List<Record> result;
        long _start;
        block20: {
            _start = System.currentTimeMillis();
            Path tableDir = this.getTableDirectory(tableSpace, tableName);
            Path pageFile = FileDataStorageManager.getPageFile(tableDir, pageId);
            try {
                if (this.pageodirect) {
                    try (ODirectFileInputStream odirect = new ODirectFileInputStream(pageFile, O_DIRECT_BLOCK_BATCH);){
                        result = FileDataStorageManager.rawReadDataPage(pageFile, (InputStream)odirect);
                        break block20;
                    }
                }
                try (InputStream input = Files.newInputStream(pageFile, new OpenOption[0]);
                     BufferedInputStream buffer = new BufferedInputStream(input, COPY_BUFFERS_SIZE);){
                    result = FileDataStorageManager.rawReadDataPage(pageFile, buffer);
                }
            }
            catch (NoSuchFileException nsfe) {
                throw new DataPageDoesNotExistException("No such page: " + tableSpace + "_" + tableName + "." + pageId, nsfe);
            }
            catch (IOException err) {
                throw new DataStorageManagerException("error reading data page: " + tableSpace + "_" + tableName + "." + pageId, err);
            }
        }
        long _stop = System.currentTimeMillis();
        long delta = _stop - _start;
        LOGGER.log(Level.FINE, "readPage {0}.{1} {2} ms", new Object[]{tableSpace, tableName, delta + ""});
        this.dataPageReads.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
        return result;
    }

    private static List<Record> rawReadDataPage(Path pageFile, InputStream stream) throws IOException, DataStorageManagerException {
        int size = (int)Files.size(pageFile);
        byte[] dataPage = new byte[size];
        int read = stream.read(dataPage);
        if (read != size) {
            throw new IOException("short read, read " + read + " instead of " + size + " bytes from " + pageFile);
        }
        try (ByteArrayCursor dataIn = ByteArrayCursor.wrap((byte[])dataPage);){
            long version = dataIn.readVLong();
            long flags = dataIn.readVLong();
            if (version != 1L || flags != 0L) {
                throw new DataStorageManagerException("corrupted data file " + pageFile.toAbsolutePath());
            }
            int numRecords = dataIn.readInt();
            ArrayList<Record> result = new ArrayList<Record>(numRecords);
            for (int i = 0; i < numRecords; ++i) {
                Bytes key = dataIn.readBytesNoCopy();
                Bytes value = dataIn.readBytesNoCopy();
                result.add(new Record(key, value));
            }
            int pos = dataIn.getPosition();
            long hashFromFile = dataIn.readLong();
            long hashFromDigest = XXHash64Utils.hash((byte[])dataPage, (int)0, (int)pos);
            if (hashFromDigest != hashFromFile) {
                throw new DataStorageManagerException("Corrupted datafile " + pageFile + ". Bad hash " + hashFromFile + " <> " + hashFromDigest);
            }
            ArrayList<Record> arrayList = result;
            return arrayList;
        }
    }

    public static List<Record> rawReadDataPage(Path pageFile) throws DataStorageManagerException, IOException {
        long hashFromFile;
        long hashFromDigest;
        ArrayList<Record> result;
        try (ODirectFileInputStream odirect = new ODirectFileInputStream(pageFile, O_DIRECT_BLOCK_BATCH);
             XXHash64Utils.HashingStream hash = new XXHash64Utils.HashingStream((InputStream)odirect);
             ExtendedDataInputStream dataIn = new ExtendedDataInputStream((InputStream)hash);){
            long version = dataIn.readVLong();
            long flags = dataIn.readVLong();
            if (version != 1L || flags != 0L) {
                throw new DataStorageManagerException("corrupted data file " + pageFile.toAbsolutePath());
            }
            int numRecords = dataIn.readInt();
            result = new ArrayList<Record>(numRecords);
            for (int i = 0; i < numRecords; ++i) {
                Bytes key = dataIn.readBytes();
                Bytes value = dataIn.readBytes();
                result.add(new Record(key, value));
            }
            hashFromDigest = hash.hash();
            hashFromFile = dataIn.readLong();
        }
        if (hashFromDigest != hashFromFile) {
            throw new DataStorageManagerException("Corrupted datafile " + pageFile + ". Bad hash " + hashFromFile + " <> " + hashFromDigest);
        }
        return result;
    }

    private static <X> X readIndexPage(DataStorageManager.DataReader<X> reader, Path pageFile, InputStream stream) throws IOException, DataStorageManagerException {
        int size = (int)Files.size(pageFile);
        byte[] dataPage = new byte[size];
        int read = stream.read(dataPage);
        if (read != size) {
            throw new IOException("short read, read " + read + " instead of " + size + " bytes from " + pageFile);
        }
        try (ByteArrayCursor dataIn = ByteArrayCursor.wrap((byte[])dataPage);){
            long version = dataIn.readVLong();
            long flags = dataIn.readVLong();
            if (version != 1L || flags != 0L) {
                throw new DataStorageManagerException("corrupted data file " + pageFile.toAbsolutePath());
            }
            X result = reader.read(dataIn);
            int pos = dataIn.getPosition();
            long hashFromFile = dataIn.readLong();
            long hashFromDigest = XXHash64Utils.hash((byte[])dataPage, (int)0, (int)pos);
            if (hashFromDigest != hashFromFile) {
                throw new DataStorageManagerException("Corrupted datafile " + pageFile + ". Bad hash " + hashFromFile + " <> " + hashFromDigest);
            }
            X x = result;
            return x;
        }
    }

    @Override
    public <X> X readIndexPage(String tableSpace, String indexName, Long pageId, DataStorageManager.DataReader<X> reader) throws DataStorageManagerException {
        X result;
        long _start;
        block19: {
            Path tableDir = this.getIndexDirectory(tableSpace, indexName);
            Path pageFile = FileDataStorageManager.getPageFile(tableDir, pageId);
            _start = System.currentTimeMillis();
            try {
                if (this.indexodirect) {
                    try (ODirectFileInputStream odirect = new ODirectFileInputStream(pageFile, O_DIRECT_BLOCK_BATCH);){
                        result = FileDataStorageManager.readIndexPage(reader, pageFile, (InputStream)odirect);
                        break block19;
                    }
                }
                try (InputStream input = Files.newInputStream(pageFile, new OpenOption[0]);
                     BufferedInputStream buffer = new BufferedInputStream(input, COPY_BUFFERS_SIZE);){
                    result = FileDataStorageManager.readIndexPage(reader, pageFile, buffer);
                }
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
        }
        long _stop = System.currentTimeMillis();
        long delta = _stop - _start;
        LOGGER.log(Level.FINE, "readIndexPage {0}.{1} {2} ms", new Object[]{tableSpace, indexName, delta + ""});
        this.indexPageReads.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
        return result;
    }

    @Override
    public void fullTableScan(String tableSpace, String tableName, FullTableScanConsumer consumer) throws DataStorageManagerException {
        try {
            TableStatus status = this.getLatestTableStatus(tableSpace, tableName);
            this.fullTableScan(tableSpace, tableName, status, consumer);
        }
        catch (HerdDBInternalException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void fullTableScan(String tableSpace, String tableUuid, LogSequenceNumber sequenceNumber, FullTableScanConsumer consumer) throws DataStorageManagerException {
        try {
            TableStatus status = this.getTableStatus(tableSpace, tableUuid, sequenceNumber);
            this.fullTableScan(tableSpace, tableUuid, status, consumer);
        }
        catch (HerdDBInternalException err) {
            throw new DataStorageManagerException(err);
        }
    }

    private void fullTableScan(String tableSpace, String tableUuid, TableStatus status, FullTableScanConsumer consumer) {
        LOGGER.log(Level.INFO, "fullTableScan table {0}.{1}, status: {2}", new Object[]{tableSpace, tableUuid, status});
        consumer.acceptTableStatus(status);
        ArrayList<Long> activePages = new ArrayList<Long>(status.activePages.keySet());
        activePages.sort(null);
        Iterator iterator = activePages.iterator();
        while (iterator.hasNext()) {
            long idpage = (Long)iterator.next();
            List<Record> records = this.readPage(tableSpace, tableUuid, idpage);
            LOGGER.log(Level.FINE, "fullTableScan table {0}.{1}, page {2}, contains {3} records", new Object[]{tableSpace, tableUuid, idpage, records.size()});
            consumer.acceptPage(idpage, records);
        }
        consumer.endTable();
    }

    @Override
    public int getActualNumberOfPages(String tableSpace, String tableName) throws DataStorageManagerException {
        TableStatus latestStatus = this.getLatestTableStatus(tableSpace, tableName);
        return latestStatus.activePages.size();
    }

    @Override
    public IndexStatus getIndexStatus(String tableSpace, String indexName, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        Path dir = this.getIndexDirectory(tableSpace, indexName);
        Path checkpointFile = FileDataStorageManager.getTableCheckPointsFile(dir, sequenceNumber);
        if (!Files.exists(checkpointFile, new LinkOption[0])) {
            throw new DataStorageManagerException("no such index checkpoint: " + checkpointFile);
        }
        return FileDataStorageManager.readIndexStatusFromFile(checkpointFile);
    }

    @Override
    public TableStatus getTableStatus(String tableSpace, String tableUuid, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        try {
            Path dir = this.getTableDirectory(tableSpace, tableUuid);
            Path checkpointFile = FileDataStorageManager.getTableCheckPointsFile(dir, sequenceNumber);
            if (!Files.exists(checkpointFile, new LinkOption[0])) {
                throw new DataStorageManagerException("no such table checkpoint: " + checkpointFile);
            }
            return FileDataStorageManager.readTableStatusFromFile(checkpointFile);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public TableStatus getLatestTableStatus(String tableSpace, String tableName) throws DataStorageManagerException {
        try {
            Path lastFile = this.getLastTableCheckpointFile(tableSpace, tableName);
            TableStatus latestStatus = lastFile == null ? new TableStatus(tableName, LogSequenceNumber.START_OF_TIME, Bytes.longToByteArray((long)1L), 1L, Collections.emptyMap()) : FileDataStorageManager.readTableStatusFromFile(lastFile);
            return latestStatus;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    public static TableStatus readTableStatusFromFile(Path checkpointsFile) throws IOException {
        byte[] fileContent = FileUtils.fastReadFile((Path)checkpointsFile);
        XXHash64Utils.verifyBlockWithFooter((byte[])fileContent, (int)0, (int)fileContent.length);
        try (SimpleByteArrayInputStream input = new SimpleByteArrayInputStream(fileContent);){
            TableStatus tableStatus;
            try (ExtendedDataInputStream dataIn = new ExtendedDataInputStream((InputStream)input);){
                long version = dataIn.readVLong();
                long flags = dataIn.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table status file " + checkpointsFile.toAbsolutePath());
                }
                tableStatus = TableStatus.deserialize(dataIn);
            }
            return tableStatus;
        }
    }

    private Path getLastTableCheckpointFile(String tableSpace, String tableName) throws IOException {
        Path dir = this.getTableDirectory(tableSpace, tableName);
        Path result = this.getMostRecentCheckPointFile(dir);
        return result;
    }

    private Path getMostRecentCheckPointFile(Path dir) throws IOException {
        Path result = null;
        long lastMod = -1L;
        Files.createDirectories(dir, new FileAttribute[0]);
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            for (Path path : stream) {
                if (FileDataStorageManager.isTableOrIndexCheckpointsFile(path)) {
                    LOGGER.log(Level.FINER, "getMostRecentCheckPointFile on " + dir.toAbsolutePath() + " -> ACCEPT " + path);
                    FileTime lastModifiedTime = Files.getLastModifiedTime(path, new LinkOption[0]);
                    long ts = lastModifiedTime.toMillis();
                    if (lastMod >= 0L && lastMod >= ts) continue;
                    result = path;
                    lastMod = ts;
                    continue;
                }
                LOGGER.log(Level.FINER, "getMostRecentCheckPointFile on " + dir.toAbsolutePath() + " -> SKIP " + path);
            }
        }
        LOGGER.log(Level.FINER, "getMostRecentCheckPointFile on " + dir.toAbsolutePath() + " -> " + result);
        return result;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static IndexStatus readIndexStatusFromFile(Path checkpointsFile) throws DataStorageManagerException {
        try {
            byte[] fileContent = FileUtils.fastReadFile((Path)checkpointsFile);
            XXHash64Utils.verifyBlockWithFooter((byte[])fileContent, (int)0, (int)fileContent.length);
            try (SimpleByteArrayInputStream input = new SimpleByteArrayInputStream(fileContent);){
                IndexStatus indexStatus;
                try (ExtendedDataInputStream dataIn = new ExtendedDataInputStream((InputStream)input);){
                    long version = dataIn.readVLong();
                    long flags = dataIn.readVLong();
                    if (version != 1L || flags != 0L) {
                        throw new DataStorageManagerException("corrupted index status file " + checkpointsFile.toAbsolutePath());
                    }
                    indexStatus = IndexStatus.deserialize(dataIn);
                }
                return indexStatus;
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public List<PostCheckpointAction> tableCheckpoint(String tableSpace, String tableName, TableStatus tableStatus, boolean pin) throws DataStorageManagerException {
        LogSequenceNumber logPosition = tableStatus.sequenceNumber;
        Path dir = this.getTableDirectory(tableSpace, tableName);
        Path checkpointFile = FileDataStorageManager.getTableCheckPointsFile(dir, logPosition);
        try {
            TableStatus actualStatus;
            if (Files.isRegularFile(checkpointFile, new LinkOption[0]) && (actualStatus = FileDataStorageManager.readTableStatusFromFile(checkpointFile)) != null && actualStatus.equals(tableStatus)) {
                LOGGER.log(Level.FINE, "tableCheckpoint " + tableSpace + ", " + tableName + ": " + tableStatus + " (pin:" + pin + ") already saved on file " + checkpointFile);
                return Collections.emptyList();
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        Path parent = FileDataStorageManager.getParent(checkpointFile);
        Path checkpointFileTemp = parent.resolve(checkpointFile.getFileName() + ".tmp");
        LOGGER.log(Level.FINE, "tableCheckpoint " + tableSpace + ", " + tableName + ": " + tableStatus + " (pin:" + pin + ") to file " + checkpointFile);
        try (ManagedFile file = ManagedFile.open((Path)checkpointFileTemp, (boolean)this.requirefsync);
             SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
             XXHash64Utils.HashingOutputStream oo = new XXHash64Utils.HashingOutputStream((OutputStream)buffer);
             ExtendedDataOutputStream dataOutputKeys = new ExtendedDataOutputStream((OutputStream)oo);){
            dataOutputKeys.writeVLong(1L);
            dataOutputKeys.writeVLong(0L);
            tableStatus.serialize(dataOutputKeys);
            dataOutputKeys.writeLong(oo.hash());
            dataOutputKeys.flush();
            file.sync();
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        try {
            Files.move(checkpointFileTemp, checkpointFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        Map<Long, Integer> pins = this.pinTableAndGetPages(tableSpace, tableName, tableStatus, pin);
        Set<LogSequenceNumber> checkpoints = this.pinTableAndGetCheckpoints(tableSpace, tableName, tableStatus, pin);
        long maxPageId = tableStatus.activePages.keySet().stream().max(Comparator.naturalOrder()).orElse(Long.MAX_VALUE);
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        List<Path> pageFiles = this.getTablePageFiles(tableSpace, tableName);
        for (Path p : pageFiles) {
            long pageId = FileDataStorageManager.getPageId(p);
            LOGGER.log(Level.FINEST, "checkpoint file {0} pageId {1}", new Object[]{p.toAbsolutePath(), pageId});
            if (pageId <= 0L || pins.containsKey(pageId) || tableStatus.activePages.containsKey(pageId) || pageId >= maxPageId) continue;
            LOGGER.log(Level.FINEST, "checkpoint file " + p.toAbsolutePath() + " pageId " + pageId + ". will be deleted after checkpoint end");
            result.add(new DeleteFileAction(tableName, "delete page " + pageId + " file " + p.toAbsolutePath(), p));
        }
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            for (Path p : stream) {
                if (!FileDataStorageManager.isTableOrIndexCheckpointsFile(p) || p.equals(checkpointFile)) continue;
                TableStatus status = FileDataStorageManager.readTableStatusFromFile(p);
                if (!logPosition.after(status.sequenceNumber) || checkpoints.contains(status.sequenceNumber)) continue;
                LOGGER.log(Level.FINEST, "checkpoint metadata file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                result.add(new DeleteFileAction(tableName, "delete checkpoint metadata file " + p.toAbsolutePath(), p));
            }
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Could not list table dir " + dir, err);
        }
        return result;
    }

    @Override
    public List<PostCheckpointAction> indexCheckpoint(String tableSpace, String indexName, IndexStatus indexStatus, boolean pin) throws DataStorageManagerException {
        IndexStatus actualStatus;
        Path dir = this.getIndexDirectory(tableSpace, indexName);
        LogSequenceNumber logPosition = indexStatus.sequenceNumber;
        Path checkpointFile = FileDataStorageManager.getTableCheckPointsFile(dir, logPosition);
        Path parent = FileDataStorageManager.getParent(checkpointFile);
        Path checkpointFileTemp = parent.resolve(checkpointFile.getFileName() + ".tmp");
        if (Files.isRegularFile(checkpointFile, new LinkOption[0]) && (actualStatus = FileDataStorageManager.readIndexStatusFromFile(checkpointFile)) != null && actualStatus.equals(indexStatus)) {
            LOGGER.log(Level.INFO, "indexCheckpoint " + tableSpace + ", " + indexName + ": " + indexStatus + " already saved on" + checkpointFile);
            return Collections.emptyList();
        }
        LOGGER.log(Level.FINE, "indexCheckpoint " + tableSpace + ", " + indexName + ": " + indexStatus + " to file " + checkpointFile);
        try (ManagedFile file = ManagedFile.open((Path)checkpointFileTemp, (boolean)this.requirefsync);
             SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
             XXHash64Utils.HashingOutputStream oo = new XXHash64Utils.HashingOutputStream((OutputStream)buffer);
             ExtendedDataOutputStream dataOutputKeys = new ExtendedDataOutputStream((OutputStream)oo);){
            dataOutputKeys.writeVLong(1L);
            dataOutputKeys.writeVLong(0L);
            indexStatus.serialize(dataOutputKeys);
            dataOutputKeys.writeLong(oo.hash());
            dataOutputKeys.flush();
            file.sync();
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        try {
            Files.move(checkpointFileTemp, checkpointFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        Map<Long, Integer> pins = this.pinIndexAndGetPages(tableSpace, indexName, indexStatus, pin);
        Set<LogSequenceNumber> checkpoints = this.pinIndexAndGetCheckpoints(tableSpace, indexName, indexStatus, pin);
        long maxPageId = indexStatus.activePages.stream().max(Comparator.naturalOrder()).orElse(Long.MAX_VALUE);
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        List<Path> pageFiles = this.getIndexPageFiles(tableSpace, indexName);
        for (Path p : pageFiles) {
            long pageId = FileDataStorageManager.getPageId(p);
            LOGGER.log(Level.FINEST, "checkpoint file {0} pageId {1}", new Object[]{p.toAbsolutePath(), pageId});
            if (pageId <= 0L || pins.containsKey(pageId) || indexStatus.activePages.contains(pageId) || pageId >= maxPageId) continue;
            LOGGER.log(Level.FINEST, "checkpoint file " + p.toAbsolutePath() + " pageId " + pageId + ". will be deleted after checkpoint end");
            result.add(new DeleteFileAction(indexName, "delete page " + pageId + " file " + p.toAbsolutePath(), p));
        }
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            for (Path p : stream) {
                if (!FileDataStorageManager.isTableOrIndexCheckpointsFile(p) || p.equals(checkpointFile)) continue;
                IndexStatus status = FileDataStorageManager.readIndexStatusFromFile(p);
                if (!logPosition.after(status.sequenceNumber) || checkpoints.contains(status.sequenceNumber)) continue;
                LOGGER.log(Level.FINEST, "checkpoint metadata file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                result.add(new DeleteFileAction(indexName, "delete checkpoint metadata file " + p.toAbsolutePath(), p));
            }
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Could not list indexName dir " + dir, err);
        }
        return result;
    }

    private static Path getParent(Path file) throws DataStorageManagerException {
        Path path = file.getParent();
        if (path != null) {
            return path;
        }
        if (file.isAbsolute()) {
            throw new DataStorageManagerException("Invalid path " + file);
        }
        try {
            return FileDataStorageManager.getParent(file.toAbsolutePath());
        }
        catch (IOError | SecurityException e) {
            throw new DataStorageManagerException("Invalid path " + file);
        }
    }

    private static long getPageId(Path p) {
        String filename = p.getFileName() + "";
        if (filename.endsWith(FILEEXTENSION_PAGE)) {
            try {
                return Long.parseLong(filename.substring(0, filename.length() - FILEEXTENSION_PAGE.length()));
            }
            catch (NumberFormatException no) {
                return -1L;
            }
        }
        return -1L;
    }

    private static boolean isPageFile(Path path) {
        return FileDataStorageManager.getPageId(path) >= 0L;
    }

    public List<Path> getTablePageFiles(String tableSpace, String tableName) throws DataStorageManagerException {
        ArrayList<Path> arrayList;
        block8: {
            Path tableDir = this.getTableDirectory(tableSpace, tableName);
            DirectoryStream<Path> files = Files.newDirectoryStream(tableDir, (DirectoryStream.Filter<? super Path>)new DirectoryStream.Filter<Path>(){

                @Override
                public boolean accept(Path entry) throws IOException {
                    return FileDataStorageManager.isPageFile(entry);
                }
            });
            try {
                ArrayList<Path> result = new ArrayList<Path>();
                files.forEach(result::add);
                arrayList = result;
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException err) {
                    throw new DataStorageManagerException(err);
                }
            }
            files.close();
        }
        return arrayList;
    }

    public List<Path> getIndexPageFiles(String tableSpace, String indexName) throws DataStorageManagerException {
        ArrayList<Path> arrayList;
        block8: {
            Path indexDir = this.getIndexDirectory(tableSpace, indexName);
            DirectoryStream<Path> files = Files.newDirectoryStream(indexDir, (DirectoryStream.Filter<? super Path>)new DirectoryStream.Filter<Path>(){

                @Override
                public boolean accept(Path entry) throws IOException {
                    return FileDataStorageManager.isPageFile(entry);
                }
            });
            try {
                ArrayList<Path> result = new ArrayList<Path>();
                files.forEach(result::add);
                arrayList = result;
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException err) {
                    throw new DataStorageManagerException(err);
                }
            }
            files.close();
        }
        return arrayList;
    }

    @Override
    public void cleanupAfterBoot(String tableSpace, String tableName, Set<Long> activePagesAtBoot) throws DataStorageManagerException {
        List<Path> pageFiles = this.getTablePageFiles(tableSpace, tableName);
        for (Path p : pageFiles) {
            long pageId = FileDataStorageManager.getPageId(p);
            LOGGER.log(Level.FINER, "cleanupAfterBoot file " + p.toAbsolutePath() + " pageId " + pageId);
            if (pageId <= 0L || activePagesAtBoot.contains(pageId)) continue;
            LOGGER.log(Level.INFO, "cleanupAfterBoot file " + p.toAbsolutePath() + " pageId " + pageId + ". will be deleted");
            try {
                Files.deleteIfExists(p);
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
        }
    }

    private static long writePage(Collection<Record> newPage, ManagedFile file, OutputStream stream) throws IOException {
        try (RecyclableByteArrayOutputStream oo = FileDataStorageManager.getWriteBuffer();){
            long l;
            try (ExtendedDataOutputStream dataOutput = new ExtendedDataOutputStream((OutputStream)((Object)oo));){
                dataOutput.writeVLong(1L);
                dataOutput.writeVLong(0L);
                dataOutput.writeInt(newPage.size());
                for (Record record : newPage) {
                    dataOutput.writeArray(record.key);
                    dataOutput.writeArray(record.value);
                }
                dataOutput.flush();
                long hash = XXHash64Utils.hash((byte[])oo.getBuffer(), (int)0, (int)oo.size());
                dataOutput.writeLong(hash);
                dataOutput.flush();
                stream.write(oo.getBuffer(), 0, oo.size());
                if (file != null) {
                    file.sync();
                }
                l = oo.size();
            }
            return l;
        }
    }

    @Override
    public void writePage(String tableSpace, String tableName, long pageId, Collection<Record> newPage) throws DataStorageManagerException {
        long size;
        long _start;
        block20: {
            _start = System.currentTimeMillis();
            Path tableDir = this.getTableDirectory(tableSpace, tableName);
            Path pageFile = FileDataStorageManager.getPageFile(tableDir, pageId);
            try {
                if (this.pageodirect) {
                    try (ODirectFileOutputStream odirect = new ODirectFileOutputStream(pageFile, O_DIRECT_BLOCK_BATCH, new OpenOption[]{StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING});){
                        size = FileDataStorageManager.writePage(newPage, null, (OutputStream)odirect);
                        break block20;
                    }
                }
                try (ManagedFile file = ManagedFile.open((Path)pageFile, (boolean)this.requirefsync, (StandardOpenOption[])new StandardOpenOption[]{StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING});
                     SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);){
                    size = FileDataStorageManager.writePage(newPage, file, (OutputStream)buffer);
                }
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
        }
        long now = System.currentTimeMillis();
        long delta = now - _start;
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.log(Level.FINER, "writePage {0} KBytes,{1} records, time {2} ms", new Object[]{size / 1024L + "", newPage.size(), delta + ""});
        }
        this.dataPageWrites.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
    }

    private static long writeIndexPage(DataStorageManager.DataWriter writer, ManagedFile file, OutputStream stream) throws IOException {
        try (RecyclableByteArrayOutputStream oo = FileDataStorageManager.getWriteBuffer();){
            long l;
            try (ExtendedDataOutputStream dataOutput = new ExtendedDataOutputStream((OutputStream)((Object)oo));){
                dataOutput.writeVLong(1L);
                dataOutput.writeVLong(0L);
                writer.write(dataOutput);
                dataOutput.flush();
                long hash = XXHash64Utils.hash((byte[])oo.getBuffer(), (int)0, (int)oo.size());
                dataOutput.writeLong(hash);
                dataOutput.flush();
                stream.write(oo.getBuffer(), 0, oo.size());
                if (file != null) {
                    file.sync();
                }
                l = oo.size();
            }
            return l;
        }
    }

    @Override
    public void writeIndexPage(String tableSpace, String indexName, long pageId, DataStorageManager.DataWriter writer) throws DataStorageManagerException {
        long size;
        long _start;
        block22: {
            _start = System.currentTimeMillis();
            Path tableDir = this.getIndexDirectory(tableSpace, indexName);
            Path pageFile = FileDataStorageManager.getPageFile(tableDir, pageId);
            try {
                if (this.indexodirect) {
                    try (ODirectFileOutputStream odirect = new ODirectFileOutputStream(pageFile, O_DIRECT_BLOCK_BATCH, new OpenOption[]{StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING});){
                        size = FileDataStorageManager.writeIndexPage(writer, null, (OutputStream)odirect);
                        break block22;
                    }
                }
                try (ManagedFile file = ManagedFile.open((Path)pageFile, (boolean)this.requirefsync, (StandardOpenOption[])new StandardOpenOption[]{StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING});
                     SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);){
                    size = FileDataStorageManager.writeIndexPage(writer, file, (OutputStream)buffer);
                }
            }
            catch (IOException err) {
                boolean exists;
                LOGGER.log(Level.SEVERE, "Failed to write on path: {0}", pageFile);
                Path path = pageFile;
                do {
                    if (exists = Files.exists(path, new LinkOption[0])) {
                        LOGGER.log(Level.INFO, "Path {0}: directory {1}, file {2}, link {3}, writable {4}, readable {5}, executable {6}", new Object[]{path, Files.isDirectory(path, new LinkOption[0]), Files.isRegularFile(path, new LinkOption[0]), Files.isSymbolicLink(path), Files.isWritable(path), Files.isReadable(path), Files.isExecutable(path)});
                        continue;
                    }
                    LOGGER.log(Level.INFO, "Path {0} doesn't exists", path);
                } while ((path = path.getParent()) != null && !exists);
                throw new DataStorageManagerException(err);
            }
        }
        long now = System.currentTimeMillis();
        long delta = now - _start;
        if (LOGGER.isLoggable(Level.FINER)) {
            LOGGER.log(Level.FINER, "writePage {0} KBytes, time {2} ms", new Object[]{size / 1024L + "", delta + ""});
        }
        this.indexPageWrites.registerSuccessfulEvent(delta, TimeUnit.MILLISECONDS);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromTablesMetadataFile(String tableSpace, Path file) throws DataStorageManagerException {
        try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table list file " + file.toAbsolutePath());
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromIndexMetadataFile(String tableSpace, Path file) throws DataStorageManagerException {
        try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted index list file " + file.toAbsolutePath());
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public List<Table> loadTables(LogSequenceNumber sequenceNumber, String tableSpace) throws DataStorageManagerException {
        try {
            Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path file = this.getTablespaceTablesMetadataFile(tableSpace, sequenceNumber);
            LOGGER.log(Level.INFO, "loadTables for tableSpace " + tableSpace + " from " + file.toAbsolutePath().toString() + ", sequenceNumber:" + sequenceNumber);
            if (!Files.isRegularFile(file, new LinkOption[0])) {
                if (sequenceNumber.isStartOfTime()) {
                    LOGGER.log(Level.INFO, "file " + file.toAbsolutePath().toString() + " not found");
                    return Collections.emptyList();
                }
                throw new DataStorageManagerException("local table data not available for tableSpace " + tableSpace + ", recovering from sequenceNumber " + sequenceNumber);
            }
            return FileDataStorageManager.readTablespaceStructure(file, tableSpace, sequenceNumber);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    public static List<Table> readTablespaceStructure(Path file, String tableSpace, LogSequenceNumber sequenceNumber) throws IOException, DataStorageManagerException {
        try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);){
            List<Table> list;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted table list file " + file.toAbsolutePath());
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                if (sequenceNumber != null && (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for sequence number " + sequenceNumber);
                }
                int numTables = din.readInt();
                ArrayList<Table> res = new ArrayList<Table>();
                for (int i = 0; i < numTables; ++i) {
                    byte[] tableData = din.readArray();
                    Table table = Table.deserialize(tableData);
                    res.add(table);
                }
                list = Collections.unmodifiableList(res);
            }
            return list;
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public List<Index> loadIndexes(LogSequenceNumber sequenceNumber, String tableSpace) throws DataStorageManagerException {
        try {
            Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path file = this.getTablespaceIndexesMetadataFile(tableSpace, sequenceNumber);
            LOGGER.log(Level.INFO, "loadIndexes for tableSpace " + tableSpace + " from " + file.toAbsolutePath().toString() + ", sequenceNumber:" + sequenceNumber);
            if (!Files.isRegularFile(file, new LinkOption[0])) {
                if (sequenceNumber.isStartOfTime()) {
                    LOGGER.log(Level.INFO, "file " + file.toAbsolutePath().toString() + " not found");
                    return Collections.emptyList();
                }
                throw new DataStorageManagerException("local index data not available for tableSpace " + tableSpace + ", recovering from sequenceNumber " + sequenceNumber);
            }
            try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);){
                List<Index> list;
                try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                    long version = din.readVLong();
                    long flags = din.readVLong();
                    if (version != 1L || flags != 0L) {
                        throw new DataStorageManagerException("corrupted index list file " + file.toAbsolutePath());
                    }
                    String readname = din.readUTF();
                    if (!readname.equals(tableSpace)) {
                        throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                    }
                    long ledgerId = din.readZLong();
                    long offset = din.readZLong();
                    if (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset) {
                        throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for sequence number " + sequenceNumber);
                    }
                    int numTables = din.readInt();
                    ArrayList<Index> res = new ArrayList<Index>();
                    for (int i = 0; i < numTables; ++i) {
                        byte[] indexData = din.readArray();
                        Index table = Index.deserialize(indexData);
                        res.add(table);
                    }
                    list = Collections.unmodifiableList(res);
                }
                return list;
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public Collection<PostCheckpointAction> writeTables(String tableSpace, LogSequenceNumber sequenceNumber, List<Table> tables, List<Index> indexlist, boolean prepareActions) throws DataStorageManagerException {
        if (sequenceNumber.isStartOfTime() && !tables.isEmpty()) {
            throw new DataStorageManagerException("impossible to write a non empty table list at start-of-time");
        }
        Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
        try {
            ExtendedDataOutputStream dout;
            SimpleBufferedOutputStream buffer;
            ManagedFile file;
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path fileTables = this.getTablespaceTablesMetadataFile(tableSpace, sequenceNumber);
            Path fileIndexes = this.getTablespaceIndexesMetadataFile(tableSpace, sequenceNumber);
            Path parent = FileDataStorageManager.getParent(fileTables);
            Files.createDirectories(parent, new FileAttribute[0]);
            LOGGER.log(Level.FINE, "writeTables for tableSpace " + tableSpace + " sequenceNumber " + sequenceNumber + " to " + fileTables.toAbsolutePath().toString());
            try {
                file = ManagedFile.open((Path)fileTables, (boolean)this.requirefsync);
                try {
                    buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
                    try {
                        dout = new ExtendedDataOutputStream((OutputStream)buffer);
                        try {
                            dout.writeVLong(1L);
                            dout.writeVLong(0L);
                            dout.writeUTF(tableSpace);
                            dout.writeZLong(sequenceNumber.ledgerId);
                            dout.writeZLong(sequenceNumber.offset);
                            dout.writeInt(tables.size());
                            for (Table table : tables) {
                                byte[] tableSerialized = table.serialize();
                                dout.writeArray(tableSerialized);
                            }
                            dout.flush();
                            file.sync();
                        }
                        finally {
                            dout.close();
                        }
                    }
                    finally {
                        buffer.close();
                    }
                }
                finally {
                    if (file != null) {
                        file.close();
                    }
                }
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
            try {
                file = ManagedFile.open((Path)fileIndexes, (boolean)this.requirefsync);
                try {
                    buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
                    try {
                        dout = new ExtendedDataOutputStream((OutputStream)buffer);
                        try {
                            dout.writeVLong(1L);
                            dout.writeVLong(0L);
                            dout.writeUTF(tableSpace);
                            dout.writeZLong(sequenceNumber.ledgerId);
                            dout.writeZLong(sequenceNumber.offset);
                            if (indexlist != null) {
                                dout.writeInt(indexlist.size());
                                for (Index index : indexlist) {
                                    byte[] indexSerialized = index.serialize();
                                    dout.writeArray(indexSerialized);
                                }
                            } else {
                                dout.writeInt(0);
                            }
                            dout.flush();
                            file.sync();
                        }
                        finally {
                            dout.close();
                        }
                    }
                    finally {
                        buffer.close();
                    }
                }
                finally {
                    if (file != null) {
                        file.close();
                    }
                }
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        if (prepareActions) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(tableSpaceDirectory);){
                for (Path p : stream) {
                    LogSequenceNumber logPositionInFile;
                    if (FileDataStorageManager.isTablespaceIndexesMetadataFile(p)) {
                        try {
                            logPositionInFile = FileDataStorageManager.readLogSequenceNumberFromIndexMetadataFile(tableSpace, p);
                            if (!sequenceNumber.after(logPositionInFile)) continue;
                            LOGGER.log(Level.FINEST, "indexes metadata file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                            result.add(new DeleteFileAction("indexes", "delete indexesmetadata file " + p.toAbsolutePath(), p));
                        }
                        catch (DataStorageManagerException ignore) {
                            LOGGER.log(Level.SEVERE, "Unparsable indexesmetadata file " + p.toAbsolutePath(), (Throwable)((Object)ignore));
                            result.add(new DeleteFileAction("indexes", "delete unparsable indexesmetadata file " + p.toAbsolutePath(), p));
                        }
                        continue;
                    }
                    if (!FileDataStorageManager.isTablespaceTablesMetadataFile(p)) continue;
                    try {
                        logPositionInFile = FileDataStorageManager.readLogSequenceNumberFromTablesMetadataFile(tableSpace, p);
                        if (!sequenceNumber.after(logPositionInFile)) continue;
                        LOGGER.log(Level.FINEST, "tables metadata file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                        result.add(new DeleteFileAction("tables", "delete tablesmetadata file " + p.toAbsolutePath(), p));
                    }
                    catch (DataStorageManagerException ignore) {
                        LOGGER.log(Level.SEVERE, "Unparsable tablesmetadata file " + p.toAbsolutePath(), (Throwable)((Object)ignore));
                        result.add(new DeleteFileAction("transactions", "delete unparsable tablesmetadata file " + p.toAbsolutePath(), p));
                    }
                }
            }
            catch (IOException err) {
                LOGGER.log(Level.SEVERE, "Could not list dir " + tableSpaceDirectory, err);
            }
        }
        return result;
    }

    @Override
    public Collection<PostCheckpointAction> writeCheckpointSequenceNumber(String tableSpace, LogSequenceNumber sequenceNumber) throws DataStorageManagerException {
        Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
        try {
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path checkPointFile = this.getTablespaceCheckPointInfoFile(tableSpace, sequenceNumber);
            Path parent = FileDataStorageManager.getParent(checkPointFile);
            Files.createDirectories(parent, new FileAttribute[0]);
            Path checkpointFileTemp = parent.resolve(checkPointFile.getFileName() + ".tmp");
            LOGGER.log(Level.INFO, "checkpoint for " + tableSpace + " at " + sequenceNumber + " to " + checkPointFile.toAbsolutePath().toString());
            try (ManagedFile file = ManagedFile.open((Path)checkpointFileTemp, (boolean)this.requirefsync);
                 SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
                 ExtendedDataOutputStream dout = new ExtendedDataOutputStream((OutputStream)buffer);){
                dout.writeVLong(1L);
                dout.writeVLong(0L);
                dout.writeUTF(tableSpace);
                dout.writeZLong(sequenceNumber.ledgerId);
                dout.writeZLong(sequenceNumber.offset);
                dout.flush();
                file.sync();
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
            Files.move(checkpointFileTemp, checkPointFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(tableSpaceDirectory);){
            for (Path p : stream) {
                if (!FileDataStorageManager.isTablespaceCheckPointInfoFile(p)) continue;
                try {
                    LogSequenceNumber logPositionInFile = FileDataStorageManager.readLogSequenceNumberFromCheckpointInfoFile(tableSpace, p);
                    if (!sequenceNumber.after(logPositionInFile)) continue;
                    LOGGER.log(Level.FINEST, "checkpoint info file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                    result.add(new DeleteFileAction("checkpoint", "delete checkpoint info file " + p.toAbsolutePath(), p));
                }
                catch (DataStorageManagerException ignore) {
                    LOGGER.log(Level.SEVERE, "unparsable checkpoint info file " + p.toAbsolutePath(), (Throwable)((Object)ignore));
                }
            }
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Could not list dir " + tableSpaceDirectory, err);
        }
        return result;
    }

    @Override
    public void dropTable(String tablespace, String tableName) throws DataStorageManagerException {
        Path tableDir = this.getTableDirectory(tablespace, tableName);
        LOGGER.log(Level.INFO, "dropTable {0}.{1} in {2}", new Object[]{tablespace, tableName, tableDir});
        try {
            FileDataStorageManager.deleteDirectory(tableDir);
        }
        catch (IOException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    @Override
    public void truncateIndex(String tablespace, String name) throws DataStorageManagerException {
        Path tableDir = this.getIndexDirectory(tablespace, name);
        LOGGER.log(Level.INFO, "truncateIndex {0}.{1} in {2}", new Object[]{tablespace, name, tableDir});
        try {
            FileDataStorageManager.cleanDirectory(tableDir);
        }
        catch (IOException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    @Override
    public void dropIndex(String tablespace, String name) throws DataStorageManagerException {
        Path tableDir = this.getIndexDirectory(tablespace, name);
        LOGGER.log(Level.INFO, "dropIndex {0}.{1} in {2}", new Object[]{tablespace, name, tableDir});
        try {
            FileDataStorageManager.deleteDirectory(tableDir);
        }
        catch (IOException ex) {
            throw new DataStorageManagerException(ex);
        }
    }

    private static LogSequenceNumber readLogSequenceNumberFromCheckpointInfoFile(String tableSpace, Path checkPointFile) throws DataStorageManagerException, IOException {
        try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(checkPointFile, StandardOpenOption.READ), 0x400000);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new IOException("corrupted checkpoint file");
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + checkPointFile.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
    }

    @Override
    public LogSequenceNumber getLastcheckpointSequenceNumber(String tableSpace) throws DataStorageManagerException {
        try {
            Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            LogSequenceNumber max = LogSequenceNumber.START_OF_TIME;
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(tableSpaceDirectory);){
                for (Path p : stream) {
                    if (!FileDataStorageManager.isTablespaceCheckPointInfoFile(p)) continue;
                    try {
                        LogSequenceNumber logPositionInFile = FileDataStorageManager.readLogSequenceNumberFromCheckpointInfoFile(tableSpace, p);
                        if (!logPositionInFile.after(max)) continue;
                        max = logPositionInFile;
                    }
                    catch (DataStorageManagerException ignore) {
                        LOGGER.log(Level.SEVERE, "unparsable checkpoint info file " + p.toAbsolutePath(), (Throwable)((Object)ignore));
                    }
                }
            }
            catch (IOException err) {
                LOGGER.log(Level.SEVERE, "Could not list dir " + tableSpaceDirectory, err);
            }
            return max;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    public static void deleteDirectory(Path f) throws IOException {
        FileDataStorageManager.deleteDirectoryContent(f, false);
    }

    public static void cleanDirectory(Path f) throws IOException {
        FileDataStorageManager.deleteDirectoryContent(f, true);
    }

    private static void deleteDirectoryContent(Path f, boolean keepDirectory) throws IOException {
        if (Files.isDirectory(f, new LinkOption[0])) {
            SimpleFileVisitor visitor = keepDirectory ? new CleanDirectoryFileVisitor() : DeleteFileVisitor.INSTANCE;
            Files.walkFileTree(f, visitor);
        } else if (Files.isRegularFile(f, new LinkOption[0])) {
            throw new IOException("name " + f.toAbsolutePath() + " is not a directory");
        }
    }

    @Override
    public KeyToPageIndex createKeyToPageMap(String tablespace, String name, MemoryManager memoryManager) throws DataStorageManagerException {
        return new BLinkKeyToPageIndex(tablespace, name, memoryManager, this);
    }

    @Override
    public void releaseKeyToPageMap(String tablespace, String name, KeyToPageIndex keyToPage) {
        if (keyToPage != null) {
            keyToPage.close();
        }
    }

    @Override
    public RecordSetFactory createRecordSetFactory() {
        return new FileRecordSetFactory(this.tmpDirectory, this.swapThreshold);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static LogSequenceNumber readLogSequenceNumberFromTransactionsFile(String tableSpace, Path file) throws DataStorageManagerException {
        try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);){
            LogSequenceNumber logSequenceNumber;
            try (ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted transaction list file " + file.toAbsolutePath());
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                logSequenceNumber = new LogSequenceNumber(ledgerId, offset);
            }
            return logSequenceNumber;
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public void loadTransactions(LogSequenceNumber sequenceNumber, String tableSpace, Consumer<Transaction> consumer) throws DataStorageManagerException {
        try {
            Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path file = this.getTablespaceTransactionsFile(tableSpace, sequenceNumber);
            boolean exists = Files.isRegularFile(file, new LinkOption[0]);
            LOGGER.log(Level.INFO, "loadTransactions " + sequenceNumber + " for tableSpace " + tableSpace + " from file " + file + " (exists: " + exists + ")");
            if (!exists) {
                return;
            }
            try (BufferedInputStream input = new BufferedInputStream(Files.newInputStream(file, StandardOpenOption.READ), 0x400000);
                 ExtendedDataInputStream din = new ExtendedDataInputStream((InputStream)input);){
                long version = din.readVLong();
                long flags = din.readVLong();
                if (version != 1L || flags != 0L) {
                    throw new DataStorageManagerException("corrupted transaction list file " + file.toAbsolutePath());
                }
                String readname = din.readUTF();
                if (!readname.equals(tableSpace)) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for spablespace " + tableSpace);
                }
                long ledgerId = din.readZLong();
                long offset = din.readZLong();
                if (ledgerId != sequenceNumber.ledgerId || offset != sequenceNumber.offset) {
                    throw new DataStorageManagerException("file " + file.toAbsolutePath() + " is not for sequence number " + sequenceNumber);
                }
                int numTransactions = din.readInt();
                for (int i = 0; i < numTransactions; ++i) {
                    Transaction tx = Transaction.deserialize(tableSpace, din);
                    consumer.accept(tx);
                }
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
    }

    @Override
    public Collection<PostCheckpointAction> writeTransactionsAtCheckpoint(String tableSpace, LogSequenceNumber sequenceNumber, Collection<Transaction> transactions) throws DataStorageManagerException {
        if (sequenceNumber.isStartOfTime() && !transactions.isEmpty()) {
            throw new DataStorageManagerException("impossible to write a non empty transactions list at start-of-time");
        }
        Path tableSpaceDirectory = this.getTablespaceDirectory(tableSpace);
        try {
            Files.createDirectories(tableSpaceDirectory, new FileAttribute[0]);
            Path checkPointFile = this.getTablespaceTransactionsFile(tableSpace, sequenceNumber);
            Path parent = FileDataStorageManager.getParent(checkPointFile);
            Files.createDirectories(parent, new FileAttribute[0]);
            Path checkpointFileTemp = parent.resolve(checkPointFile.getFileName() + ".tmp");
            LOGGER.log(Level.FINE, "writeTransactionsAtCheckpoint for tableSpace {0} sequenceNumber {1} to {2}, active transactions {3}", new Object[]{tableSpace, sequenceNumber, checkPointFile.toAbsolutePath().toString(), transactions.size()});
            try (ManagedFile file = ManagedFile.open((Path)checkpointFileTemp, (boolean)this.requirefsync);
                 SimpleBufferedOutputStream buffer = new SimpleBufferedOutputStream(file.getOutputStream(), COPY_BUFFERS_SIZE);
                 ExtendedDataOutputStream dout = new ExtendedDataOutputStream((OutputStream)buffer);){
                dout.writeVLong(1L);
                dout.writeVLong(0L);
                dout.writeUTF(tableSpace);
                dout.writeZLong(sequenceNumber.ledgerId);
                dout.writeZLong(sequenceNumber.offset);
                dout.writeInt(transactions.size());
                for (Transaction t : transactions) {
                    t.serialize(dout);
                }
                dout.flush();
                file.sync();
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
            try {
                Files.move(checkpointFileTemp, checkPointFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException err) {
                throw new DataStorageManagerException(err);
            }
        }
        catch (IOException err) {
            throw new DataStorageManagerException(err);
        }
        ArrayList<PostCheckpointAction> result = new ArrayList<PostCheckpointAction>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(tableSpaceDirectory);){
            for (Path p : stream) {
                if (!FileDataStorageManager.isTransactionsFile(p)) continue;
                try {
                    LogSequenceNumber logPositionInFile = FileDataStorageManager.readLogSequenceNumberFromTransactionsFile(tableSpace, p);
                    if (!sequenceNumber.after(logPositionInFile)) continue;
                    LOGGER.log(Level.FINEST, "transactions metadata file " + p.toAbsolutePath() + ". will be deleted after checkpoint end");
                    result.add(new DeleteFileAction("transactions", "delete transactions file " + p.toAbsolutePath(), p));
                }
                catch (DataStorageManagerException ignore) {
                    LOGGER.log(Level.SEVERE, "Unparsable transactions file " + p.toAbsolutePath(), (Throwable)((Object)ignore));
                    result.add(new DeleteFileAction("transactions", "delete unparsable transactions file " + p.toAbsolutePath(), p));
                }
            }
        }
        catch (IOException err) {
            LOGGER.log(Level.SEVERE, "Could not list dir " + tableSpaceDirectory, err);
        }
        return result;
    }

    private static RecyclableByteArrayOutputStream getWriteBuffer() {
        RecyclableByteArrayOutputStream res = (RecyclableByteArrayOutputStream)((Object)WRITE_BUFFERS_RECYCLER.get());
        res.closed = false;
        res.reset();
        return res;
    }

    private static class DeleteFileAction
    extends PostCheckpointAction {
        private final Path p;

        public DeleteFileAction(String tableName, String description, Path p) {
            super(tableName, description);
            this.p = p;
        }

        @Override
        public void run() {
            try {
                LOGGER.log(Level.FINE, this.description);
                Files.deleteIfExists(this.p);
            }
            catch (IOException err) {
                LOGGER.log(Level.SEVERE, "Could not delete file " + this.p.toAbsolutePath() + ":" + err, err);
            }
        }
    }

    private static class RecyclableByteArrayOutputStream
    extends VisibleByteArrayOutputStream {
        private static final int DEFAULT_INITIAL_SIZE = 0x100000;
        private final Recycler.Handle<RecyclableByteArrayOutputStream> handle;
        private boolean closed;

        RecyclableByteArrayOutputStream(Recycler.Handle<RecyclableByteArrayOutputStream> handle) {
            super(0x100000);
            this.handle = handle;
        }

        public void close() {
            if (!this.closed) {
                super.close();
                this.handle.recycle((Object)this);
                this.closed = true;
            }
        }
    }
}

