/*
 * Decompiled with CFR 0.152.
 */
package herddb.cluster;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import herddb.cluster.LedgersInfo;
import herddb.log.LogNotAvailableException;
import herddb.metadata.MetadataStorageManager;
import herddb.metadata.MetadataStorageManagerException;
import herddb.model.DDLException;
import herddb.model.NodeMetadata;
import herddb.model.TableSpace;
import herddb.model.TableSpaceAlreadyExistsException;
import herddb.model.TableSpaceDoesNotExistException;
import herddb.model.TableSpaceReplicaState;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

public class ZookeeperMetadataStorageManager
extends MetadataStorageManager {
    private static final Logger LOGGER = Logger.getLogger(ZookeeperMetadataStorageManager.class.getName());
    private ZooKeeper zooKeeper;
    private final String zkAddress;
    private final int zkSessionTimeout;
    private final String basePath;
    private final String ledgersPath;
    private final String tableSpacesPath;
    private final String tableSpacesReplicasPath;
    private final String nodesPath;
    private volatile boolean started;
    private final Watcher mainWatcher = event -> {
        switch (event.getState()) {
            case SyncConnected: 
            case SaslAuthenticated: {
                this.notifyMetadataChanged("zkevent " + event.getState() + " " + event.getType() + " " + event.getPath());
                break;
            }
        }
    };

    public String getZkAddress() {
        return this.zkAddress;
    }

    public int getZkSessionTimeout() {
        return this.zkSessionTimeout;
    }

    private synchronized void restartZooKeeper() throws IOException, InterruptedException {
        ZooKeeper old = this.zooKeeper;
        if (old != null) {
            old.close();
        }
        final CountDownLatch firstConnectionLatch = new CountDownLatch(1);
        ZooKeeper zk = new ZooKeeper(this.zkAddress, this.zkSessionTimeout, new Watcher(){

            public void process(WatchedEvent event) {
                switch (event.getState()) {
                    case SyncConnected: 
                    case SaslAuthenticated: {
                        firstConnectionLatch.countDown();
                        ZookeeperMetadataStorageManager.this.notifyMetadataChanged("zkevent " + event.getState() + " " + event.getType() + " " + event.getPath());
                        break;
                    }
                }
            }
        });
        if (!firstConnectionLatch.await(this.zkSessionTimeout, TimeUnit.SECONDS)) {
            zk.close();
            throw new IOException("Could not connect to zookeeper at " + this.zkAddress + " within " + this.zkSessionTimeout + " ms");
        }
        this.zooKeeper = zk;
        LOGGER.info("Connected to ZK " + zk);
    }

    private void handleSessionExpiredError(Throwable error) {
        if (!(error instanceof KeeperException.SessionExpiredException)) {
            return;
        }
        try {
            this.restartZooKeeper();
        }
        catch (IOException | InterruptedException err) {
            LOGGER.log(Level.SEVERE, "Error handling session expired", err);
        }
    }

    public ZookeeperMetadataStorageManager(String zkAddress, int zkSessionTimeout, String basePath) {
        this.zkAddress = zkAddress;
        this.zkSessionTimeout = zkSessionTimeout;
        this.basePath = basePath;
        this.ledgersPath = basePath + "/ledgers";
        this.tableSpacesPath = basePath + "/tableSpaces";
        this.tableSpacesReplicasPath = basePath + "/replicas";
        this.nodesPath = basePath + "/nodes";
    }

    @Override
    @SuppressFBWarnings(value={"RV_RETURN_VALUE_IGNORED"})
    public void start() throws MetadataStorageManagerException {
        this.start(true);
    }

    public synchronized void start(boolean formatIfNeeded) throws MetadataStorageManagerException {
        if (this.started) {
            return;
        }
        LOGGER.log(Level.SEVERE, "start, zkAddress " + this.zkAddress + ", zkSessionTimeout:" + this.zkSessionTimeout + ", basePath:" + this.basePath);
        try {
            this.restartZooKeeper();
            if (formatIfNeeded) {
                this.ensureRoot();
            }
            this.started = true;
        }
        catch (IOException | InterruptedException | KeeperException err) {
            throw new MetadataStorageManagerException(err);
        }
    }

    private void ensureRoot() throws KeeperException, InterruptedException, IOException {
        try {
            this.zooKeeper.create(this.basePath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            // empty catch block
        }
        try {
            this.zooKeeper.create(this.tableSpacesPath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            // empty catch block
        }
        try {
            this.zooKeeper.create(this.tableSpacesReplicasPath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            // empty catch block
        }
        try {
            this.zooKeeper.create(this.ledgersPath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            // empty catch block
        }
        try {
            this.zooKeeper.create(this.nodesPath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            // empty catch block
        }
    }

    public synchronized ZooKeeper getZooKeeper() {
        return this.zooKeeper;
    }

    private synchronized ZooKeeper ensureZooKeeper() throws KeeperException, InterruptedException, IOException {
        if (!this.started) {
            throw new IOException("MetadataStorageManager not yet started");
        }
        if (this.zooKeeper == null) {
            this.restartZooKeeper();
        }
        return this.zooKeeper;
    }

    public static LedgersInfo readActualLedgersListFromZookeeper(ZooKeeper zooKeeper, String ledgersPath, String tableSpaceUUID) throws LogNotAvailableException {
        while (zooKeeper.getState() != ZooKeeper.States.CLOSED) {
            try {
                Stat stat = new Stat();
                byte[] actualLedgers = zooKeeper.getData(ledgersPath + "/" + tableSpaceUUID, false, stat);
                return LedgersInfo.deserialize(actualLedgers, stat.getVersion());
            }
            catch (KeeperException.NoNodeException firstboot) {
                LOGGER.log(Level.INFO, "node " + ledgersPath + "/" + tableSpaceUUID + " not found");
                return LedgersInfo.deserialize(null, -1);
            }
            catch (KeeperException.ConnectionLossException error) {
                LOGGER.log(Level.SEVERE, "error while loading actual ledgers list at " + ledgersPath + "/" + tableSpaceUUID, error);
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException err) {
                    throw new LogNotAvailableException(err);
                }
            }
            catch (Exception error) {
                LOGGER.log(Level.SEVERE, "error while loading actual ledgers list at " + ledgersPath + "/" + tableSpaceUUID, error);
                throw new LogNotAvailableException(error);
            }
        }
        throw new LogNotAvailableException(new Exception("zk client closed"));
    }

    public LedgersInfo getActualLedgersList(String tableSpaceUUID) throws LogNotAvailableException {
        try {
            return ZookeeperMetadataStorageManager.readActualLedgersListFromZookeeper(this.ensureZooKeeper(), this.ledgersPath, tableSpaceUUID);
        }
        catch (IOException | InterruptedException | KeeperException err) {
            throw new LogNotAvailableException(err);
        }
    }

    public void saveActualLedgersList(String tableSpaceUUID, LedgersInfo info) throws LogNotAvailableException {
        byte[] actualLedgers = info.serialize();
        try {
            while (true) {
                try {
                    while (true) {
                        try {
                            Stat newStat = this.ensureZooKeeper().setData(this.ledgersPath + "/" + tableSpaceUUID, actualLedgers, info.getZkVersion());
                            info.setZkVersion(newStat.getVersion());
                            LOGGER.log(Level.SEVERE, "save new ledgers list " + info + " to " + this.ledgersPath + "/" + tableSpaceUUID);
                            return;
                        }
                        catch (KeeperException.NoNodeException firstboot) {
                            this.ensureZooKeeper().create(this.ledgersPath + "/" + tableSpaceUUID, actualLedgers, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
                            continue;
                        }
                        catch (KeeperException.BadVersionException fenced) {
                            throw new LogNotAvailableException(new Exception("ledgers actual list was fenced, expecting version " + info.getZkVersion() + " " + (Object)((Object)fenced), fenced).fillInStackTrace());
                        }
                        break;
                    }
                }
                catch (KeeperException.ConnectionLossException anyError) {
                    LOGGER.log(Level.SEVERE, "temporary error", anyError);
                    Thread.sleep(10000L);
                    continue;
                }
                catch (Exception anyError) {
                    this.handleSessionExpiredError(anyError);
                    throw new LogNotAvailableException(anyError);
                }
                break;
            }
        }
        catch (InterruptedException stop) {
            LOGGER.log(Level.SEVERE, "fatal error", stop);
            throw new LogNotAvailableException(stop);
        }
    }

    private TableSpaceList listTablesSpaces() throws KeeperException, InterruptedException, IOException {
        Stat stat = new Stat();
        List children = this.ensureZooKeeper().getChildren(this.tableSpacesPath, this.mainWatcher, stat);
        return new TableSpaceList(stat.getVersion(), children);
    }

    private void createTableSpaceNode(TableSpace tableSpace) throws KeeperException, InterruptedException, IOException, TableSpaceAlreadyExistsException {
        try {
            this.ensureZooKeeper().create(this.tableSpacesPath + "/" + tableSpace.name.toLowerCase(), tableSpace.serialize(), (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
        catch (KeeperException.NodeExistsException err) {
            throw new TableSpaceAlreadyExistsException(tableSpace.uuid);
        }
    }

    private boolean updateTableSpaceNode(TableSpace tableSpace, int metadataStorageVersion) throws KeeperException, InterruptedException, IOException, TableSpaceDoesNotExistException {
        try {
            this.ensureZooKeeper().setData(this.tableSpacesPath + "/" + tableSpace.name.toLowerCase(), tableSpace.serialize(), metadataStorageVersion);
            this.notifyMetadataChanged("updateTableSpaceNode " + tableSpace + " metadataStorageVersion " + metadataStorageVersion);
            return true;
        }
        catch (KeeperException.BadVersionException changed) {
            return false;
        }
        catch (KeeperException.NoNodeException changed) {
            throw new TableSpaceDoesNotExistException(tableSpace.uuid);
        }
    }

    private boolean deleteTableSpaceNode(String tableSpaceName, int metadataStorageVersion) throws KeeperException, InterruptedException, IOException, TableSpaceDoesNotExistException {
        try {
            this.ensureZooKeeper().delete(this.tableSpacesPath + "/" + tableSpaceName.toLowerCase(), metadataStorageVersion);
            this.notifyMetadataChanged("deleteTableSpaceNode " + tableSpaceName + " metadataStorageVersion " + metadataStorageVersion);
            return true;
        }
        catch (KeeperException.BadVersionException changed) {
            return false;
        }
        catch (KeeperException.NoNodeException changed) {
            throw new TableSpaceDoesNotExistException(tableSpaceName);
        }
    }

    @Override
    public void ensureDefaultTableSpace(String localNodeId) throws MetadataStorageManagerException {
        try {
            TableSpaceList list = this.listTablesSpaces();
            if (!list.tableSpaces.contains("herd")) {
                TableSpace tableSpace = TableSpace.builder().leader(localNodeId).replica(localNodeId).expectedReplicaCount(1).maxLeaderInactivityTime(0L).name("herd").build();
                this.createTableSpaceNode(tableSpace);
            }
        }
        catch (TableSpaceAlreadyExistsException list) {
        }
        catch (IOException | InterruptedException | KeeperException err) {
            this.handleSessionExpiredError(err);
            throw new MetadataStorageManagerException(err);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws MetadataStorageManagerException {
        try {
            ZookeeperMetadataStorageManager zookeeperMetadataStorageManager = this;
            synchronized (zookeeperMetadataStorageManager) {
                if (this.zooKeeper == null) {
                    return;
                }
            }
            ZooKeeper zk = this.ensureZooKeeper();
            if (zk != null) {
                try {
                    zk.close();
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        catch (IOException | InterruptedException | KeeperException throwable) {
            // empty catch block
        }
    }

    @Override
    public Collection<String> listTableSpaces() throws MetadataStorageManagerException {
        try {
            return this.listTablesSpaces().tableSpaces;
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            this.handleSessionExpiredError(ex);
            throw new MetadataStorageManagerException(ex);
        }
    }

    @Override
    public TableSpace describeTableSpace(String name) throws MetadataStorageManagerException {
        name = name.toLowerCase();
        try {
            Stat stat = new Stat();
            byte[] result = this.ensureZooKeeper().getData(this.tableSpacesPath + "/" + name.toLowerCase(), this.mainWatcher, stat);
            return TableSpace.deserialize(result, (Object)stat.getVersion(), stat.getCtime());
        }
        catch (KeeperException.NoNodeException ex) {
            return null;
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            this.handleSessionExpiredError(ex);
            throw new MetadataStorageManagerException(ex);
        }
    }

    @Override
    public void registerTableSpace(TableSpace tableSpace) throws DDLException, MetadataStorageManagerException {
        try {
            this.createTableSpaceNode(tableSpace);
            this.notifyMetadataChanged("registerTableSpace " + tableSpace);
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            this.handleSessionExpiredError(ex);
            throw new MetadataStorageManagerException(ex);
        }
    }

    @Override
    public boolean updateTableSpace(TableSpace tableSpace, TableSpace previous) throws DDLException, MetadataStorageManagerException {
        if (previous == null || previous.metadataStorageVersion == null) {
            throw new MetadataStorageManagerException("metadataStorageVersion not read from ZK");
        }
        try {
            boolean result = this.updateTableSpaceNode(tableSpace, (Integer)previous.metadataStorageVersion);
            if (result) {
                this.notifyMetadataChanged("updateTableSpace " + tableSpace);
            }
            return result;
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            this.handleSessionExpiredError(ex);
            throw new MetadataStorageManagerException(ex);
        }
    }

    @Override
    public void dropTableSpace(String name, TableSpace previous) throws DDLException, MetadataStorageManagerException {
        if (previous == null || previous.metadataStorageVersion == null) {
            throw new MetadataStorageManagerException("metadataStorageVersion not read from ZK");
        }
        try {
            boolean result = this.deleteTableSpaceNode(name, (Integer)previous.metadataStorageVersion);
            if (result) {
                this.notifyMetadataChanged("dropTableSpace " + name);
            }
        }
        catch (IOException | InterruptedException | KeeperException ex) {
            this.handleSessionExpiredError(ex);
            throw new MetadataStorageManagerException(ex);
        }
    }

    @Override
    public List<NodeMetadata> listNodes() throws MetadataStorageManagerException {
        try {
            List children = this.ensureZooKeeper().getChildren(this.nodesPath, this.mainWatcher, null);
            ArrayList<NodeMetadata> result = new ArrayList<NodeMetadata>();
            for (String child : children) {
                NodeMetadata nodeMetadata = this.getNode(child);
                result.add(nodeMetadata);
            }
            return result;
        }
        catch (IOException | InterruptedException | KeeperException err) {
            this.handleSessionExpiredError(err);
            throw new MetadataStorageManagerException(err);
        }
    }

    @Override
    public void clear() throws MetadataStorageManagerException {
        try {
            List children = this.ensureZooKeeper().getChildren(this.nodesPath, false);
            for (String child : children) {
                this.ensureZooKeeper().delete(this.nodesPath + "/" + child, -1);
            }
            children = this.ensureZooKeeper().getChildren(this.tableSpacesPath, false);
            for (String child : children) {
                this.ensureZooKeeper().delete(this.tableSpacesPath + "/" + child, -1);
            }
            children = this.ensureZooKeeper().getChildren(this.ledgersPath, false);
            for (String child : children) {
                this.ensureZooKeeper().delete(this.ledgersPath + "/" + child, -1);
            }
        }
        catch (IOException | InterruptedException | KeeperException error) {
            LOGGER.log(Level.SEVERE, "Cannot clear metadata", error);
            throw new MetadataStorageManagerException(error);
        }
    }

    private NodeMetadata getNode(String nodeId) throws KeeperException, IOException, InterruptedException {
        Stat stat = new Stat();
        byte[] node = this.ensureZooKeeper().getData(this.nodesPath + "/" + nodeId, this.mainWatcher, stat);
        NodeMetadata nodeMetadata = NodeMetadata.deserialize(node, (Object)stat.getVersion());
        return nodeMetadata;
    }

    @Override
    public void registerNode(NodeMetadata nodeMetadata) throws MetadataStorageManagerException {
        try {
            String path = this.nodesPath + "/" + nodeMetadata.nodeId;
            LOGGER.severe("registerNode at " + path + " -> " + nodeMetadata);
            byte[] data = nodeMetadata.serialize();
            try {
                this.ensureZooKeeper().create(path, data, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
            }
            catch (KeeperException.NodeExistsException ok) {
                LOGGER.severe("registerNode at " + path + " " + (Object)((Object)ok));
                this.ensureZooKeeper().setData(path, data, -1);
            }
            this.notifyMetadataChanged("registerNode " + nodeMetadata);
        }
        catch (IOException | InterruptedException | KeeperException err) {
            this.handleSessionExpiredError(err);
            throw new MetadataStorageManagerException(err);
        }
    }

    @Override
    public List<TableSpaceReplicaState> getTableSpaceReplicaState(String tableSpaceUuid) throws MetadataStorageManagerException {
        try {
            List children;
            try {
                children = this.ensureZooKeeper().getChildren(this.tableSpacesReplicasPath + "/" + tableSpaceUuid, false);
            }
            catch (KeeperException.NoNodeException err) {
                return Collections.emptyList();
            }
            ArrayList<TableSpaceReplicaState> result = new ArrayList<TableSpaceReplicaState>();
            for (String child : children) {
                String path = this.tableSpacesReplicasPath + "/" + tableSpaceUuid + "/" + child;
                try {
                    byte[] data = this.ensureZooKeeper().getData(path, false, null);
                    TableSpaceReplicaState nodeMetadata = TableSpaceReplicaState.deserialize(data);
                    result.add(nodeMetadata);
                }
                catch (IOException deserializeError) {
                    LOGGER.log(Level.SEVERE, "error reading " + path, deserializeError);
                }
            }
            return result;
        }
        catch (IOException | InterruptedException | KeeperException err) {
            this.handleSessionExpiredError(err);
            throw new MetadataStorageManagerException(err);
        }
    }

    @Override
    public void updateTableSpaceReplicaState(TableSpaceReplicaState state) throws MetadataStorageManagerException {
        try {
            String tableSpacePath = this.tableSpacesReplicasPath + "/" + state.uuid;
            byte[] data = state.serialize();
            try {
                this.ensureZooKeeper().setData(tableSpacePath + "/" + state.nodeId, data, -1);
            }
            catch (KeeperException.NoNodeException notExists) {
                try {
                    this.ensureZooKeeper().create(tableSpacePath, data, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
                }
                catch (KeeperException.NodeExistsException nodeExistsException) {
                    // empty catch block
                }
                this.ensureZooKeeper().create(tableSpacePath + "/" + state.nodeId, data, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
            }
        }
        catch (IOException | InterruptedException | KeeperException err) {
            this.handleSessionExpiredError(err);
            throw new MetadataStorageManagerException(err);
        }
    }

    private static class TableSpaceList {
        private final int version;
        private final List<String> tableSpaces;

        public TableSpaceList(int version, List<String> tableSpaces) {
            this.version = version;
            this.tableSpaces = tableSpaces;
        }
    }
}

