package com.intellij.psi.builder;

import com.intellij.AyaModified;
import com.intellij.psi.ArrayTokenSequence;
import com.intellij.psi.FleetPsiParser;
import com.intellij.psi.tree.IElementType;
import com.intellij.psi.tree.ILazyParseableElementType;
import com.intellij.psi.tree.TokenSet;
import kala.collection.SeqView;
import kala.collection.mutable.MutableList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@AyaModified
public record ASTMarkerVisitor(
  @NotNull FleetPsiParser psiParser,
  @NotNull ASTMarkers<?> root,
  @NotNull TokenSet whitespaces,
  @NotNull TokenSet comments,
  @NotNull ArrayTokenSequence tokens,
  @NotNull String text
) {
  public void visitTree(
    @NotNull ASTMarkers<?> astMarkers,
    int startLexemeOffset,
    int currentMarker,
    int currentOffset,
    @NotNull Node parent
  ) {
    if (astMarkers.elementType(currentMarker) instanceof ILazyParseableElementType elementType && astMarkers.collapsed(currentMarker)) {
      parseChameleon(
        startLexemeOffset,
        astMarkers.lexemeCount(currentMarker),
        elementType,
        parent,
        currentOffset
      );
      return;
    }

    ASTMarkers.check(!(astMarkers.kind(currentMarker) != ASTMarkers.MarkerKind.Start && astMarkers.kind(currentMarker) != ASTMarkers.MarkerKind.Error), "");
    if (astMarkers.kind(currentMarker) == ASTMarkers.MarkerKind.Error) {
      return;
    }
    var endLexemeOffset = getEndLexemeIndex(astMarkers, startLexemeOffset, currentMarker);
    var child = astMarkers.firstChild(currentMarker);
    var prevOffset = startLexemeOffset;
    var i = startLexemeOffset;
    while (i < endLexemeOffset) {
      if (child != -1 && prevOffset + astMarkers.lexemeRelOffset(child) == i) {
        var childStartLexemeOffset = prevOffset + astMarkers.lexemeRelOffset(child);
        i = getEndLexemeIndex(astMarkers, childStartLexemeOffset, child);
        prevOffset = i;
        var node = new Node(
          astMarkers.elementType(child),
          offsetByLexemeIndex(childStartLexemeOffset) + currentOffset,
          offsetByLexemeIndex(i) + currentOffset,
          null,
          parent);
        visitTree(astMarkers, childStartLexemeOffset, child, currentOffset, node);
        child = astMarkers.nextSibling(child);
      } else {
        var node = new Node(
          tokens.lexType(i),
          offsetByLexemeIndex(i) + currentOffset,
          offsetByLexemeIndex(i + 1) + currentOffset,
          tokens.lexText(i),
          parent);
        if (tokens.lexType(i) instanceof ILazyParseableElementType lexType &&
          (parent.elementType != node.elementType || parent.startOffset != node.startOffset || parent.endOffset != node.endOffset)) {
          parseChameleon(i, 1, lexType, node, currentOffset);
          i++;
          continue;
        }
        i++;
      }
    }
    while (child != -1) {
      new Node(
        astMarkers.elementType(child),
        offsetByLexemeIndex(i) + currentOffset,
        offsetByLexemeIndex(i + astMarkers.lexemeRelOffset(child)) + currentOffset,
        null,
        parent);
      i += astMarkers.lexemeRelOffset(child);
      child = astMarkers.nextSibling(child);
    }
  }

  private void parseChameleon(
    int startLexemeOffset,
    int lexCount,
    @NotNull ILazyParseableElementType elementType,
    @NotNull Node parent,
    int currentOffset
  ) {
    var lexer = elementType.createInnerLexer();
    // note: kotlin code here uses `Any?` as generic type, I am quite confused.
    FleetPsiBuilder<ASTMarkers<Object>> chameleonPsiBuilder;
    ArrayTokenSequence chameleonTokens;
    String chameleonText;
    int lexemeChameleonStart;
    int newCurrentOffset;

    if (lexer != null) {
      var beginIndex = offsetByLexemeIndex(startLexemeOffset);
      var endIndex = offsetByLexemeIndex(startLexemeOffset + lexCount);
      newCurrentOffset = beginIndex + currentOffset;
      chameleonText = text.substring(beginIndex, endIndex);
      chameleonTokens = new ArrayTokenSequence.Builder(chameleonText, lexer).performLexing();
      lexemeChameleonStart = 0;
      chameleonPsiBuilder = psiParser.wrap(new MarkerPsiBuilder<>(
        chameleonText,
        chameleonTokens,
        whitespaces,
        comments,
        0, chameleonTokens.getLexemeCount()));
    } else {
      chameleonTokens = tokens;
      newCurrentOffset = currentOffset;
      lexemeChameleonStart = startLexemeOffset;
      chameleonText = text;
      chameleonPsiBuilder = psiParser.wrap(new MarkerPsiBuilder<>(
        text,
        tokens,
        whitespaces,
        comments,
        startLexemeOffset, lexCount));
    }
    elementType.parse(chameleonPsiBuilder);
    var treePrinter = new ASTMarkerVisitor(
      psiParser,
      chameleonPsiBuilder.getRoot(),
      whitespaces,
      comments,
      chameleonTokens,
      chameleonText);
    treePrinter.visitTree(
      chameleonPsiBuilder.getRoot(),
      lexemeChameleonStart,
      0,
      newCurrentOffset,
      parent);
  }

  private int offsetByLexemeIndex(int i) {
    return i > tokens.getLexemeCount() ? text.length() : tokens.lexStart(i);
  }

  private int getEndLexemeIndex(@NotNull ASTMarkers<?> astMarkers, int startLexemeIndex, int index) {
    return switch (astMarkers.kind(index)) {
      case ASTMarkers.MarkerKind.Error, ASTMarkers.MarkerKind.Start -> startLexemeIndex + astMarkers.lexemeCount(index);
      default -> throw new IllegalStateException("No default");
    };
  }

  public record Node(
    @NotNull IElementType elementType,
    int startOffset,
    int endOffset,
    @Nullable String text,
    @Nullable Node parent,
    @NotNull MutableList<Node> children
  ) {
    public Node(@NotNull IElementType elementType, int startOffset, int endOffset, @Nullable String text, @Nullable Node parent, @NotNull MutableList<Node> children) {
      this.elementType = elementType;
      this.startOffset = startOffset;
      this.endOffset = endOffset;
      this.parent = parent;
      this.text = text;
      this.children = children;
      if (parent != null) parent.children.append(this);
    }

    public Node(@NotNull IElementType elementType, int startOffset, int endOffset, @Nullable String text, @Nullable Node parent) {
      this(elementType, startOffset, endOffset, text, parent, MutableList.create());
    }

    public @NotNull String tokenText() {
      if (text != null) return text; // lexer token
      var sb = new StringBuilder();
      buildText(sb);
      return sb.toString();
    }

    private void buildText(@NotNull StringBuilder builder) {
      if (text != null) builder.append(text);
      else children.forEach(c -> c.buildText(builder));
    }

    @Override public @NotNull String toString() {
      return "%s(%d, %d)".formatted(elementType, startOffset, endOffset);
    }

    public @NotNull SeqView<Node> childrenOfType(IElementType type) {
      return children.view().filter(c -> c.elementType == type);
    }

    public @NotNull String toDebugString() {
      var builder = new StringBuilder();
      toDebugString(0, builder);
      return builder.toString();
    }

    private void toDebugString(int indent, @NotNull StringBuilder builder) {
      ASTMarkers.repeat(indent, i -> builder.append("\t"));
      builder.append("%s(%d..%d)%n".formatted(elementType, startOffset, endOffset));
      children.forEach(it -> it.toDebugString(indent + 1, builder));
    }
  }
}
