package com.jetbrains.lang.parsing.builder

import com.intellij.psi.tree.IElementType
import com.jetbrains.lang.syntax.SyntaxNode
import java.util.concurrent.atomic.AtomicReference

typealias ChameleonRef<T> = AtomicReference<T?>

interface ASTMarkers<T> {
  val size: Int
  fun kind(i: Int): Byte

  fun errorMessage(i: Int): String?

  fun lexemeCount(i: Int): Int
  fun lexemeRelOffset(i: Int): Int

  fun collapsed(i: Int): Boolean

  fun markersCount(i: Int): Int

  fun elementType(i: Int): IElementType
  fun chameleonAt(lexemeIndex: Int): ChameleonRef<T>

  fun chameleons(): List<Pair<Int, ChameleonRef<T>>>

  fun mutate(mutator: MutableContext<T>.() -> Unit): ASTMarkers<T>

  interface MutableContext<T> {
    fun substitute(i: Int, lexemeIndex: Int, astMarkers: ASTMarkers<T>)
    fun changeChameleons(pairs: List<Pair<Int, ChameleonRef<T>>>)
    fun changeLexCount(startMarker: Int, endMarker: Int, lexCount: Int)
    fun changeMarkerCount(startMarker: Int, endMarker: Int, markerCount: Int)
  }
}

object MarkerKind {
  const val Undone: Byte = 0
  const val Start: Byte = 1
  const val End: Byte = 2
  const val Error: Byte = 3
}

fun ASTMarkers<*>.prevSibling(markerIndex: Int): Int {
  val x: SyntaxNode
  return if (markerIndex == 0) {
    -1
  } else {
    val prevMarkerIndex = markerIndex - 1
    return when (kind(prevMarkerIndex)) {
      MarkerKind.Start -> -1
      MarkerKind.End -> {
        prevMarkerIndex - markersCount(prevMarkerIndex)
      }

      MarkerKind.Error -> prevMarkerIndex
      else -> error("no else")
    }
  }
}

fun ASTMarkers<*>.nextSibling(markerIndex: Int): Int {
  return when (kind(markerIndex)) {
    MarkerKind.Start -> {
      val endIndex = markerIndex + markersCount(markerIndex)
      if (endIndex == size - 1) {
        -1
      } else {
        val nextToEndIndex = endIndex + 1
        when (kind(nextToEndIndex)) {
          MarkerKind.Start -> nextToEndIndex
          MarkerKind.End -> -1
          MarkerKind.Error -> nextToEndIndex
          else -> error("no else")
        }
      }
    }

    MarkerKind.Error -> {
      val nextKind = kind(markerIndex + 1)
      if (nextKind == MarkerKind.End) {
        -1
      } else {
        markerIndex + 1
      }
    }

    MarkerKind.End -> error("should never be at the end")
    else -> error("no else")
  }
}

fun ASTMarkers<*>.lastChild(markerIndex: Int): Int {
  return when (kind(markerIndex)) {
    MarkerKind.End -> error("never at end")
    MarkerKind.Error -> -1
    MarkerKind.Start -> {
      val prevToEndIndex = markerIndex + markersCount(markerIndex) - 1
      when (kind(prevToEndIndex)) {
        MarkerKind.Start -> -1
        MarkerKind.End -> {
          prevToEndIndex - markersCount(prevToEndIndex)
        }

        MarkerKind.Error -> prevToEndIndex
        else -> error("no else")
      }
    }

    else -> error("no else")
  }
}

fun ASTMarkers<*>.firstChild(markerIndex: Int): Int {
  require(markerIndex < size - 1) {
    "at least there is an end"
  }

  val nextMarkerIndex = markerIndex + 1
  return when (kind(nextMarkerIndex)) {
    MarkerKind.Start -> nextMarkerIndex
    MarkerKind.End -> -1
    MarkerKind.Error -> nextMarkerIndex
    else -> error("no else")
  }
}
