// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.lexer;

import com.intellij.openapi.util.TextRange;
import com.intellij.psi.tree.IElementType;
import kala.collection.immutable.ImmutableSeq;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;

@NonNls
public interface FlexLexer {
  void yybegin(int state);

  int yystate();

  int getTokenStart();

  int getTokenEnd();

  @Nullable IElementType advance() throws IOException;

  /**
   * Generated implementation of {@link FlexLexer#advance()} method
   * do not throw {@link IOException}.
   */
  default @Nullable IElementType advanceUnchecked() {
    try {
      return advance();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  default @NotNull ImmutableSeq<Token> allTheWayDown() {
    return ImmutableSeq.generateUntilNull(() -> {
      var type = advanceUnchecked();
      if (type == null) return null;
      return new Token(new TextRange(getTokenStart(), getTokenEnd()), type);
    });
  }

  record Token(@NotNull TextRange range, @NotNull IElementType type) { }

  void reset(CharSequence buf, int start, int end, int initialState);
}
