/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jetbrains.lang.parsing.builder;

import org.aya.kala.KalaTODO;
import com.intellij.lang.WhitespacesAndCommentsBinder;
import kala.collection.mutable.primitive.MutableIntSet;
import org.aya.kala.Int2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.BitSet;

import static com.intellij.lang.WhitespacesBinders.DEFAULT_LEFT_BINDER;
import static com.intellij.lang.WhitespacesBinders.DEFAULT_RIGHT_BINDER;

/**
 * @author peter
 */
final class MarkerOptionalData extends BitSet {
  private final Int2ObjectOpenHashMap<Throwable> myDebugAllocationPositions = new Int2ObjectOpenHashMap<>();
  private final Int2ObjectOpenHashMap<String> myDoneErrors = new Int2ObjectOpenHashMap<>();
  private final Int2ObjectOpenHashMap<WhitespacesAndCommentsBinder> myLeftBinders = new Int2ObjectOpenHashMap<>();
  private final Int2ObjectOpenHashMap<WhitespacesAndCommentsBinder> myRightBinders = new Int2ObjectOpenHashMap<>();
  private final MutableIntSet myCollapsed = MutableIntSet.create();

  void clean(int markerId) {
    if (get(markerId)) {
      set(markerId, false);
      myLeftBinders.remove(markerId);
      myRightBinders.remove(markerId);
      myDoneErrors.remove(markerId);
      myCollapsed.remove(markerId);
      myDebugAllocationPositions.remove(markerId);
    }
  }

  @KalaTODO void compact() {
    myLeftBinders.trim();
    myRightBinders.trim();
    myDebugAllocationPositions.trim();
    // myCollapsed.trim();
    myDoneErrors.trim();
  }

  @Nullable
  String getDoneError(int markerId) {
    if (!get(markerId)) return null;

    return myDoneErrors.get(markerId);
  }

  boolean isCollapsed(int markerId) {
    return get(markerId) && myCollapsed.contains(markerId);
  }

  void setErrorMessage(int markerId, @NotNull String message) {
    markAsHavingOptionalData(markerId);
    myDoneErrors.put(markerId, message);
  }

  void markCollapsed(int markerId) {
    markAsHavingOptionalData(markerId);
    myCollapsed.add(markerId);
  }

  private void markAsHavingOptionalData(int markerId) {
    set(markerId);
  }

  void notifyAllocated(int markerId) {
    markAsHavingOptionalData(markerId);
    myDebugAllocationPositions.put(markerId, new Throwable("Created at the following trace."));
  }

  Throwable getAllocationTrace(MarkerPsiBuilder.StartMarker marker) {
    return myDebugAllocationPositions.get(marker.markerId);
  }

  WhitespacesAndCommentsBinder getBinder(int markerId, boolean right) {
    if (!get(markerId)) {
      return getDefaultBinder(right);
    }

    WhitespacesAndCommentsBinder binder = getBinderMap(right).get(markerId);
    return binder != null ? binder : getDefaultBinder(right);
  }

  void assignBinder(int markerId, @NotNull WhitespacesAndCommentsBinder binder, boolean right) {
    Int2ObjectOpenHashMap<WhitespacesAndCommentsBinder> map = getBinderMap(right);
    if (binder != getDefaultBinder(right)) {
      markAsHavingOptionalData(markerId);
      map.put(markerId, binder);
    } else {
      map.remove(markerId);
    }
  }

  private static WhitespacesAndCommentsBinder getDefaultBinder(boolean right) {
    return right ? DEFAULT_RIGHT_BINDER : DEFAULT_LEFT_BINDER;
  }

  private Int2ObjectOpenHashMap<WhitespacesAndCommentsBinder> getBinderMap(boolean right) {
    return right ? myRightBinders : myLeftBinders;
  }

}
