// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.psi.templateLanguages;

import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.lang.ParserDefinition;
import com.intellij.lexer.Lexer;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.ArrayTokenSequence;
import com.intellij.psi.tree.IElementType;
import com.intellij.psi.tree.TokenSet;
import com.intellij.util.CharTable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.function.Function;

public class TemplateDataElementType {
  private static final int CHECK_PROGRESS_AFTER_TOKENS = 1000;

  private final @NotNull ParserDefinition myTemplateParserDefinition;
  private final @NotNull IElementType myTemplateElementType;
  final @NotNull IElementType myOuterElementType;

  public TemplateDataElementType(@NotNull ParserDefinition templateParserDefinition,
                                 @NotNull IElementType templateElementType,
                                 @NotNull IElementType outerElementType) {
    myTemplateParserDefinition = templateParserDefinition;
    myTemplateElementType = templateElementType;
    myOuterElementType = outerElementType;
  }

  protected Lexer createBaseLexer() {
    return myTemplateParserDefinition.createLexer(null);
  }

  public ArrayTokenSequence buildTemplateDataLexemes(@NotNull CharSequence sourceCode, @NotNull Lexer lexer) {
    RangeCollectorImpl collector = new RangeCollectorImpl(this);
    CharSequence modifiedTemplateDataSource = createTemplateFile(sourceCode, collector);

    ArrayTokenSequence dataTokens = new ArrayTokenSequence.Builder(modifiedTemplateDataSource, lexer).performLexing();
    return collector.insertOuterElementsAndRemoveRanges(dataTokens);
  }

  /**
   * Creates psi tree without base language elements. The result PsiFile can contain additional elements.
   * Ranges of the removed tokens/additional elements should be stored in the rangeCollector
   *
   * @param sourceCode     source code: base language with template language
   * @param rangeCollector collector for ranges with non-template/additional elements
   * @return template psiFile
   */
  protected CharSequence createTemplateFile(final CharSequence sourceCode,
                                            @NotNull TemplateDataElementType.RangeCollector rangeCollector) {
    CharSequence templateSourceCode = createTemplateText(sourceCode, createBaseLexer(), rangeCollector);
    if (rangeCollector instanceof RangeCollectorImpl) {
      Language templateLanguage = myTemplateParserDefinition.getFileNodeType().getLanguage();
      ((RangeCollectorImpl) rangeCollector).prepareFileForParsing(templateLanguage, sourceCode, templateSourceCode);
    }
    return templateSourceCode;
  }

  /**
   * Creates source code without template tokens. May add additional pieces of code.
   * Ranges of such additions should be added in rangeCollector using {@link RangeCollector#addRangeToRemove(TextRange)}
   * for later removal from the resulting tree.
   * <p>
   * Consider overriding {@link #collectTemplateModifications(CharSequence, Lexer)} instead.
   *
   * @param sourceCode     source code with base and template languages
   * @param baseLexer      base language lexer
   * @param rangeCollector collector for ranges with non-template/additional symbols
   * @return template source code
   */
  protected CharSequence createTemplateText(@NotNull CharSequence sourceCode,
                                            @NotNull Lexer baseLexer,
                                            @NotNull TemplateDataElementType.RangeCollector rangeCollector) {
    TemplateDataModifications modifications = collectTemplateModifications(sourceCode, baseLexer);
    return ((RangeCollectorImpl) rangeCollector).applyTemplateDataModifications(sourceCode, modifications);
  }

  /**
   * Collects changes to apply to template source code for later parsing by underlying language.
   *
   * @param sourceCode source code with base and template languages
   * @param baseLexer  base language lexer
   */
  protected @NotNull TemplateDataModifications collectTemplateModifications(@NotNull CharSequence sourceCode, @NotNull Lexer baseLexer) {
    TemplateDataModifications modifications = new TemplateDataModifications();
    baseLexer.start(sourceCode);
    TextRange currentRange = TextRange.EMPTY_RANGE;
    int tokenCounter = 0;
    while (baseLexer.getTokenType() != null) {
      if (++tokenCounter % CHECK_PROGRESS_AFTER_TOKENS == 0) {
        ProgressManager.checkCanceled();
      }
      TextRange newRange = TextRange.create(baseLexer.getTokenStart(), baseLexer.getTokenEnd());
      assert currentRange.getEndOffset() == newRange.getStartOffset() :
        "Inconsistent tokens stream from " + baseLexer +
          ": " + getRangeDump(currentRange, sourceCode) + " followed by " + getRangeDump(newRange, sourceCode);
      currentRange = newRange;
      if (baseLexer.getTokenType() == myTemplateElementType) {
        TemplateDataModifications tokenModifications = appendCurrentTemplateToken(baseLexer.getTokenEnd(), baseLexer.getTokenSequence());
        modifications.addAll(tokenModifications);
      } else {
        modifications.addOuterRange(currentRange, isInsertionToken(baseLexer.getTokenType(), baseLexer.getTokenSequence()));
      }
      baseLexer.advance();
    }

    return modifications;
  }

  @NotNull
  private static String getRangeDump(@NotNull TextRange range, @NotNull CharSequence sequence) {
    return "'" + StringUtil.escapeStringCharacters(range.subSequence(sequence).toString()) + "' " + range;
  }

  /**
   * @deprecated Override {@link #appendCurrentTemplateToken(int, CharSequence)} instead.
   */
  @Deprecated
  protected void appendCurrentTemplateToken(@NotNull StringBuilder result,
                                            @NotNull CharSequence buf,
                                            @NotNull Lexer lexer,
                                            @NotNull TemplateDataElementType.RangeCollector collector) {
    result.append(buf, lexer.getTokenStart(), lexer.getTokenEnd());
  }

  /**
   * Collects modifications for tokens having {@link #myTemplateElementType} type.
   *
   * @return modifications need to be applied for the current token
   */
  protected @NotNull TemplateDataModifications appendCurrentTemplateToken(int tokenEndOffset, @NotNull CharSequence tokenText) {
    return TemplateDataModifications.EMPTY;
  }

  /**
   * @deprecated Use {@link #isInsertionToken(IElementType, CharSequence)} instead.
   */
  @Deprecated(forRemoval = true)
  protected @NotNull TokenSet getTemplateDataInsertionTokens() {
    return TokenSet.EMPTY;
  }

  /**
   * Returns true if a string is expected to be inserted into resulting file in place of a current token.
   * <p>
   * If insertion range contains several tokens, <code>true</code> may be returned only for the starting one. For example, if
   * <code><?=$myVar?></code> has three tokens <code><?=</code>, <code>$myVar</code> and <code>?></code>, only <code><?=</code>
   * may be an insertion token.
   * <p>
   * Override this method when overriding {@link #collectTemplateModifications(CharSequence, Lexer)} is not required.
   *
   * @see RangeCollector#addOuterRange(TextRange, boolean)
   */
  protected boolean isInsertionToken(@Nullable IElementType tokenType, @NotNull CharSequence tokenSequence) {
    return false;
  }

  public static @NotNull ASTNode parseWithOuterAndRemoveRangesApplied(@NotNull ASTNode chameleon,
                                                                      @NotNull Language language,
                                                                      @NotNull Function<? super @NotNull CharSequence, ? extends @NotNull ASTNode> parser) {
    RangeCollectorImpl collector = chameleon.getUserData(RangeCollectorImpl.OUTER_ELEMENT_RANGES);
    return collector != null ? collector.applyRangeCollectorAndExpandChameleon(chameleon, language, parser)
      : parser.apply(chameleon.getChars());
  }

  /**
   * This collector is used for storing ranges of outer elements and ranges of artificial elements, that should be stripped from the resulting tree
   * At the time of creating source code for the data language we need to memorize positions with template language elements.
   * For such positions we use {@link RangeCollector#addOuterRange}
   * Sometimes to build a correct tree we need to insert additional symbols into resulting source:
   * e.g. put an identifier instead of the base language fragment: {@code something={% $var %}} => {@code something=dummyidentifier}
   * that must be removed after building the tree.
   * For such additional symbols {@link RangeCollector#addRangeToRemove} must be used
   *
   * @apiNote Please note that all start offsets for the ranges must be in terms of "original source code". So, outer ranges are ranges
   * of outer elements in original source code. Ranges to remove don't correspond to any text range neither in original nor in modified text.
   * But their start offset is the offset in original text, and length is the length of inserted dummy identifier.
   * @implNote Should be interface, but abstract class with empty method bodies for keeping binary compatibility with plugins.
   */
  public static abstract class RangeCollector {

    /**
     * Adds range corresponding to the outer element inside original source code.
     * After building the data template tree these ranges will be used for inserting outer language elements.
     * If it's known whether this template element adds some string to resulting text, consider using {@link #addOuterRange(TextRange, boolean)}.
     */
    public void addOuterRange(@NotNull TextRange newRange) { }

    /**
     * Adds range corresponding to the outer element inside original source code.
     * After building the data template tree these ranges will be used for inserting outer language elements.
     *
     * @param isInsertion <tt>true</tt> if element is expected to insert some text into template data fragment. For example, PHP's
     *                    <code><?= $myVar ?></code> are insertions, while <code><?php foo() ?></code> are not.
     */
    public abstract void addOuterRange(@NotNull TextRange newRange, boolean isInsertion);

    /**
     * Adds the fragment that must be removed from the tree on the stage inserting outer elements.
     * This method should be called after adding "fake" symbols inside the data language text for building syntax correct tree
     */
    public void addRangeToRemove(@NotNull TextRange rangeToRemove) { }
  }

  /**
   * Customizes template data language-specific parsing in templates.
   */
  public interface OuterLanguageRangePatcher {

    /**
     * @return Text to be inserted for parsing in outer element insertion ranges provided by
     * {@link RangeCollector#addOuterRange(TextRange, boolean)} where <tt>isInsertion == true</tt>
     */
    @Nullable String getTextForOuterLanguageInsertionRange(@NotNull TemplateDataElementType templateDataElementType,
                                                           @NotNull CharSequence outerElementText);
  }

  private static class DummyCharTable implements CharTable {

    private static final DummyCharTable INSTANCE = new DummyCharTable();

    @Override
    public @NotNull CharSequence intern(@NotNull CharSequence text) {
      return text;
    }

    @Override
    public @NotNull CharSequence intern(@NotNull CharSequence baseText, int startOffset, int endOffset) {
      return baseText.subSequence(startOffset, endOffset);
    }
  }
}
