// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package com.intellij.psi.impl.source.codeStyle.lineIndent;

import com.intellij.formatting.Indent;
import com.intellij.formatting.Indent.Type;
import com.intellij.lang.Language;
import com.intellij.psi.ITokenSequence;
import com.intellij.psi.codeStyle.IndentOptions;
import com.intellij.psi.codeStyle.lineIndent.LineIndentProvider;
import com.intellij.psi.impl.source.codeStyle.LexemeIterator;
import com.intellij.psi.impl.source.codeStyle.LexemeIteratorImpl;
import com.intellij.psi.impl.source.codeStyle.SemanticEditorPosition;
import com.intellij.psi.tree.IElementType;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static com.intellij.formatting.Indent.Type.*;
import static com.intellij.psi.impl.source.codeStyle.lineIndent.JavaLikeLangLineIndentProvider.JavaLikeElement.*;

public abstract class JavaLikeLangLineIndentProvider implements LineIndentProvider {

  public enum JavaLikeElement implements SemanticEditorPosition.SyntaxElement {
    Whitespace,
    Semicolon,
    BlockOpeningBrace,
    BlockClosingBrace,
    ArrayOpeningBracket,
    ArrayClosingBracket,
    RightParenthesis,
    LeftParenthesis,
    Colon,
    SwitchCase,
    SwitchDefault,
    ElseKeyword,
    IfKeyword,
    ForKeyword,
    TryKeyword,
    DoKeyword,
    BlockComment,
    DocBlockStart,
    DocBlockEnd,
    LineComment,
    Comma,
    LanguageStartDelimiter
  }

  @NotNull
  private final CharSequence myChars;
  @NotNull
  private final ITokenSequence myTokens;
  @NotNull
  private final IndentOptions myIndentOptions;

  public JavaLikeLangLineIndentProvider(@NotNull CharSequence charSequence,
                                        @NotNull ITokenSequence tokenSequence,
                                        @NotNull IndentOptions indentOptions) {
    myChars = charSequence;
    myTokens = tokenSequence;
    myIndentOptions = indentOptions;
  }

  @Nullable
  @Override
  public String getLineIndent(int offset) {
    if (offset > 0) {
      IndentCalculator indentCalculator = getIndent(offset - 1);
      if (indentCalculator != null) {
        return indentCalculator.getIndentString(getPosition(offset - 1));
      }
    } else {
      return "";
    }
    return null;
  }

  @Nullable
  protected IndentCalculator getIndent(int offset) {
    IndentCalculatorFactory myFactory = new IndentCalculatorFactory();
    if (getPosition(offset).matchesRule(
      position -> position.isAt(Whitespace) &&
        position.isAtMultiline())) {
      if (getPosition(offset).before().isAt(Comma)) {
        SemanticEditorPosition position = getPosition(offset);
        if (position.hasEmptyLineAfter(offset) &&
          !position.after().matchesRule(
            p -> p.isAtAnyOf(ArrayClosingBracket, BlockOpeningBrace, BlockClosingBrace, RightParenthesis) || p.isAtEnd()) &&
          position.findLeftParenthesisBackwardsSkippingNestedWithPredicate(
            LeftParenthesis,
            RightParenthesis,
            self -> self.isAtAnyOf(BlockClosingBrace, BlockOpeningBrace, Semicolon)).isAt(LeftParenthesis)) {
          return myFactory.createIndentCalculator(NONE, IndentCalculator.LINE_AFTER);
        }
      } else if (afterOptionalWhitespaceOnSameLine(offset).matchesRule(
        position -> position.isAt(BlockClosingBrace) && !position.after().afterOptional(Whitespace).isAt(Comma))) {
        return myFactory.createIndentCalculator(
          NONE,
          position -> {
            position.moveToLeftParenthesisBackwardsSkippingNested(BlockOpeningBrace, BlockClosingBrace);
            if (!position.isAtEnd()) {
              return getBlockStatementStartOffset(position);
            }
            return -1;
          });
      } else if (getPosition(offset).beforeOptional(Whitespace).isAt(BlockClosingBrace)) {
        SemanticEditorPosition position = getPosition(offset).beforeOptional(Whitespace).before();
        boolean isOnNewLine = position.isAtMultiline(Whitespace);
        position.moveToLeftParenthesisBackwardsSkippingNested(BlockOpeningBrace, BlockClosingBrace);
        position.moveBefore();
        int statementStart = getStatementStartOffset(position, true);
        position = getPosition(statementStart);
        if (!isStartOfStatementWithOptionalBlock(position)) {
          if (!isOnNewLine) return null;
          return myFactory.createIndentCalculator(getBlockIndentType(), IndentCalculator.LINE_BEFORE);
        } else {
          return myFactory
            .createIndentCalculator(getBlockIndentType(), this::getFirstUppermostControlStructureKeywordOffset);
        }
      } else if (getPosition(offset).before().isAt(Semicolon)) {
        SemanticEditorPosition beforeSemicolon = getPosition(offset).before().beforeOptional(Semicolon);
        if (beforeSemicolon.isAt(BlockClosingBrace)) {
          beforeSemicolon.moveBeforeParentheses(BlockOpeningBrace, BlockClosingBrace);
        }
        int statementStart = getStatementStartOffset(beforeSemicolon, dropIndentAfterReturnLike(beforeSemicolon), true);
        SemanticEditorPosition atStatementStart = getPosition(statementStart);
        if (isAtBlockOpeningOnSameLine(atStatementStart)) {
          return myFactory.createIndentCalculator(getIndentInBlock(atStatementStart), this::getDeepBlockStatementStartOffset);
        }
        if (!isInsideForLikeConstruction(atStatementStart)) {
          return myFactory.createIndentCalculator(NONE, position -> statementStart);
        }
      } else if (isInArray(offset)) {
        return myFactory.createIndentCalculator(getIndentInBrackets(), IndentCalculator.LINE_BEFORE);
      } else if (getPosition(offset).before().isAt(LeftParenthesis)) {
        return myFactory.createIndentCalculator(CONTINUATION, IndentCalculator.LINE_BEFORE);
      } else if (getPosition(offset).matchesRule(
        position -> {
          moveBeforeEndLineComments(position);
          if (position.isAt(BlockOpeningBrace)) {
            return !position.before().beforeOptionalMix(LineComment, BlockComment, Whitespace).isAt(LeftParenthesis);
          }
          return false;
        }
      )) {
        SemanticEditorPosition position = getPosition(offset).before().beforeOptionalMix(LineComment, BlockComment, Whitespace);
        return myFactory.createIndentCalculator(getIndentInBlock(position), this::getBlockStatementStartOffset);
      } else if (getPosition(offset).before().matchesRule(
        position -> isColonAfterLabelOrCase(position) || position.isAtAnyOf(ElseKeyword, DoKeyword))) {
        Type indentType = getPosition(offset).afterOptional(Whitespace).isAt(BlockOpeningBrace) ?
          NONE : // e.g. else <caret> {
          NORMAL;
        return myFactory.createIndentCalculator(indentType, IndentCalculator.LINE_BEFORE);
      } else if (getPosition(offset).matchesRule(
        position -> {
          position.moveBefore();
          if (position.isAt(BlockComment)) {
            return position.before().isAt(Whitespace) && position.isAtMultiline();
          }
          return false;
        }
      )) {
        int offsetBeforeComment = getPosition(offset).findStartOf(BlockComment);
        return getIndent(offsetBeforeComment);
      } else if (getPosition(offset).before().isAt(DocBlockEnd)) {
        return myFactory.createIndentCalculator(NONE, position -> position.findStartOf(DocBlockStart));
      } else {
        SemanticEditorPosition position = getPosition(offset);
        position = position.before().beforeOptionalMix(LineComment, BlockComment, Whitespace);
        if (position.isAt(RightParenthesis)) {
          int offsetAfterParen = position.getStartOffset() + 1;
          position.moveBeforeParentheses(LeftParenthesis, RightParenthesis);
          if (!position.isAtEnd()) {
            position.moveBeforeOptional(Whitespace);
            if (position.isAt(IfKeyword) || position.isAt(ForKeyword)) {
              SemanticEditorPosition.SyntaxElement element = position.getCurrElement();
              assert element != null;
              final int controlKeywordOffset = position.getStartOffset();
              Type indentType = getPosition(offsetAfterParen).afterOptional(Whitespace).isAt(BlockOpeningBrace) ? NONE : NORMAL;
              return myFactory.createIndentCalculator(indentType, baseLineOffset -> controlKeywordOffset);
            }
          }
        }
      }
    }
    //return myFactory.createIndentCalculator(NONE, IndentCalculator.LINE_BEFORE); /* TO CHECK UNCOVERED CASES */
    return null;
  }

  private static boolean isAtBlockOpeningOnSameLine(@NotNull SemanticEditorPosition position) {
    SemanticEditorPosition pos = position.copy();
    while (!pos.isAt(BlockOpeningBrace)) {
      pos.moveBefore();
      if (pos.isAtEnd() || pos.isAtMultiline()) {
        return false;
      }
    }
    return true;
  }

  private SemanticEditorPosition afterOptionalWhitespaceOnSameLine(int offset) {
    SemanticEditorPosition position = getPosition(offset);
    if (position.isAt(Whitespace)) {
      if (position.hasLineBreaksAfter(offset)) return position;
      position.moveAfter();
    }
    return position;
  }

  /**
   * Checks that the current offset is inside array. By default it is assumed to be after opening array bracket
   * but can be overridden for more complicated logic, for example, the following case in Java: []{&lt;caret&gt;}.
   *
   * @param offset The current offset in the editor.
   * @return {@code true} if the position is inside array.
   */
  protected boolean isInArray(int offset) {
    return getPosition(offset).before().isAt(ArrayOpeningBracket);
  }

  /**
   * Checking the document context in position for return-like token (i.e. {@code return}, {@code break}, {@code continue}),
   * after that we need to reduce the indent (for example after {@code break;} in {@code switch} statement).
   *
   * @param statementBeforeSemicolon position in the document context
   * @return true, if need to reduce the indent
   */
  protected boolean dropIndentAfterReturnLike(@NotNull SemanticEditorPosition statementBeforeSemicolon) {
    return false;
  }

  protected boolean isColonAfterLabelOrCase(@NotNull SemanticEditorPosition position) {
    return position.isAt(Colon)
      && getPosition(position.getStartOffset()).isAfterOnSameLine(SwitchCase, SwitchDefault);
  }

  protected boolean isInsideForLikeConstruction(SemanticEditorPosition position) {
    return position.isAfterOnSameLine(ForKeyword);
  }

  /**
   * Returns the start offset of the statement or new-line-'{' that owns the code block in {@code position}.
   * <p>
   * Custom implementation for language can overwrite the default behavior for multi-lines statements like
   * <pre>{@code
   *    template<class T>
   *    class A {};
   * }</pre>
   * or check indentation after new-line-'{' vs the brace style.
   *
   * @param position the position in the code block
   */
  protected int getBlockStatementStartOffset(@NotNull SemanticEditorPosition position) {
    moveBeforeEndLineComments(position);
    position.moveBeforeOptional(BlockOpeningBrace);
    if (position.isAt(Whitespace)) {
      if (position.isAtMultiline()) {
        return position.after().getStartOffset();
      }
      position.moveBefore();
    }
    return getStatementStartOffset(position, false);
  }

  private static void moveBeforeEndLineComments(@NotNull SemanticEditorPosition position) {
    position.moveBefore();
    while (!position.isAtMultiline() && position.isAtAnyOf(LineComment, BlockComment, Whitespace)) {
      position.moveBefore();
    }
  }

  /**
   * Returns the start offset of the statement that owns the code block in {@code position}
   *
   * @param position the position in the code block
   */
  protected int getDeepBlockStatementStartOffset(@NotNull SemanticEditorPosition position) {
    position.moveToLeftParenthesisBackwardsSkippingNested(BlockOpeningBrace, BlockClosingBrace);
    return getBlockStatementStartOffset(position);
  }

  private int getStatementStartOffset(@NotNull SemanticEditorPosition position, boolean ignoreLabels) {
    return getStatementStartOffset(position, ignoreLabels, false);
  }

  private int getStatementStartOffset(@NotNull SemanticEditorPosition position, boolean ignoreLabels, boolean useParentControlStructures) {
    Language currLanguage = position.getLanguage();
    while (!position.isAtEnd()) {
      if (currLanguage == Language.ANY || currLanguage == null) currLanguage = position.getLanguage();
      if (!ignoreLabels && isColonAfterLabelOrCase(position)) {
        SemanticEditorPosition afterColon = getPosition(position.getStartOffset())
          .afterOptionalMix(Whitespace, BlockComment)
          .after()
          .afterOptionalMix(Whitespace, LineComment);
        return afterColon.getStartOffset();
      } else if (position.isAt(RightParenthesis)) {
        position.moveBeforeParentheses(LeftParenthesis, RightParenthesis);
        continue;
      } else if (position.isAt(BlockClosingBrace)) {
        position.moveBeforeParentheses(BlockOpeningBrace, BlockClosingBrace);
        continue;
      } else if (position.isAt(ArrayClosingBracket)) {
        position.moveBeforeParentheses(ArrayOpeningBracket, ArrayClosingBracket);
        continue;
      } else if (isStartOfStatementWithOptionalBlock(position)) {
        return useParentControlStructures ? getFirstUppermostControlStructureKeywordOffset(position) : position.getStartOffset();
      } else if (position.isAtAnyOf(Semicolon,
        BlockOpeningBrace,
        BlockComment,
        DocBlockEnd,
        LeftParenthesis,
        LanguageStartDelimiter)) {
        SemanticEditorPosition statementStart = position.copy();
        statementStart = statementStart.after().afterOptionalMix(Whitespace, LineComment);
        if (!isIndentProvider(statementStart, ignoreLabels)) {
          final SemanticEditorPosition maybeColon = statementStart.afterOptionalMix(Whitespace, BlockComment).after();
          final SemanticEditorPosition afterColonStatement = maybeColon.after().after();
          if (atColonWithNewLineAfterColonStatement(maybeColon, afterColonStatement)) {
            return afterColonStatement.getStartOffset();
          }
          if (atBlockStartAndNeedBlockIndent(position)) {
            return position.getStartOffset();
          }
        } else if (!statementStart.isAtEnd()) {
          return statementStart.getStartOffset();
        }
      }
      position.moveBefore();
    }
    return 0;
  }

  /**
   * Returns {@code true} if the {@code position} starts a statement that <i>can</i> have a code block and the statement
   * is the first in the code line.
   * In C-like languages it is one of {@code if, else, for, while, do, try}.
   *
   * @param position
   */
  protected boolean isStartOfStatementWithOptionalBlock(@NotNull SemanticEditorPosition position) {
    return position.matchesRule(
      self -> {
        final SemanticEditorPosition before = self.before();
        return before.isAt(Whitespace)
          && before.isAtMultiline()
          && self.isAtAnyOf(ElseKeyword,
          IfKeyword,
          ForKeyword,
          TryKeyword,
          DoKeyword);
      });
  }

  private static boolean atBlockStartAndNeedBlockIndent(@NotNull SemanticEditorPosition position) {
    return position.isAt(BlockOpeningBrace);
  }

  private static boolean atColonWithNewLineAfterColonStatement(@NotNull SemanticEditorPosition maybeColon,
                                                               @NotNull SemanticEditorPosition afterColonStatement) {
    return maybeColon.isAt(Colon)
      && maybeColon.after().isAtMultiline(Whitespace)
      && !afterColonStatement.isAtEnd();
  }

  /**
   * Search for the topmost control structure keyword which doesn't have any code block delimiters like {@code {...}}. For example:
   * <pre>
   * 1  for (...)
   * 2      if (...)
   * 3          for(...) {}
   * 4  [position]
   * </pre>
   * The method will return an offset of the first {@code for} on line 1.
   *
   * @return
   */
  private int getFirstUppermostControlStructureKeywordOffset(@NotNull SemanticEditorPosition position) {
    SemanticEditorPosition curr = position.copy();
    while (!curr.isAtEnd()) {
      if (isStartOfStatementWithOptionalBlock(curr)) {
        SemanticEditorPosition candidate = curr.copy();
        curr.moveBefore();
        curr.moveBeforeOptionalMix(Whitespace, LineComment, BlockComment);
        if (curr.isAt(RightParenthesis)) {
          curr.moveBeforeParentheses(LeftParenthesis, RightParenthesis);
          SemanticEditorPosition controlStructureCheck = curr.copy();
          controlStructureCheck.moveBeforeOptionalMix(Whitespace, LineComment, BlockComment);
          if (isStartOfStatementWithOptionalBlock(controlStructureCheck)) {
            continue;
          }
        }
        return candidate.getStartOffset();
      } else if (curr.isAt(BlockClosingBrace)) {
        curr.moveBeforeParentheses(BlockOpeningBrace, BlockClosingBrace);
        continue;
      }
      curr.moveBefore();
    }
    return position.before().getStartOffset();
  }

  /**
   * Checking the document context in position as indent-provider.
   *
   * @param statementStartPosition position is the document
   * @param ignoreLabels           {@code true}, if labels cannot be used as indent-providers in the context.
   * @return {@code true}, if statement is indent-provider (by default)
   */
  protected boolean isIndentProvider(@NotNull SemanticEditorPosition statementStartPosition, boolean ignoreLabels) {
    return true;
  }

  /**
   * Returns abstract semantic position in {@code editor} for indent calculation.
   *
   * @param offset the offset in the {@code editor}
   */
  public SemanticEditorPosition getPosition(int offset) {
    return SemanticEditorPosition.createEditorPosition(offset,
      myChars,
      this::getIteratorAtPosition,
      this::mapType);
  }

  @NotNull
  protected LexemeIterator getIteratorAtPosition(int offset) {
    return new LexemeIteratorImpl(myTokens, offset);
  }

  @Nullable
  protected abstract SemanticEditorPosition.SyntaxElement mapType(@NotNull IElementType tokenType);

  @Nullable
  protected Indent getIndentInBlock(@NotNull SemanticEditorPosition blockStartPosition) {
    if (myIndentOptions.BRACE_STYLE == NEXT_LINE_SHIFTED) {
      return getDefaultIndentFromType(myIndentOptions.METHOD_BRACE_STYLE == NEXT_LINE_SHIFTED ? NONE : null);
    }
    return getDefaultIndentFromType(NORMAL);
  }

  private Type getBlockIndentType() {
    if (myIndentOptions.BRACE_STYLE == NEXT_LINE || myIndentOptions.BRACE_STYLE == END_OF_LINE) {
      return NONE;
    }
    return null;
  }

  @Contract("null -> null")
  protected static Indent getDefaultIndentFromType(@Nullable Type type) {
    return type == null
      ? null
      : Indent.getIndent(type, 0, false, false);
  }

  public class IndentCalculatorFactory {

    @Nullable
    public IndentCalculator createIndentCalculator(@Nullable Type indentType, @Nullable IndentCalculator.BaseLineOffsetCalculator baseLineOffsetCalculator) {
      return createIndentCalculator(getDefaultIndentFromType(indentType), baseLineOffsetCalculator);
    }

    @Nullable
    public IndentCalculator createIndentCalculator(@Nullable Indent indent, @Nullable IndentCalculator.BaseLineOffsetCalculator baseLineOffsetCalculator) {
      return indent != null ?
        new IndentCalculator(baseLineOffsetCalculator != null
          ? baseLineOffsetCalculator
          : IndentCalculator.LINE_BEFORE,
          myChars,
          myIndentOptions,
          indent)
        : null;
    }

    public @Nullable IndentCalculator createIndentCalculatorWithCustomBaseIndent(@Nullable Indent indent, @NotNull String baseIndent) {
      if (indent == null) {
        return null;
      }

      return new IndentCalculator(IndentCalculator.LINE_BEFORE, myChars, myIndentOptions, indent) {
        @Override
        protected @NotNull String getBaseIndent(@NotNull SemanticEditorPosition currPosition) {
          return baseIndent;
        }
      };
    }
  }

  @Override
  @Contract("null -> false")
  public final boolean isSuitableFor(@Nullable Language language) {
    return language != null && isSuitableForLanguage(language);
  }

  public abstract boolean isSuitableForLanguage(@NotNull Language language);

  protected Type getIndentTypeInBrackets() {
    return CONTINUATION;
  }

  protected Indent getIndentInBrackets() {
    return getDefaultIndentFromType(getIndentTypeInBrackets());
  }

  private static final int END_OF_LINE = 1;
  private static final int NEXT_LINE = 2;
  private static final int NEXT_LINE_SHIFTED = 3;
}
