
/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.impl.source.codeStyle.lineIndent;

import com.intellij.formatting.Indent;
import com.intellij.formatting.IndentImpl;
import com.intellij.formatting.IndentInfo;
import com.intellij.psi.codeStyle.IndentOptions;
import com.intellij.psi.impl.source.codeStyle.SemanticEditorPosition;
import com.intellij.util.text.CharArrayUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static com.intellij.formatting.Indent.Type.*;

public class IndentCalculator {

  private @NotNull final BaseLineOffsetCalculator myBaseLineOffsetCalculator;
  private @NotNull final IndentOptions myIndentOptions;
  private @NotNull final Indent myIndent;
  private @NotNull final CharSequence myChars;

  public IndentCalculator(@NotNull BaseLineOffsetCalculator baseLineOffsetCalculator,
                          @NotNull CharSequence chars,
                          @NotNull IndentOptions indentOptions,
                          @NotNull Indent indent) {
    myIndent = indent;
    myChars = chars;
    myBaseLineOffsetCalculator = baseLineOffsetCalculator;
    myIndentOptions = indentOptions;
  }

  public final static BaseLineOffsetCalculator LINE_BEFORE =
    currPosition -> CharArrayUtil.shiftBackward(currPosition.getChars(), currPosition.getStartOffset(), " \t\n\r");

  public final static BaseLineOffsetCalculator LINE_AFTER =
    currPosition -> CharArrayUtil.shiftForward(currPosition.getChars(), currPosition.getStartOffset(), " \t\n\r");

  @Nullable
  String getIndentString(@NotNull SemanticEditorPosition currPosition) {
    String baseIndent = getBaseIndent(currPosition);
    final int indentLength =
      baseIndent.replaceAll("\t", " ".repeat(myIndentOptions.TAB_SIZE)).length()
        + indentToSize(myIndent, myIndentOptions);
    return new IndentInfo(0, indentLength, 0, false).generateNewWhiteSpace(myIndentOptions);
  }

  @NotNull
  protected String getBaseIndent(@NotNull SemanticEditorPosition currPosition) {
    CharSequence docChars = myChars;
    int offset = currPosition.getStartOffset();
    if (offset > 0) {
      int indentLineOffset = myBaseLineOffsetCalculator.getOffsetInBaseIndentLine(currPosition);
      if (indentLineOffset > 0) {
        int indentStart = CharArrayUtil.shiftBackwardUntil(docChars, indentLineOffset, "\n") + 1;
        if (indentStart >= 0) {
          int indentEnd = CharArrayUtil.shiftForward(docChars, indentStart, " \t");
          if (indentEnd > indentStart) {
            return docChars.subSequence(indentStart, indentEnd).toString();
          }
        }
      }
    }
    return "";
  }

  private static int indentToSize(@NotNull Indent indent, @NotNull IndentOptions options) {
    if (indent.getType() == NORMAL) {
      return options.INDENT_SIZE;
    } else if (indent.getType() == CONTINUATION) {
      return options.CONTINUATION_INDENT_SIZE;
    } else if (indent.getType() == SPACES && indent instanceof IndentImpl) {
      return ((IndentImpl) indent).getSpaces();
    }
    return 0;
  }

  public interface BaseLineOffsetCalculator {
    int getOffsetInBaseIndentLine(@NotNull SemanticEditorPosition position);
  }
}
