// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.psi.builder;

import com.intellij.psi.ITokenSequence;
import com.intellij.psi.tree.ILazyParseableElementType;
import com.intellij.psi.tree.TokenSet;
import com.intellij.util.containers.IntStack;
import com.intellij.util.containers.Stack;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.atomic.AtomicReference;

public class MarkerPsiBuilder<T> extends FleetPsiBuilder<ASTMarkers<T>> {
  private ASTMarkersImpl<T> myTree = null;

  public MarkerPsiBuilder(@NotNull CharSequence text,
                          @NotNull ITokenSequence tokens,
                          @NotNull TokenSet whitespaceTokens,
                          @NotNull TokenSet commentTokens,
                          int startLexeme,
                          int lexemeCount) {
    super(text, tokens, whitespaceTokens, commentTokens, startLexeme, lexemeCount);
  }

  @Override
  @NotNull
  public ASTMarkers<T> getRoot() {
    return prepareLightTree();
  }

  @NotNull
  private ASTMarkers<T> prepareLightTree() {
    if (myProduction.isEmpty()) {
      LOG.error("Parser produced no markers. Text:\n" + myText);
    }
    // build tree only once to avoid threading issues in read-only PSI
    if (myTree != null) return myTree;

    myOptionalData.compact();
    myTokenTypeChecked = true;
    balanceWhiteSpaces();

    myTree = new ASTMarkersImpl<>();

    int lastErrorIndex = -1;
    int maxDepth = 0;
    int curDepth = 0;
    boolean insideChameleon = false;
    int markersInsideChameleon = 0;
    StartMarker curNode = null;
    final Stack<StartMarker> nodes = new Stack<>();
    final IntStack productionIndices = new IntStack();
    final IntStack markerIndices = new IntStack();


    for (int i = 0; i < myProduction.size(); i++) {
      ProductionMarker item = myProduction.getStartMarkerAt(i);

      if (item instanceof final StartMarker marker) {
        if (insideChameleon) {
          ++markersInsideChameleon;
          continue;
        }
        int index = myTree.pushBack();
        myTree.setMarker(index, marker.markerId, ASTMarkers.MarkerKind.Start, myOptionalData.isCollapsed(marker.markerId), null, marker.myType);
        if (marker.myType instanceof ILazyParseableElementType && myOptionalData.isCollapsed(marker.markerId)) {
          myTree.setChameleon(marker.myLexemeIndex - myStartLexeme, new AtomicReference<>(null));
          insideChameleon = true;
        }
        nodes.push(marker);
        markerIndices.push(index);
        productionIndices.push(i);

        curNode = marker;
        curDepth++;
        if (curDepth > maxDepth) maxDepth = curDepth;
      } else if (item instanceof ErrorItem errorItem) {
        int curToken = item.myLexemeIndex;
        if (curToken == lastErrorIndex) continue;
        int prevLexemeIndex = (i > 0) ? myProduction.getLexemeIndexAt(i - 1) : 0;
        int startLexeme = errorItem.getStartIndex();
        int endLexeme = errorItem.getEndIndex();
        lastErrorIndex = curToken;
        int index = myTree.pushBack();
        myTree.setMarker(index, errorItem.markerId, ASTMarkers.MarkerKind.Error, false, errorItem.myMessage, errorItem.getTokenType());
        myTree.setLexemeInfo(index, endLexeme - startLexeme, startLexeme - prevLexemeIndex);
      } else {
        if (insideChameleon) {
          if (markersInsideChameleon != 0) {
            --markersInsideChameleon;
            continue;
          }
          insideChameleon = false;
        }
        assertMarkersBalanced(myProduction.getDoneMarkerAt(i) == curNode, item);
        curNode = nodes.pop();
        int productionStart = productionIndices.pop();
        int markersStart = markerIndices.pop();
        int prevLexemeIndex = (markersStart > 0) ? myProduction.getLexemeIndexAt(productionStart - 1) : 0;
        int index = myTree.pushBack();
        myTree.setMarker(index, curNode.markerId, ASTMarkers.MarkerKind.End, myOptionalData.isCollapsed(curNode.markerId), null, curNode.myType);
        int startLexeme = myProduction.getLexemeIndexAt(productionStart);
        int endLexeme = myProduction.getLexemeIndexAt(i);
        myTree.setLexemeInfo(index, endLexeme - startLexeme, startLexeme - prevLexemeIndex);
        myTree.setLexemeInfo(markersStart, endLexeme - startLexeme, startLexeme - prevLexemeIndex);
        myTree.setMarkersCount(index, index - markersStart);
        myTree.setMarkersCount(markersStart, index - markersStart);
        curDepth--;
      }
    }

    for (int i = 0; i < myLexemeCount - myStartLexeme; ++i) {
      if (myTokens.lexType(myStartLexeme + i) instanceof ILazyParseableElementType) {
        myTree.setChameleon(i, new AtomicReference<>(null));
      }
    }

    clearCachedTokenType();
    return myTree;
  }
}
