// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.lang;

import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.UserDataHolder;
import com.intellij.psi.tree.IElementType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public interface PsiBuilder extends SyntaxTreeBuilder, UserDataHolder {
  @Override
  @NotNull Marker mark();

  interface Marker extends SyntaxTreeBuilder.Marker {
    @Override
    @NotNull
    Marker precede();

    @Override
    default void doneBefore(@NotNull IElementType type, SyntaxTreeBuilder.@NotNull Marker before) {
      doneBefore(type, (Marker) before);
    }

    @Override
    default void doneBefore(@NotNull IElementType type,
                            SyntaxTreeBuilder.@NotNull Marker before,
                            @NotNull String errorMessage) {
      doneBefore(type, (Marker) before, errorMessage);
    }

    @Override
    default void errorBefore(@NotNull String message, SyntaxTreeBuilder.@NotNull Marker before) {
      errorBefore(message, (Marker) before);
    }

    void doneBefore(@NotNull IElementType type, @NotNull Marker before);

    void doneBefore(@NotNull IElementType type,
                    @NotNull Marker before,
                    @NotNull String errorMessage);

    void errorBefore(@NotNull String message, @NotNull Marker before);
  }

  /**
   * @return a user data value associated with this object. Doesn't require read action.
   */
  @Nullable <T> T getUserData(@NotNull Key<T> key);

  /**
   * Add a new user data value to this object. Doesn't require write action.
   */
  <T> void putUserData(@NotNull Key<T> key, @Nullable T value);

  default ASTNode getTreeBuilt() {
    throw new IllegalStateException();
  }

  default void rawAdvanceLexer(int steps) {
    if (steps < 0) {
      throw new IllegalArgumentException("Steps must be a positive integer - lexer can only be advanced. " +
              "Use Marker.rollbackTo if you want to rollback PSI building.");
    }
    if (steps == 0) return;
    int offset = rawTokenTypeStart(steps);
    while (!eof() && getCurrentOffset() < offset) {
      advanceLexer();
    }
  }
}
