// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.util;

import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

/**
 * A stripped-down version of {@link ArrayUtil}.
 * Intended to use by external (out-of-IDE-process) runners and helpers so it should not contain any library dependencies.
 */
public final class ArrayUtilRt {
  public static final short[] EMPTY_SHORT_ARRAY = new short[0];
  public static final char[] EMPTY_CHAR_ARRAY = new char[0];
  public static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
  public static final int[] EMPTY_INT_ARRAY = new int[0];
  public static final boolean[] EMPTY_BOOLEAN_ARRAY = new boolean[0];
  public static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];
  public static final String[] EMPTY_STRING_ARRAY = new String[0];
  public static final long[] EMPTY_LONG_ARRAY = new long[0];

  @NotNull
  @Contract(pure = true)
  public static String[] toStringArray(@Nullable Collection<String> collection) {
    return collection == null || collection.isEmpty()
      ? EMPTY_STRING_ARRAY : collection.toArray(EMPTY_STRING_ARRAY);
  }

  /**
   * @param src source array.
   * @param obj object to be found.
   * @return index of {@code obj} in the {@code src} array.
   * Returns {@code -1} if passed object isn't found. This method uses
   * {@code equals} of arrays elements to compare {@code obj} with
   * these elements.
   */
  @Contract(pure = true)
  public static <T> int find(@NotNull T[] src, @Nullable T obj) {
    return indexOf(src, obj, 0, src.length);
  }

  @Contract(pure = true)
  public static <T> int indexOf(@NotNull T[] src, @Nullable T obj, int start, int end) {
    if (obj == null) {
      for (int i = start; i < end; i++) {
        if (src[i] == null) return i;
      }
    } else {
      for (int i = start; i < end; i++) {
        if (obj.equals(src[i])) return i;
      }
    }
    return -1;
  }
}
