package com.intellij.psi.builder;

import com.intellij.openapi.util.TextRange;
import com.intellij.psi.tree.IElementType;
import kala.collection.mutable.MutableList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public record MarkerNode(
  @Override @NotNull IElementType elementType,
  @Override @NotNull TextRange range,
  @Nullable MarkerNode parent,
  @NotNull MutableList<MarkerNode> children
) {
  public MarkerNode(@NotNull IElementType elementType, int startOffset, int endOffset, @Nullable MarkerNode parent, @NotNull MutableList<MarkerNode> children) {
    this(elementType, new TextRange(startOffset, endOffset), parent, children);
    if (parent != null) parent.children.append(this);
  }

  public MarkerNode(@NotNull IElementType elementType, int startOffset, int endOffset, @Nullable MarkerNode parent) {
    this(elementType, startOffset, endOffset, parent, MutableList.create());
  }

  boolean compareAsToken(MarkerNode node) {
    return elementType != node.elementType || !range.equals(node.range);
  }

  @Override public @NotNull String toString() {
    return elementType + String.valueOf(range);
  }

  public @NotNull String toDebugString(String indentSymbol) {
    var builder = new StringBuilder();
    toDebugString(0, builder, indentSymbol);
    return builder.toString();
  }

  private void toDebugString(int indent, @NotNull StringBuilder builder, String indentSymbol) {
    ASTMarkers.repeat(indent, i -> builder.append(indentSymbol));
    builder.append(elementType).append(range).append(System.lineSeparator());
    children.forEach(it -> it.toDebugString(indent + 1, builder, indentSymbol));
  }
}
