// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.util.containers;

//import com.intellij.openapi.Disposable;

import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

public class ContainerUtil
  //extends ContainerUtilRt
{
  private static final int INSERTION_SORT_THRESHOLD = 10;

  @SafeVarargs
  @NotNull
  @Contract(pure = true)
  public static <T> T[] ar(@NotNull T... elements) {
    return elements;
  }

  @NotNull
  @Contract(pure = true)
  public static <K, V> HashMap<K, V> newHashMap() {
    return new HashMap<>();
  }

  @NotNull
  @Contract(pure = true)
  public static <K, V> HashMap<K, V> newHashMap(@NotNull Map<? extends K, ? extends V> map) {
    return new HashMap<>(map);
  }

  //@NotNull
  //@SafeVarargs
  //@Contract(pure=true)
  //public static <K, V> Map<K, V> newHashMap(@NotNull Pair<? extends K, ? extends V> first, @NotNull Pair<? extends K, ? extends V>... entries) {
  //  return ContainerUtilRt.newHashMap(first, entries);
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <K, V> Map<K, V> newHashMap(@NotNull List<? extends K> keys, @NotNull List<? extends V> values) {
  //  return ContainerUtilRt.newHashMap(keys, values);
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <K extends Comparable<? super K>, V> TreeMap<K, V> newTreeMap() {
  //  return ContainerUtilRt.newTreeMap();
  //}

  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T> TObjectHashingStrategy<T> identityStrategy() {
  //  //noinspection unchecked
  //  return TObjectHashingStrategy.IDENTITY;
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <K, V> IdentityHashMap<K, V> newIdentityHashMap() {
  //  return new IdentityHashMap<>();
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T> LinkedList<T> newLinkedList() {
  //  return ContainerUtilRt.newLinkedList();
  //}
  //
  //@SafeVarargs
  //@NotNull
  //@Contract(pure=true)
  //public static <T> LinkedList<T> newLinkedList(@NotNull T... elements) {
  //  return ContainerUtilRt.newLinkedList(elements);
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T> LinkedList<T> newLinkedList(@NotNull Iterable<? extends T> elements) {
  //  return ContainerUtilRt.newLinkedList(elements);
  //}

  @NotNull
  @Contract(pure = true)
  public static <T> ArrayList<T> newArrayList() {
    return new ArrayList<>();
  }

  @NotNull
  @Contract(pure = true)
  public static <E> ArrayList<E> newArrayList(@NotNull Iterable<? extends E> iterable) {
    ArrayList<E> result = new ArrayList<>();
    for (E elem : iterable) {
      result.add(elem);
    }
    return result;
  }

  //@NotNull
  //@Contract(pure=true)
  //public static <T> ArrayList<T> newArrayListWithCapacity(int size) {
  //  return new ArrayList<>(size);
  //}

  @NotNull
  @Contract(pure = true)
  public static <T> List<T> newArrayList(@NotNull final T[] elements, final int start, final int end) {
    if (start < 0 || start > end || end > elements.length) {
      throw new IllegalArgumentException("start:" + start + " end:" + end + " length:" + elements.length);
    }

    return new AbstractList<>() {
      private final int size = end - start;

      @Override
      public T get(final int index) {
        if (index < 0 || index >= size) throw new IndexOutOfBoundsException("index:" + index + " size:" + size);
        return elements[start + index];
      }

      @Override
      public int size() {
        return size;
      }
    };
  }

  //@NotNull
  //@Contract(pure = true)
  //public static <T> List<T> newUnmodifiableList(List<? extends T> originalList) {
  //  int size = originalList.size();
  //  if (size == 0) {
  //    return emptyList();
  //  }
  //  if (size == 1) {
  //    return Collections.singletonList(originalList.get(0));
  //  }
  //  return Collections.unmodifiableList(new ArrayList<T>(originalList));
  //}
  //
  //@NotNull
  //@Contract(pure = true)
  //public static <T> Collection<T> unmodifiableOrEmptyCollection(@NotNull Collection<? extends T> original) {
  //  int size = original.size();
  //  if (size == 0) {
  //    return emptyList();
  //  }
  //  if (size == 1) {
  //    return Collections.singletonList(original.iterator().next());
  //  }
  //  return Collections.unmodifiableCollection(original);
  //}
  //
  //@NotNull
  //@Contract(pure = true)
  //public static <T> List<T> unmodifiableOrEmptyList(@NotNull List<? extends T> original) {
  //  int size = original.size();
  //  if (size == 0) {
  //    return emptyList();
  //  }
  //  if (size == 1) {
  //    return Collections.singletonList(original.iterator().next());
  //  }
  //  return Collections.unmodifiableList(original);
  //}
  //
  //@NotNull
  //@Contract(pure = true)
  //public static <T> Set<T> unmodifiableOrEmptySet(@NotNull Set<? extends T> original) {
  //  int size = original.size();
  //  if (size == 0) {
  //    return Collections.emptySet();
  //  }
  //  if (size == 1) {
  //    return Collections.singleton(original.iterator().next());
  //  }
  //  return Collections.unmodifiableSet(original);
  //}
  //
  //@NotNull
  //@Contract(pure = true)
  //public static <K,V> Map<K,V> unmodifiableOrEmptyMap(@NotNull Map<? extends K, ? extends V> original) {
  //  int size = original.size();
  //  if (size == 0) {
  //    return Collections.emptyMap();
  //  }
  //  if (size == 1) {
  //    Map.Entry<? extends K, ? extends V> entry = original.entrySet().iterator().next();
  //    return Collections.singletonMap(entry.getKey(), entry.getValue());
  //  }
  //  else {
  //    return Collections.unmodifiableMap(original);
  //  }
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T> List<T> newSmartList() {
  //  return new SmartList<>();
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T> List<T> newSmartList(T element) {
  //  return new SmartList<>(element);
  //}
  //
  //@SafeVarargs
  //@NotNull
  //@Contract(pure=true)
  //public static <T> List<T> newSmartList(@NotNull T... elements) {
  //  return new SmartList<>(elements);
  //}
  //
  @NotNull
  @Contract(pure = true)
  public static <T> HashSet<T> newHashSet() {
    return new HashSet<>();
  }

  @NotNull
  @Contract(pure = true)
  public static <T> Set<T> newConcurrentSet() {
    return Collections.newSetFromMap(newConcurrentMap());
  }

  @NotNull
  @Contract(pure = true)
  public static <K, V> ConcurrentMap<K, V> newConcurrentMap() {
    return new ConcurrentHashMap<>();
  }

  @NotNull
  @Contract(pure = true)
  public static <T, V> V[] map2Array(@NotNull T[] array, @NotNull Class<V> aClass, @NotNull Function<? super T, ? extends V> mapper) {
    V[] result = ArrayUtil.newArray(aClass, array.length);
    for (int i = 0; i < array.length; i++) {
      result[i] = mapper.apply(array[i]);
    }
    return result;
  }
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T, V> V[] map2Array(@NotNull Collection<? extends T> collection, @NotNull Class<V> aClass, @NotNull Function<? super T, ? extends V> mapper) {
  //  V[] result = ArrayUtil.newArray(aClass, collection.size());
  //  int i = 0;
  //  for (T t : collection) {
  //    result[i++] = mapper.fun(t);
  //  }
  //  return result;
  //}
  //
  //@NotNull
  //@Contract(pure=true)
  //public static <T, V> V[] map2Array(@NotNull Collection<? extends T> collection, @NotNull V[] to, @NotNull Function<? super T, V> mapper) {
  //  return map2List(collection, mapper).toArray(to);
  //}

  @Contract(pure = true)
  public static <T> T getLastItem(@Nullable List<? extends T> list, @Nullable T def) {
    return isEmpty(list) ? def : list.get(list.size() - 1);
  }

  @Contract(pure = true)
  public static <T> T getLastItem(@Nullable List<? extends T> list) {
    return getLastItem(list, null);
  }


  @Contract(value = "null -> true", pure = true)
  public static <T> boolean isEmpty(@Nullable Collection<? extends T> collection) {
    return collection == null || collection.isEmpty();
  }

  @NotNull
  @Contract(pure = true)
  public static <T> List<T> sorted(@NotNull Collection<? extends T> list, @NotNull Comparator<? super T> comparator) {
    return sorted((Iterable<? extends T>) list, comparator);
  }

  @NotNull
  @Contract(pure = true)
  public static <T> List<T> sorted(@NotNull Iterable<? extends T> list, @NotNull Comparator<? super T> comparator) {
    List<T> sorted = newArrayList(list);
    sort(sorted, comparator);
    return sorted;
  }

  @NotNull
  @Contract(pure = true)
  public static <T extends Comparable<? super T>> List<T> sorted(@NotNull Collection<? extends T> list) {
    return sorted(list, Comparator.naturalOrder());
  }

  public static <T> void sort(@NotNull List<T> list, @NotNull Comparator<? super T> comparator) {
    int size = list.size();

    if (size < 2) return;
    if (size == 2) {
      T t0 = list.get(0);
      T t1 = list.get(1);

      if (comparator.compare(t0, t1) > 0) {
        list.set(0, t1);
        list.set(1, t0);
      }
    } else if (size < INSERTION_SORT_THRESHOLD) {
      for (int i = 0; i < size; i++) {
        for (int j = 0; j < i; j++) {
          T ti = list.get(i);
          T tj = list.get(j);

          if (comparator.compare(ti, tj) < 0) {
            list.set(i, tj);
            list.set(j, ti);
          }
        }
      }
    } else {
      list.sort(comparator);
    }
  }

  public static <T> void sort(@NotNull T[] a, @NotNull Comparator<? super T> comparator) {
    int size = a.length;

    if (size < 2) return;
    if (size == 2) {
      T t0 = a[0];
      T t1 = a[1];

      if (comparator.compare(t0, t1) > 0) {
        a[0] = t1;
        a[1] = t0;
      }
    } else if (size < INSERTION_SORT_THRESHOLD) {
      for (int i = 0; i < size; i++) {
        for (int j = 0; j < i; j++) {
          T ti = a[i];
          T tj = a[j];

          if (comparator.compare(ti, tj) < 0) {
            a[i] = tj;
            a[j] = ti;
          }
        }
      }
    } else {
      Arrays.sort(a, comparator);
    }
  }

  @NotNull
  @Contract(value = " -> new", pure = true)
  public static <T> List<T> createLockFreeCopyOnWriteList() {
    return new ArrayList<>();
  }

  public static <T> void addIfNotNull(@NotNull Collection<? super T> result, @Nullable T element) {
    if (element != null) {
      result.add(element);
    }
  }

  public static class ImmutableMapBuilder<K, V> {
    private final Map<K, V> myMap = new HashMap<>();

    public ImmutableMapBuilder<K, V> put(K key, V value) {
      myMap.put(key, value);
      return this;
    }

    @Contract(pure=true)
    public Map<K, V> build() {
      return Collections.unmodifiableMap(myMap);
    }
  }

  @SafeVarargs
  @NotNull
  @Contract(pure=true)
  public static <E> Set<E> immutableSet(@NotNull E... elements) {
    return switch (elements.length) {
      case 0 -> Collections.emptySet();
      case 1 -> Collections.singleton(elements[0]);
      default -> Set.of(elements);
    };
  }
}

