package com.intellij.openapi.components;

import com.intellij.lang.Language;
import com.intellij.util.containers.MultiMap;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class ExtensionsMap {
  private static final MultiMap<Class<?>, Object> ourExtensions = MultiMap.createConcurrentSet();
  private static final Map<Class<?>, Map<Language, Object>> ourLanguageExtensions = new ConcurrentHashMap<>();

  public static <T> void registerExtension(Class<T> extensionPoint, T extension) {
    ourExtensions.putValue(extensionPoint, extension);
  }

  public static <T> void unregisterExtension(Class<T> extensionPoint, T extension) {
    ourExtensions.remove(extensionPoint, extension);
  }

  public static <T> List<T> getExtensions(Class<T> extensionPoint) {
    return new ArrayList<>((Collection<T>) ourExtensions.get(extensionPoint));
  }

  public static <T> void registerLanguageExtension(Class<T> extensionPoint, T extension, Language language) {
    Map<Language, Object> extensions = ourLanguageExtensions.computeIfAbsent(extensionPoint, (k) -> new ConcurrentHashMap<>());
    extensions.put(language, extension);
  }

  public static <T> void unregisterLanguageExtension(Class<T> extensionPoint, Language language) {
    Map<Language, Object> extensions = ourLanguageExtensions.get(extensionPoint);
    if (extensions != null) {
      extensions.remove(language);
    }
  }

  public static <T> T getLanguageExtension(Class<T> extensionPoint, Language language) {
    return (T) ourLanguageExtensions.getOrDefault(extensionPoint, Collections.emptyMap()).get(language);
  }
}
